/* Copyright (C) 2008 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <lua.h>
#include <lauxlib.h>
#include <GL/gl.h>
#include <fontconfig/fontconfig.h>

#include "face.h"
#include "string.h"

static FT_Library library;

@implementation Face

-(Face *) initFromFile: (const char *)name
{
    FT_Error error;
    FcPattern *pattern, *match;
    FcResult result;
    FcChar8 *file;
    int size, index;
    
    [super init];
    
    if (!library) {
	error = FT_Init_FreeType(&library);
	
	if (error) {
	    printf ("Could not initialize the FreeType library.\n");
	}
    }
    
    if (!FcInit ()) {
	printf ("Could not initialize the fontconfig library.\n");
    }

    /* Match the font name to a file. */
    
    pattern = FcNameParse ((FcChar8 *) name);
    
    FcConfigSubstitute (0, pattern, FcMatchPattern);
    FcDefaultSubstitute (pattern);
    
    match = FcFontMatch (0, pattern, &result);

    if (match) {
	FcPatternGetString (match, FC_FILE, 0, &file);
	FcPatternGetInteger (match, FC_INDEX, 0, &index);
	FcPatternGetInteger (match, FC_SIZE, 0, &size);
    }

    FcPatternDestroy (pattern);
    FcPatternDestroy (match);

    printf ("  %s\n", file);
    
    if (FT_New_Face(library, (char *)file, index, &self->face )) {
	printf ("Font file could not be read or not supported.\n");
	return NULL;
    } else if (FT_Set_Char_Size(face, 0, size * 64, 72, 72)) {
	printf ("Could not select the requested font size.\n");
	return self;
    } else {
	return self;
    }
}

-(id) free
{
    FT_Done_Face(self->face);

    return [super free];
}

-(id)layoutString: (const char *)string
{
    id object;
    FT_GlyphSlot slot;
    FT_Error error;
    FT_UInt j, k;
    int x, y, i, size[2];

    FT_Vector *position;
    FT_Glyph *glyphs;
    FT_BBox glyph, layout;

    unsigned char *buffer;

    position = malloc (strlen(string) * sizeof (FT_Vector));
    glyphs = malloc (strlen(string) * sizeof (FT_Glyph));
    slot = face->glyph;

    for(i = 0, j = 0, k = 0, x = 0, y = 0;
	i < strlen(string);
	i += 1, j = k, x += slot->advance.x / 64) {
	k = FT_Get_Char_Index(face, string[i]);

	if (FT_HAS_KERNING(face) && j && k ) {
	    FT_Vector delta;

	    FT_Get_Kerning(face, j, k, FT_KERNING_DEFAULT, &delta);
	    x += delta.x / 64;
	}

	position[i].x = x;
	position[i].y = y;

	error = FT_Load_Glyph(face, k, FT_LOAD_DEFAULT);
	error = FT_Get_Glyph(face->glyph, &glyphs[i]);
    }

    layout.xMin = 32000;
    layout.yMin = 32000;
    layout.xMax = -32000;
    layout.yMax = -32000;

    for(i = 0 ; i < strlen(string) ; i += 1) {
	FT_Glyph_Get_CBox(glyphs[i], FT_GLYPH_BBOX_PIXELS, &glyph);

	glyph.xMin += position[i].x;
	glyph.xMax += position[i].x;
	glyph.yMin += position[i].y;
	glyph.yMax += position[i].y;

	layout.xMin = glyph.xMin < layout.xMin ? glyph.xMin : layout.xMin;
	layout.yMin = glyph.yMin < layout.yMin ? glyph.yMin : layout.yMin;
	layout.xMax = glyph.xMax > layout.xMax ? glyph.xMax : layout.xMax;
	layout.yMax = glyph.yMax > layout.yMax ? glyph.yMax : layout.yMax;
    }

    size[0] = layout.xMax - layout.xMin + 2;
    size[1] = layout.yMax - layout.yMin + 2;

    buffer = calloc (size[0] * size[1], sizeof(unsigned char));

    for(i = 0 ; i < strlen(string) ; i += 1) {
	error = FT_Glyph_To_Bitmap(&glyphs[i],
				   FT_RENDER_MODE_NORMAL,
				   0, 0);
	
	if (!error) {
	    FT_BitmapGlyph bitmap;
	    bitmap = (FT_BitmapGlyph)glyphs[i];

	    for (k = 0 ; k < bitmap->bitmap.rows ; k += 1) {
		for (j = 0 ; j < bitmap->bitmap.width ; j += 1) {
		    buffer[(position[i].y - layout.yMin +
			    bitmap->top - k) * size[0] +
			   position[i].x - layout.xMin +
			   bitmap->left + j + 1] =
			bitmap->bitmap.buffer[k * bitmap->bitmap.pitch + j];
		}
	    }
	}
	
	FT_Done_Glyph(glyphs[i]);
    }

    object = [[String alloc] initFromPixels: buffer ofSize: size];
    
    free(buffer);
    free(position);
    free(glyphs);

    return object;
}

@end
