/* Copyright (C) 2008 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <lua.h>
#include <lauxlib.h>
#include <GL/gl.h>
#include <GL/glu.h>

#include "texture.h"
#include "light.h"

#include "static.h"

static void drawSubtree (id root)
{
    id child;
    
    if ([root isKindOf: [Geometry class]]) {
	[root draw];
    }
    
    for(child = [root children] ; child ; child = [child sister]) {
	drawSubtree (child);
    }
}

@implementation Light

-(GLfloat *) intensityConstant
{
    return self->intensityConstant;
}

-(GLuint) intensityMap
{
    return [self->intensityMap index];
}

-(GLfloat *) attenuation
{
    return self->attenuation;
}

-(GLfloat *) shadowMatrix
{
    return self->shadowMatrix;
}

-(GLuint) shadowMap
{
    return self->shadowMap;
}

-(Light *) init
{
    [super init];
    
    self->size[0] = 0;
    self->size[1] = 0;

    self->volume[0] = 0;
    self->volume[1] = 0;
    self->volume[2] = 0;
    self->volume[3] = 0;
    self->volume[4] = 0;
    self->volume[5] = 0;

    self->intensityConstant[0] = 1;
    self->intensityConstant[1] = 1;
    self->intensityConstant[2] = 1;

    self->attenuation[0] = 1;
    self->attenuation[1] = 0;
    self->attenuation[2] = 0;

    self->buffer = 0;
    self->shadowMap = 0;
    self->intensityMap = nil;
    
    return self;
}

-(void) resetBuffers
{
    glDeleteTextures(1, &self->shadowMap);
    glDeleteFramebuffersEXT(1, &self->buffer);

    if (self->size[0] * self->size[1] > 0) {
	/* Create the shadow map texture. */

	glGenTextures(1, &self->shadowMap);
	glBindTexture(GL_TEXTURE_2D, self->shadowMap);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_DEPTH_COMPONENT,
		     self->size[0], self->size[1], 0,
		     GL_DEPTH_COMPONENT, GL_UNSIGNED_BYTE, NULL);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
    
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_COMPARE_FUNC, GL_LEQUAL);
	glTexParameteri(GL_TEXTURE_2D, GL_DEPTH_TEXTURE_MODE, GL_INTENSITY);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_COMPARE_MODE,
			GL_COMPARE_R_TO_TEXTURE);

	/* Create the frame buffer object. */

	glGenFramebuffersEXT (1, &self->buffer);
	glBindFramebufferEXT (GL_FRAMEBUFFER_EXT, self->buffer);
	glFramebufferTexture2DEXT (GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT,
				   GL_TEXTURE_2D, self->shadowMap, 0);
	glDrawBuffer (GL_NONE);
	glReadBuffer (GL_NONE);

	if(glCheckFramebufferStatusEXT (GL_FRAMEBUFFER_EXT) !=
	   GL_FRAMEBUFFER_COMPLETE_EXT) {
	    printf ("Could not create the framebuffer "
		    "object for the shadow map (%x).\n",
		    glCheckFramebufferStatusEXT(GL_FRAMEBUFFER_EXT));
	}
    
	glBindFramebufferEXT (GL_FRAMEBUFFER_EXT, 0);
    }
}

-(void) prepare: (lua_State *)L
{
    GLfloat *r, *R;

    if (self->size[0] * self->size[1] > 0) {
	/* Set up the viewport and transforms. */
    
	r = [self translation];
	R = [self rotation];
    
	glBindFramebufferEXT (GL_FRAMEBUFFER_EXT, self->buffer);
	glPushAttrib (GL_VIEWPORT_BIT);
	glViewport (0, 0, self->size[0], self->size[1]);

	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadMatrixf (self->projection);

	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadIdentity();

	gluLookAt(r[0], r[1], r[2], 
		  r[0] + R[2], r[1] + R[5], r[2] + R[8],
		  R[1], R[4], R[7]);
    
	glGetFloatv(GL_MODELVIEW_MATRIX, self->modelview);

	/* Configure and render the scene. */
    
	glCullFace (GL_FRONT);
	glPolygonOffset (self->offset[0], self->offset[1]);
	glEnable (GL_POLYGON_OFFSET_FILL);
	glClear(GL_DEPTH_BUFFER_BIT);
    
	glUseProgramObjectARB(0);

	glEnable (GL_DEPTH_TEST);
	glEnable (GL_CULL_FACE);

	drawSubtree([self parent]);

	glDisable (GL_DEPTH_TEST);
	glDisable (GL_CULL_FACE);
    
	/* Restore the state. */
  
	glCullFace(GL_BACK);
	glDisable(GL_POLYGON_OFFSET_FILL);

	glMatrixMode(GL_PROJECTION);
	glPopMatrix();

	glMatrixMode(GL_MODELVIEW);
	glPopMatrix();

	glPopAttrib();
	glBindFramebufferEXT (GL_FRAMEBUFFER_EXT, 0);

	/* Copy the depth buffer into the texture. */
    
	glMatrixMode (GL_TEXTURE);
	glPushMatrix();
	glLoadIdentity();
	glTranslatef (0.5, 0.5, 0.5);
	glScalef (0.5, 0.5, 0.5);
	glMultMatrixf(self->projection);
	glMultMatrixf(self->modelview);
	glGetFloatv(GL_TRANSPOSE_TEXTURE_MATRIX, self->shadowMatrix);
	glPopMatrix();
    }
    
    [super prepare: L];
}

-(void) get: (lua_State *)L
{
    const char *k;
    int i;

    k = lua_tostring(L, 2);

    if (!strcmp(k, "intensity")) {
	lua_getmetatable (L, 1);
	lua_replace (L, 1);
	lua_gettable (L, 1);
    } else if (!strcmp(k, "attenuation")) {
        lua_newtable(L);
        
        for(i = 0; i < 3; i += 1) {
            lua_pushnumber(L, self->attenuation[i]);
            lua_rawseti(L, -2, i + 1);
        }
    } else if (!strcmp(k, "size")) {
        lua_newtable(L);
        
        for(i = 0; i < 2; i += 1) {
            lua_pushnumber(L, self->size[i]);
            lua_rawseti(L, -2, i + 1);
        }
    } else if (!strcmp(k, "offset")) {
        lua_newtable(L);
        
        for(i = 0; i < 2; i += 1) {
            lua_pushnumber(L, self->offset[i]);
            lua_rawseti(L, -2, i + 1);
        }
    } else if (!strcmp(k, "volume")) {
        lua_newtable(L);
        
        for(i = 0; i < 6; i += 1) {
            lua_pushnumber(L, self->volume[i]);
            lua_rawseti(L, -2, i + 1);
        }
    } else {
	[super get: L];
    }
}

-(void) set: (lua_State *)L
{    
    const char *k;
    int i;

    k = lua_tostring(L, 2);

    if (!strcmp(k, "intensity")) {
        if(lua_istable(L, 3)) {
            for(i = 0 ; i < 3 ; i += 1) {
                lua_rawgeti(L, 3, i + 1);
                self->intensityConstant[i] = lua_tonumber(L, -1);
                
                lua_pop(L, 1);
            }
	    
	    self->intensityMap = nil;
        } else if(lua_isuserdata (L, 3)) {
            for(i = 0 ; i < 3 ; i += 1) {
                self->intensityConstant[i] = 0;
            }
	    
	    self->intensityMap = *(id *)lua_touserdata (L, 3);
	}

	lua_getmetatable (L, 1);
	lua_replace (L, 1);
	lua_settable (L, 1);
    } else if (!strcmp(k, "attenuation")) {
        if(lua_istable(L, 3)) {
            for(i = 0 ; i < 3 ; i += 1) {
                lua_rawgeti(L, 3, i + 1);
                self->attenuation[i] = lua_tonumber(L, -1);
                
                lua_pop(L, 1);
            }
        }
    } else if (!strcmp(k, "size")) {
        if(lua_istable(L, 3)) {
            for(i = 0 ; i < 2 ; i += 1) {
                lua_rawgeti(L, 3, i + 1);
                self->size[i] = lua_tonumber(L, -1);
                
                lua_pop(L, 1);
            }

	    [self resetBuffers];
        }
    } else if (!strcmp(k, "offset")) {
        if(lua_istable(L, 3)) {
            for(i = 0 ; i < 2 ; i += 1) {
                lua_rawgeti(L, 3, i + 1);
                self->offset[i] = lua_tonumber(L, -1);
                
                lua_pop(L, 1);
            }
        }
    } else if (!strcmp(k, "volume")) {
        if(lua_istable(L, 3)) {
            for(i = 0 ; i < 6 ; i += 1) {
                lua_rawgeti(L, 3, i + 1);
                self->volume[i] = lua_tonumber(L, -1);
                
                lua_pop(L, 1);
            }
    
	    glMatrixMode(GL_PROJECTION);
	    glPushMatrix();
	    glLoadIdentity();
	    glFrustum (self->volume[0], self->volume[1],
		       self->volume[2], self->volume[3],
		       self->volume[4], self->volume[5]);
	    
	    glGetFloatv(GL_PROJECTION_MATRIX, self->projection);
	    glPopMatrix();
        }
    } else {
	[super set: L];
    }
}

/* -(void) traverse */
/* { */
/*     glUseProgramObjectARB(0); */

/*     glColor3f(1, 1, 0); */
    
/*     glBegin(GL_LINES); */
/*     glVertex3d(0, 0, 0); */
/*     glVertex3d(0, 0, 1); */
/*     glEnd(); */

/*     glColor3f(1, 0, 0); */
    
/*     glBegin(GL_POINTS); */
/*     glVertex3d(0, 0, 1); */
/*     glEnd(); */
/* } */

@end
