/* Copyright (C) 2008 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdlib.h>
#include <lua.h>
#include <lauxlib.h>
#include <AL/al.h>
#include <AL/alc.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glx.h>
#include <time.h>
#include "node.h"

static dWorldID world;
static dSpaceID space;
static dJointGroupID group;

static struct timespec once;

static double timescale, stepsize, now, then;
static int collision = LUA_REFNIL;
static int iterations;

static void callhooks (lua_State *L, int ref)
{
    if (ref != LUA_REFNIL) {
	lua_rawgeti(L, LUA_REGISTRYINDEX, ref);
	    
	if (lua_isfunction (L, -1)) {
	    lua_call(L, 0, 0);
	} else if (lua_istable (L, -1)) {
	    lua_pushnil(L);
	
	    while(lua_next(L, -2)) {
		lua_call (L, 0, 0);
	    }
	    
	    lua_pop (L, 1);
	} else {
	    lua_pop (L, 1);
	}
    }
}

static void callback (void *data, dGeomID a, dGeomID b)
{
    dBodyID p, q;
    dContactGeom geoms[16];
    dContact contacts[16];
    double parameters[2];
    int i, j, n, m;

    if (dGeomIsSpace (a) || dGeomIsSpace (b)) {
	dSpaceCollide2 (a, b, data, &callback);

	if (dGeomIsSpace (a)) {
	    dSpaceCollide ((dSpaceID)a, data, &callback);
	}
      
	if (dGeomIsSpace (b)) {
	    dSpaceCollide ((dSpaceID)b, data, &callback);
	}
    } else {
	p = dGeomGetBody (a);
	q = dGeomGetBody (b);

	if (p != NULL || q != NULL) {
	    n = dCollide (a, b, 16, geoms, sizeof(dContactGeom));
	    
	    for (i = 0, j = 0 ; i < n ; i += 1) {
		dVector3 u, v, delta;
		dReal *r, *n, ndotdelta;

		r = geoms[i].pos;
		n = geoms[i].normal;
		
		dSetZero (u, 3);
		dSetZero (v, 3);
		
		if (p) {
		    dBodyGetPointVel (p, r[0], r[1], r[2], u);
		}
		    
		if (q) {
		    dBodyGetPointVel (q, r[0], r[1], r[2], v);
		}

		delta[0] = u[0] - v[0];
		delta[1] = u[1] - v[1];
		delta[2] = u[2] - v[2];

		ndotdelta = dDOT(n, delta);

		/* Consider such a small normal component zero. */
		
		if (ndotdelta > -1e-6 && ndotdelta < 0) {
		    ndotdelta = 0;
		}
		
		/* Only consider this contact if the
		   relative velocity points inward. */

		if (ndotdelta <= 0) {
		    contacts[j].geom = geoms[i];

		    contacts[j].fdir1[0] = delta[0] - ndotdelta * n[0];
		    contacts[j].fdir1[1] = delta[1] - ndotdelta * n[1];
		    contacts[j].fdir1[2] = delta[2] - ndotdelta * n[2];
		    
		    dNormalize3(contacts[j].fdir1);
		
		    j += 1;
		}
	    }

	    if (j > 0) {
		lua_State *L = (lua_State *)data;

		/* Create a collision event. */

		if (collision != LUA_REFNIL) {
		    lua_rawgeti (L, LUA_REGISTRYINDEX, collision);

		    lua_getfield (L, LUA_REGISTRYINDEX, "geoms");
		    
		    lua_pushlightuserdata (L, (void *)a);
		    lua_gettable (L, -2);

		    lua_pushlightuserdata (L, (void *)b);
		    lua_gettable (L, -3);
		    lua_replace (L, -3);

		    lua_call (L, 2, 3);
	
		    parameters[0] = lua_tonumber (L, -3);
		    parameters[1] = lua_tonumber (L, -2);

		    m = (int)lua_tonumber (L, -1);
		    m = m < j ? m : j;

		    lua_pop(L, 3);
		} else {
		    parameters[0] = 0;
		    parameters[1] = 0;
		    m = j;
		}

		/* Create the collision joints. */
	    
		for (i = 0 ; i < m ; i += 1) {
		    dJointID joint;

		    contacts[i].surface.mode = dContactBounce |
			                       dContactApprox1 |
			                       dContactFDir1;
		    
		    contacts[i].surface.mu = parameters[0];
		    contacts[i].surface.bounce = parameters[1];
		    contacts[i].surface.bounce_vel = 0.01;

/* 		    if (dGeomGetClass (b) == dCapsuleClass || */
/* 			dGeomGetClass (a) == dCapsuleClass) { */
/* 			float ks = 0.1e9, kd = 0; */

/* 			puts ("*"); */
			
/* 			contacts[i].surface.mode |= */
/* 			    dContactSoftCFM | */
/* 			    dContactSoftERP; */
			    
/* 			contacts[i].surface.soft_erp = 1e-4 * ks / (1e-4 * ks + kd); */
/* 			contacts[i].surface.soft_cfm = 1 / (1e-4 * ks + kd); */
/* 		    } */
		    
		    joint = dJointCreateContact (world, group, &contacts[i]);
		    dJointAttach (joint, p, q);
		}
	    }
	}
    }
}

int simulator_index (lua_State *L)
{
    const char *k;
    int i;

    k = lua_tostring(L, 2);

    if (!strcmp(k, "stepsize")) {
        lua_pushnumber(L, stepsize);
    } else if (!strcmp(k, "timescale")) {
        lua_pushnumber(L, timescale);
    } else if (!strcmp(k, "iterations")) {
        lua_pushnumber(L, iterations);
    } else if (!strcmp(k, "time")) {
        lua_pushnumber(L, now);
    } else if (!strcmp(k, "collision")) {
        lua_rawgeti(L, LUA_REGISTRYINDEX, collision);
    } else if (!strcmp(k, "tolerance")) {
	lua_newtable (L);
	lua_pushnumber (L, dWorldGetCFM (world));
	lua_rawseti (L, -2, 1);
	lua_pushnumber (L, dWorldGetERP (world));
	lua_rawseti (L, -2, 2);
    } else if (!strcmp(k, "popvelocity")) {
	lua_pushnumber(L, dWorldGetContactMaxCorrectingVel (world));
    } else if (!strcmp(k, "surfacelayer")) {
	lua_pushnumber(L, dWorldGetContactSurfaceLayer (world));
    } else if (!strcmp(k, "gravity")) {
	dVector3 gravity;

	dWorldGetGravity (world, gravity);
	
        lua_newtable(L);
        
        for(i = 0; i < 3; i += 1) {
            lua_pushnumber(L, gravity[i]);
            lua_rawseti(L, -2, i + 1);
        }
    } else {
	lua_rawget (L, 1);
    }

    return 1;
}

int simulator_newindex (lua_State *L)
{
    const char *k;
    int i;

    k = lua_tostring(L, 2);

    if (!strcmp(k, "stepsize")) {
        stepsize = lua_tonumber(L, 3);
    } else if (!strcmp(k, "timescale")) {
        timescale = lua_tonumber(L, 3);
    } else if (!strcmp(k, "iterations")) {
        iterations = lua_tonumber(L, 3);

	dWorldSetQuickStepNumIterations (world, iterations);
    } else if (!strcmp(k, "collision")) {
        luaL_unref(L, LUA_REGISTRYINDEX, collision);
        collision = luaL_ref(L, LUA_REGISTRYINDEX);
    } else if (!strcmp(k, "time")) {
        struct timespec now;
	double t;

	t = lua_tonumber(L, 3);

	clock_gettime (CLOCK_REALTIME, &now);

	once.tv_sec = now.tv_sec - (int)floor (t);
	once.tv_nsec = now.tv_nsec - (int)((t - floor (t)) * 1e9);
    } else if (!strcmp(k, "tolerance")) {
	lua_rawgeti (L, 3, 1);
	dWorldSetCFM (world, lua_tonumber (L, -1));
	lua_rawgeti (L, 3, 2);
	dWorldSetERP (world, lua_tonumber (L, -1));
	lua_pop(L, 2);
    } else if (!strcmp(k, "popvelocity")) {
	dWorldSetContactMaxCorrectingVel (world, lua_tonumber(L, 3));
    } else if (!strcmp(k, "surfacelayer")) {
	dWorldSetContactSurfaceLayer (world, lua_tonumber(L, 3));
    } else if (!strcmp(k, "gravity")) {
	dVector3 gravity;
	
        if(lua_istable(L, 3)) {
            for(i = 0 ; i < 3 ; i += 1) {
                lua_rawgeti(L, 3, i + 1);
                gravity[i] = lua_tonumber(L, -1);
                
                lua_pop(L, 1);
            }
        }

	dWorldSetGravity (world,
			  gravity[0],
			  gravity[1],
			  gravity[2]);
    } else {
	lua_rawset (L, 1);
    }

    return 0;
}

@implementation Node

+(void) load
{
    /* Create an ODE world. */

    dInitODE();
    
    world = dWorldCreate();
    group = dJointGroupCreate (0);
    space = dSimpleSpaceCreate (NULL);
    
    stepsize = 0.01;
    iterations = 0;
    timescale = 1;
    
    dSpaceSetCleanup (space, 0);
    dWorldSetGravity (world, 0, 0, -9.81);
}

-(Node *) init
{
    int i, j;
    
    [super init];
    
    self->link = LUA_REFNIL;
    self->unlink = LUA_REFNIL;
    self->transform = LUA_REFNIL;
    self->traverse = LUA_REFNIL;
    self->prepare = LUA_REFNIL;
    self->cleanup = LUA_REFNIL;
    
    self->linked = 0;

    self->up = NULL;
    self->down = NULL;
    self->left = NULL;
    self->right = NULL;

    for (i = 0 ; i < 3 ; i += 1) {
	for (j = 0 ; j < 3 ; j += 1) {
	    self->orientation[i * 3 + j] = i == j ? 1 : 0;
	}

	self->position[i] = 0;
    }

    return self;
}

-(int) linked
{
    return self->linked;
}

-(GLfloat *) position
{
    return self->position;
}

-(GLfloat *) orientation
{
    return self->orientation;
}

-(GLfloat *) translation
{
    return self->translation;
}

-(GLfloat *) rotation
{
    return self->rotation;
}

-(dWorldID) world
{
    return world;
}
 
-(dSpaceID) space
{
    return space;
}
 
-(dJointGroupID) group
{
    return group;
}

-(void) toggle: (lua_State *)L
{
    id child;

    self->linked = !self->linked;

    if (linked) {
	callhooks(L, self->link);
    } else {
	callhooks(L, self->unlink);
    }
    
    /* Recurse on the node's children. */
    
    for(child = [self children] ; child ; child = [child sister]) {
        [child toggle: L];
    }
}

-(void) transform: (lua_State *)L
{
    id child;
    
    GLfloat *r, *R, *p, *W;
    int i, j;

    r = [[self parent] translation];
    R = [[self parent] rotation];

    p = [self position];
    W = [self orientation];

    /* Call the node's transform hook before
       doing anything else. */
    
    callhooks(L, self->transform);
    
    if ([self parent]) {
	for (i = 0 ; i < 3 ; i += 1) {
	    for (j = 0 ; j < 3 ; j += 1) {
		self->rotation[i * 3 + j] = R[i * 3] * W[j] +
		                            R[i * 3 + 1] * W[j + 3] +
		                            R[i * 3 + 2] * W[j + 6];
	    }

	    self->translation[i] = R[i * 3] * p[0] +
		                   R[i * 3 + 1] * p[1] +
		                   R[i * 3 + 2] * p[2] +
		                   r[i]; 
	}
    } else {
	struct timespec time;    
	double t, dt_0, dt;
	dReal h;

	if (once.tv_sec == 0) {
	    clock_gettime (CLOCK_REALTIME, &once);
	}
	
	/* Simulate the system forward. */

	clock_gettime (CLOCK_REALTIME, &time);   
	now = time.tv_sec - once.tv_sec + (time.tv_nsec - once.tv_nsec) / 1e9;

	if (then == 0) {
	    then = now;
	}

	for (t = 0,
	     dt_0 = timescale * (now - then),
	     dt = dt_0;
	     dt > 0;
	     t += stepsize, dt = dt_0 - t) {
	    
	    dJointGroupEmpty (group);
	    dSpaceCollide (space, (void *)L, callback);

	    h = dt > stepsize ? stepsize : dt;
    
	    if (iterations > 0) {
		dWorldQuickStep (world, h);
	    } else {
		dWorldStep (world, h);
	    }
	}

	then = now;

	/* Just return the position and orientation. */
	
	for (i = 0 ; i < 3 ; i += 1) {
	    self->translation[i] = self->position[i];
	}

	for (i = 0 ; i < 9 ; i += 1) {
	    self->rotation[i] = self->orientation[i];
	}
    }

    for(child = [self children] ; child ; child = [child sister]) {
	[child transform: L];
    }
}

-(void) prepare: (lua_State *)L
{
    id child;
    
    callhooks(L, self->prepare);
    
    for(child = [self children] ; child ; child = [child sister]) {
	[child prepare: L];
    }
}

-(void) traverse: (lua_State *)L
{
    id child;

    callhooks(L, self->traverse);

    for(child = [self children] ; child ; child = [child sister]) {
	[child traverse: L];
    }
}

-(void) cleanup: (lua_State *)L
{
    id child;
    
    callhooks(L, self->cleanup);
    
    for(child = [self children] ; child ; child = [child sister]) {
	[child cleanup: L];
    }
}

-(void) get: (lua_State *)L
{
    const char *k;
    int i;

    k = lua_tostring(L, 2);

    if (!strcmp(k, "position")) {
        lua_newtable(L);
        
        for(i = 0; i < 3; i += 1) {
            lua_pushnumber(L, self->position[i]);
            lua_rawseti(L, -2, i + 1);
        }
    } else if (!strcmp(k, "orientation")) {
        lua_newtable(L);
        
        for(i = 0; i < 9; i += 1) {
            lua_pushnumber(L, self->orientation[i]);
            lua_rawseti(L, -2, i + 1);
        }
    } else if (!strcmp(k, "link")) {
        lua_rawgeti(L, LUA_REGISTRYINDEX, self->link);
    } else if (!strcmp(k, "unlink")) {
        lua_rawgeti(L, LUA_REGISTRYINDEX, self->unlink);
    } else if (!strcmp(k, "transform")) {
        lua_rawgeti(L, LUA_REGISTRYINDEX, self->transform);
    } else if (!strcmp(k, "traverse")) {
        lua_rawgeti(L, LUA_REGISTRYINDEX, self->traverse);
    } else if (!strcmp(k, "prepare")) {
        lua_rawgeti(L, LUA_REGISTRYINDEX, self->prepare);
    } else if (!strcmp(k, "cleanup")) {
        lua_rawgeti(L, LUA_REGISTRYINDEX, self->cleanup);
    } else  {
	lua_getmetatable(L, 1);
	lua_replace(L, 1);
	lua_gettable(L, 1);
    }
}

-(void) set: (lua_State *)L
{    
    const char *k;
    int i;

    k = lua_tostring(L, 2);

    if (!strcmp(k, "position")) {
        if(lua_istable(L, 3)) {
            for(i = 0 ; i < 3 ; i += 1) {
                lua_rawgeti(L, 3, i + 1);
                self->position[i] = lua_tonumber(L, -1);
                
                lua_pop(L, 1);
            }
        }
    } else if (!strcmp(k, "orientation")) {
        if(lua_istable(L, 3)) {
            for(i = 0 ; i < 9 ; i += 1) {
                lua_rawgeti(L, 3, i + 1);
                self->orientation[i] = lua_tonumber(L, -1);
                
                lua_pop(L, 1);
            }
        }
    } else if (!strcmp(k, "unlink")) {
        luaL_unref(L, LUA_REGISTRYINDEX, self->unlink);
        self->unlink = luaL_ref(L, LUA_REGISTRYINDEX);
    } else if (!strcmp(k, "link")) {
        luaL_unref(L, LUA_REGISTRYINDEX, self->link);
        self->link = luaL_ref(L, LUA_REGISTRYINDEX);
    } else if (!strcmp(k, "transform")) {
        luaL_unref(L, LUA_REGISTRYINDEX, self->transform);
        self->transform = luaL_ref(L, LUA_REGISTRYINDEX);
    } else if (!strcmp(k, "traverse")) {
        luaL_unref(L, LUA_REGISTRYINDEX, self->traverse);
        self->traverse = luaL_ref(L, LUA_REGISTRYINDEX);
    } else if (!strcmp(k, "prepare")) {
        luaL_unref(L, LUA_REGISTRYINDEX, self->prepare);
        self->prepare = luaL_ref(L, LUA_REGISTRYINDEX);
    } else if (!strcmp(k, "cleanup")) {
        luaL_unref(L, LUA_REGISTRYINDEX, self->cleanup);
        self->cleanup = luaL_ref(L, LUA_REGISTRYINDEX);
    } else {
	Node *child;
	const char *name;
    
	/* Check if there's a node linked with
	   the same key and if so free it. */
	
	lua_getmetatable(L, 1);
	lua_pushvalue(L, 2);
	lua_gettable(L, -2);
        
	if(lua_isuserdata(L, -1)) {
	    child = *(Node **)lua_touserdata(L, -1);
            
	    if([child linked]) {
		[child toggle: L];
	    }

	    [self renounce: child];
	}

	lua_pop(L, 1);
        
	/* Link the new node. */
	
	if(lua_isuserdata(L, 3)) {
	    child = *(Node **)lua_touserdata(L, 3);

	    if ([child isKindOf: [Node class]]) {
		name = lua_tostring(L, 2);

		[self adopt: child named: name];
		
		if([self linked]) {
		    [child toggle: L];
		}
	    }
	}
        
	/* Update the children table. */

	lua_replace(L, 1);
	lua_settable(L, 1);
    }
}

@end
