/* Copyright (C) 2008 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <lua.h>
#include <lauxlib.h>
#include <GL/gl.h>

#include "flat.h"
#include "fixed.h"
#include "shadow.h"
#include "lambert.h"
#include "phong.h"
#include "blinn.h"
#include "minnaert.h"
#include "oren.h"
#include "light.h"
#include "fog.h"
#include "ambient.h"

#include "texture.h" 
#include "clamped.h" 
#include "periodic.h" 
#include "mirrored.h" 

#include "static.h" 

static char *decapitalize (char *s)
{
    s[0] = tolower(s[0]);

    return s;
}

static int generic_tostring(lua_State *L)
{
    id N;

    N = *(id *)lua_touserdata(L, 1);
    lua_pushstring(L, [N name]);
   
    return 1;
}

static int generic_index(lua_State *L)
{
    id N;

    N = *(id *)lua_touserdata(L, 1);
    
    [N get: L];
    
    return 1;
}

static int generic_newindex(lua_State *L)
{
    id N;

    N = *(id *)lua_touserdata(L, 1);
    
    [N set: L];

    return 0;
}

static int generic_gc(lua_State *L)
{
    id N;

    N = *(id *)lua_touserdata(L, 1);

    [N free];

    return 0;
}

static int static_len(lua_State *L)
{
    id M;

    M = *(id *)lua_touserdata(L, 1);

    lua_newtable(L);
    
    lua_pushnumber(L, [M vertices]);
    lua_rawseti(L, -2, 1);

    lua_pushnumber(L, [M indices] / 3);
    lua_rawseti(L, -2, 2);

    return 1;
}

static int static_call (lua_State *L)
{
    id N;

    N = *(id *)lua_touserdata(L, 1);

    /* Create the userdata... */
	
    *(id *)lua_newuserdata(L, sizeof(id)) = [[N copy] init];
    
    lua_newtable (L);
    lua_pushstring(L, "__len");
    lua_pushcfunction(L, static_len);
    lua_settable(L, -3);
    lua_pushstring(L, "__index");
    lua_pushcfunction(L, generic_index);
    lua_settable(L, -3);
    lua_pushstring(L, "__newindex");
    lua_pushcfunction(L, generic_newindex);
    lua_settable(L, -3);
    lua_pushstring(L, "__tostring");
    lua_pushcfunction(L, generic_tostring);
    lua_settable(L, -3);
    lua_pushstring(L, "__gc");
    lua_pushcfunction(L, generic_gc);
    lua_settable(L, -3);
    lua_pushstring(L, "__reference");
    lua_pushvalue(L, 1);
    lua_settable(L, -3);
    lua_setmetatable(L, -2);
    
    /* ...and initialize it. */

    if(lua_istable(L, 2)) {
	lua_pushnil(L);
	
	while(lua_next(L, 2)) {
	    lua_pushvalue(L, -2);
	    lua_insert(L, -2);
	    lua_settable(L, 3);
	}
    }

    return 1;
}

static int static_tostring(lua_State *L)
{
    id M;

    M = *(id *)lua_touserdata(L, 1);

    lua_pushstring(L, [M name]);
    lua_pushstring(L, " geometry foundry");
    lua_concat(L, 2);
    
    return 1;
}

static int static_gc(lua_State *L)
{
    id N;

    N = *(id *)lua_touserdata(L, 1);

    [N freeBuffers];
    [N free];

    return 0;
}

static int constructstatic(lua_State *L)
{
    id M;
    GLfloat *vertices;
    GLushort *indices;
    int i, size[2];
    
    luaL_checktype(L, 1, LUA_TTABLE);

    lua_pushstring(L, "size");
    lua_gettable(L, 1);

    if(lua_istable(L, -1)) {
        for(i = 0 ; i < 2 ; i += 1) {
            lua_rawgeti(L, -1, i + 1);
            size[i] = lua_tonumber(L, -1);
                
            lua_pop(L, 1);
        }
    }
    
    lua_pop(L, 1);

    lua_pushstring(L, "vertices");
    lua_gettable(L, 1);
    
    /* Get the vertices. */

    vertices = (GLfloat *)malloc(8 * size[0] * sizeof(GLfloat));

    for(i = 0 ; i < 8 * size[0] ; i += 1) {
	lua_rawgeti(L, -1, i + 1);
	    
	vertices[i] = (GLfloat)lua_tonumber(L, -1);
	
	lua_pop(L, 1);
    }
    
    lua_pushstring(L, "indices");
    lua_gettable(L, 1);

    /* And the indices. */

    indices = (GLushort *)malloc(size[1] * sizeof(GLushort));

    for(i = 0 ; i < size[1] ; i += 1) {
	lua_rawgeti(L, -1, i + 1);
	
	indices[i] = (GLushort)lua_tonumber(L, -1);
	
	lua_pop(L, 1);
    }

    /* Create and initialize the static userdata. */
    
    M = [[Static alloc] initWithVertices: vertices
	                andIndices: indices
	                ofSize: size];
    
    *(id *)lua_newuserdata(L, sizeof(id)) = M;

    lua_newtable(L);
    lua_pushstring(L, "__tostring");
    lua_pushcfunction(L, static_tostring);
    lua_settable(L, -3);
    lua_pushstring(L, "__call");
    lua_pushcfunction(L, static_call);
    lua_settable(L, -3);
    lua_pushstring(L, "__gc");
    lua_pushcfunction(L, static_gc);
    lua_settable(L, -3);
    lua_setmetatable(L, -2);

    free(vertices);
    free(indices);

    return 1;
}

static int texture_len(lua_State *L)
{
    id T;

    T = *(id *)lua_touserdata(L, 1);

    lua_newtable(L);
    
    lua_pushnumber(L, [T width]);
    lua_rawseti(L, -2, 1);

    lua_pushnumber(L, [T height]);
    lua_rawseti(L, -2, 2);

    lua_pushnumber(L, [T components]);
    lua_rawseti(L, -2, 3);

    return 1;
}

static int texture_tostring(lua_State *L)
{
    id T;

    T = *(id *)lua_touserdata(L, 1);

    lua_pushstring(L, [T name]);
    lua_pushstring(L, " texture map");
    lua_concat(L, 2);
    
    return 1;
}

int constructtexture(lua_State *L)
{
    Class class;
    id T;
    unsigned char *pixels;
    int i, size[3];
    
    luaL_checktype(L, 1, LUA_TTABLE);

    lua_pushstring(L, "size");
    lua_gettable(L, 1);

    if(lua_istable(L, -1)) {
        for(i = 0 ; i < 3 ; i += 1) {
            lua_rawgeti(L, -1, i + 1);
            size[i] = lua_tonumber(L, -1);
                
            lua_pop(L, 1);
        }
    }
    
    lua_pop(L, 1);

    lua_pushstring(L, "pixels");
    lua_gettable(L, 1);
    
    pixels = (unsigned char *)malloc(size[0] * size[1] * size[2] *
				     sizeof(unsigned char));

    for(i = 0 ; i < size[0] * size[1] * size[2] ; i += 1) {
	lua_rawgeti(L, -1, i + 1);
	    
	pixels[i] = (unsigned char)(lua_tonumber(L, -1));
	
	lua_pop(L, 1);
    }

    /* Create and initialize the texture userdata. */

    lua_pushvalue (L, lua_upvalueindex (1));
    class = (Class)lua_touserdata(L, -1);
    lua_pop(L, 1);

    T = [[class alloc] initWithPixels: pixels
		       ofSize: size];
    
    *(id *)lua_newuserdata(L, sizeof(id)) = T;

    lua_newtable(L);
    lua_pushstring(L, "__tostring");
    lua_pushcfunction(L, texture_tostring);
    lua_settable(L, -3);
    lua_pushstring(L, "__len");
    lua_pushcfunction(L, texture_len);
    lua_settable(L, -3);
    lua_pushstring(L, "__gc");
    lua_pushcfunction(L, generic_gc);
    lua_settable(L, -3);
    lua_setmetatable(L, -2);

    free (pixels);
    
    return 1;
}

static int constructsurface (lua_State *L)
{
    Class class;

    lua_pushvalue (L, lua_upvalueindex (1));
    class = (Class)lua_touserdata(L, -1);
    lua_pop(L, 1);

    /* Create the userdata... */
    
    *(id *)lua_newuserdata(L, sizeof(id)) = [[class alloc] init];
    
    lua_newtable (L);
    lua_pushstring(L, "__index");
    lua_pushcfunction(L, (lua_CFunction)generic_index);
    lua_settable(L, -3);
    lua_pushstring(L, "__newindex");
    lua_pushcfunction(L, (lua_CFunction)generic_newindex);
    lua_settable(L, -3);
    lua_pushstring(L, "__tostring");
    lua_pushcfunction(L, (lua_CFunction)generic_tostring);
    lua_settable(L, -3);
    lua_pushstring(L, "__gc");
    lua_pushcfunction(L, (lua_CFunction)generic_gc);
    lua_settable(L, -3);
    lua_setmetatable(L, -2);
    
    /* ...and initialize it. */

    if(lua_istable(L, 1)) {
	lua_pushnil(L);
	
	while(lua_next(L, 1)) {
	    lua_pushvalue(L, -2);
	    lua_insert(L, -2);
	    lua_settable(L, 2);
	}
    }

    return 1;
}

int luaopen_shading (lua_State *L)
{
    int i;
    
    const luaL_Reg meshes[] = {
	{"static", constructstatic},
	{NULL, NULL}
    };

    Class surfaces[] = {
	[Light class], [Lambert class],
	[Phong class], [Blinn class], [Minnaert class],
	[Oren class], [Flat class], [Ambient class],
	[Fog class], [Fixed class], [Shadow class]
    };	
    
    Class textures[] = {
	[Clamped class], [Periodic class], [Mirrored class]
    };

    luaL_register (L, "meshes", meshes);
    
    /* Create the texture constructors. */
    
    lua_newtable (L);
    
    for (i = 0 ; i < sizeof(textures) / sizeof(textures[0]) ; i += 1) {
	lua_pushlightuserdata (L, textures[i]);
	lua_pushcclosure (L, constructtexture, 1);
	lua_setfield(L, -2, decapitalize(strdupa([textures[i] name])));
    }

    lua_setglobal (L, "textures");

    /* And the surface node constructors. */

    lua_newtable (L);
    
    for (i = 0 ; i < sizeof(surfaces) / sizeof(surfaces[0]) ; i += 1) {
	lua_pushlightuserdata (L, surfaces[i]);
	lua_pushcclosure (L, constructsurface, 1);
	lua_setfield(L, -2, decapitalize(strdupa([surfaces[i] name])));
    }

    lua_setglobal (L, lua_tostring (L, 1));

    return 0;
}
