/* Copyright (C) 2008 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <AL/al.h>
#include <lua.h>
#include <lualib.h>
#include <lauxlib.h>
#include <time.h>

#include "source.h"

@implementation Source

-(Source *) init
{
    [super init];

    alGenSources (1, &self->index);
    
    return self;
}

-(void) free
{
    alDeleteSources (1, &self->index);
}

-(ALint) index
{
    return self->index;
}

-(void) toggle: (lua_State *)L
{
    struct timespec time;
    
    if (!linked) {
	clock_gettime (CLOCK_REALTIME, &time);
    
	self->tick = time.tv_sec + time.tv_nsec / 1e9;
	alSourcePlay (self->index);
    } else {
	alSourcePause (self->index);
    }

    [super toggle: L];
}

-(void) transform: (lua_State *)L
{
    struct timespec time;
    ALfloat v[3];
    double now, dt;
    float *r, *r_0;

    clock_gettime (CLOCK_REALTIME, &time);
    now = time.tv_sec + time.tv_nsec / 1e9;
    dt = now - self->tick;
    self->tick = now;
    
    r_0 = [self translation];

    [super transform: L];

    r = [self translation];
    
    v[0] = (r[0] - r_0[0]) / dt;
    v[1] = (r[1] - r_0[1]) / dt;
    v[2] = (r[2] - r_0[2]) / dt;

    alSourcefv (self->index, AL_POSITION, r);    
    alSourcefv (self->index, AL_VELOCITY, v);    
}

-(void) get: (lua_State *)L
{    
    const char *k;
    
    k = lua_tostring(L, -1);

    if (!strcmp(k, "gain")) {
	ALfloat gain;

	alGetSourcef (self->index, AL_GAIN, &gain);

	lua_pushnumber (L, gain);
    } else if (!strcmp(k, "bounds")) {
	ALfloat gain;
	
        lua_newtable(L);
	
	alGetSourcef (self->index, AL_MIN_GAIN, &gain);
	lua_pushnumber (L, gain);
	lua_rawseti (L, -2, 1);
        
	alGetSourcef (self->index, AL_MAX_GAIN, &gain);
	lua_pushnumber (L, gain);
	lua_rawseti (L, -2, 2);
    } else if (!strcmp(k, "reference")) {
	ALfloat distance;

	alGetSourcef (self->index, AL_REFERENCE_DISTANCE, &distance);
	lua_pushnumber (L, distance);
    } else if (!strcmp(k, "rolloff")) {
	ALfloat factor;

	alGetSourcef (self->index, AL_ROLLOFF_FACTOR, &factor);
	lua_pushnumber (L, factor);
    } else if (!strcmp(k, "pitch")) {
	ALfloat factor;

	alGetSourcef (self->index, AL_PITCH, &factor);
	lua_pushnumber (L, factor);
    } else if (!strcmp(k, "state")) {
	ALenum state;
	
	alGetSourcei (self->index, AL_SOURCE_STATE, &state);

	if (state == AL_PLAYING) {
	    lua_pushliteral (L, "playing");
	} else if (state == AL_PAUSED) {
	    lua_pushliteral (L, "paused");
	} else if (state == AL_STOPPED || state == AL_INITIAL) {
	    lua_pushliteral (L, "stopped");
	} else {
	    lua_pushliteral (L, "huh?");
	}
    } else {
	[super get: L];
    }
}

-(void) set: (lua_State *)L
{
    const char *k;
    int i;

    k = lua_tostring(L, -2);

    if (!strcmp(k, "gain")) {
	alSourcef (self->index, AL_GAIN, lua_tonumber (L, 3));
    } else if (!strcmp(k, "bounds")) {
	if (lua_istable (L, 3)) {
	    lua_rawgeti (L, 3, 1);
	    alSourcef (self->index, AL_MIN_GAIN, lua_tonumber (L, -1));

	    lua_rawgeti (L, 3, 2);
	    alSourcef (self->index, AL_MAX_GAIN, lua_tonumber (L, -1));

	    lua_pop (L, 2);
	}
    } else if (!strcmp(k, "reference")) {
	alSourcef (self->index, AL_REFERENCE_DISTANCE, lua_tonumber (L, 3));
    } else if (!strcmp(k, "rolloff")) {
	alSourcef (self->index, AL_ROLLOFF_FACTOR, lua_tonumber (L, 3));
    } else if (!strcmp(k, "pitch")) {
	alSourcef (self->index, AL_PITCH, lua_tonumber (L, 3));
    } else if (!strcmp(k, "queue")) {
	alSourcef (self->index, AL_BUFFER, 0);

	if (lua_istable (L, 3)) {
	    id waveform;
	    ALuint buffer;
	    
	    for (i = 0 ; i < lua_objlen (L, 3) ; i += 1) {
		lua_rawgeti (L, 3, i + 1);

		waveform = *(id *)lua_touserdata(L, -1);
		buffer = [waveform index];
		
		alSourceQueueBuffers(self->index, 1, &buffer);

		lua_pop (L, 1);
	    }

	    if([self linked]) {
		alSourcePlay(self->index);
	    }
	}
    } else {
	[super set: L];
    }
}

@end
