/* Copyright (C) 2008 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <lua.h>
#include <lauxlib.h>
#include "timer.h"

static void calltimedhooks (lua_State *L, int ref,
			    int tick, double delta, double elapsed)
{
    if (ref != LUA_REFNIL) {
	lua_rawgeti(L, LUA_REGISTRYINDEX, ref);
	    
	if (lua_isfunction (L, -1)) {
	    lua_pushnumber (L, tick);
	    lua_pushnumber (L, delta);
	    lua_pushnumber (L, elapsed);
	    lua_call(L, 3, 0);
	} else if (lua_istable (L, -1)) {
	    lua_pushnil(L);
	
	    while(lua_next(L, -2)) {
		lua_pushnumber (L, tick);
		lua_pushnumber (L, delta);
		lua_pushnumber (L, elapsed);
		lua_call (L, 3, 0);
	    }
	    lua_pop (L, 1);
	} else {
	    lua_pop (L, 1);
	}
    }
}

@implementation Timer

-(Timer *) init
{
    [super init];

    self->timedlink = LUA_REFNIL;
    self->timedunlink = LUA_REFNIL;
    self->timedtransform = LUA_REFNIL;
    self->timedtraverse = LUA_REFNIL;
    self->timedprepare = LUA_REFNIL;
    self->timedcleanup = LUA_REFNIL;
    
    self->period = 1;
    
    return self;
}

-(void) toggle: (lua_State *)L
{
    [super toggle: L];

    if ([self linked]) {
	clock_gettime (CLOCK_REALTIME, &self->checkpoint);
	self->elapsed = 0;
	self->delta = 0;
	self->tick = 0;
	self->fire = 0;

	calltimedhooks(L, self->timedlink, self->tick,
		       self->delta, self->elapsed);
    } else {
	calltimedhooks(L, self->timedunlink, self->tick,
		       self->delta, self->elapsed);
    }
}

-(void) transform: (lua_State *)L
{
    struct timespec time;

    clock_gettime (CLOCK_REALTIME, &time);   

    self->delta = time.tv_sec - self->checkpoint.tv_sec +
	         (time.tv_nsec - self->checkpoint.tv_nsec) / 1e9;

    self->elapsed += self->delta;

    if (self->delta > self->period) {
	self->checkpoint = time;
	self->tick += 1;
	self->fire = 1;
    } else {
	self->fire = 0;
    }

    if (self->fire) {
	calltimedhooks(L, self->timedtransform, self->tick,
		       self->delta, self->elapsed);
    }
    
    [super transform: L];
}

-(void) traverse: (lua_State *)L
{
    if (self->fire) {
	calltimedhooks(L, self->timedtraverse, self->tick,
		       self->delta, self->elapsed);
    }
    
    [super traverse: L];
}

-(void) prepare: (lua_State *)L
{
    if (self->fire) {
	calltimedhooks(L, self->timedprepare, self->tick,
		       self->delta, self->elapsed);
    }
    
    [super prepare: L];
}

-(void) cleanup: (lua_State *)L
{
    if (self->fire) {
	calltimedhooks(L, self->timedcleanup, self->tick,
		       self->delta, self->elapsed);
    }
    
    [super cleanup: L];
}

-(void) get: (lua_State *)L
{
    const char *k;

    k = lua_tostring(L, 2);

    if (!strcmp(k, "period")) {
        lua_pushnumber(L, self->period);
    } else if (!strcmp(k, "link")) {
        lua_rawgeti(L, LUA_REGISTRYINDEX, self->timedlink);
    } else if (!strcmp(k, "unlink")) {
        lua_rawgeti(L, LUA_REGISTRYINDEX, self->timedunlink);
    } else if (!strcmp(k, "transform")) {
        lua_rawgeti(L, LUA_REGISTRYINDEX, self->timedtransform);
    } else if (!strcmp(k, "traverse")) {
        lua_rawgeti(L, LUA_REGISTRYINDEX, self->timedtraverse);
    } else if (!strcmp(k, "prepare")) {
        lua_rawgeti(L, LUA_REGISTRYINDEX, self->timedprepare);
    } else if (!strcmp(k, "cleanup")) {
        lua_rawgeti(L, LUA_REGISTRYINDEX, self->timedcleanup);
    } else  {
	[super get: L];
    }
}

-(void) set: (lua_State *)L
{    
    const char *k;

    k = lua_tostring(L, 2);

    if (!strcmp(k, "period")) {
        self->period = lua_tonumber(L, 3);
    } else if (!strcmp(k, "unlink")) {
        luaL_unref(L, LUA_REGISTRYINDEX, self->timedunlink);
        self->timedunlink = luaL_ref(L, LUA_REGISTRYINDEX);
    } else if (!strcmp(k, "link")) {
        luaL_unref(L, LUA_REGISTRYINDEX, self->timedlink);
        self->timedlink = luaL_ref(L, LUA_REGISTRYINDEX);
    } else if (!strcmp(k, "transform")) {
        luaL_unref(L, LUA_REGISTRYINDEX, self->timedtransform);
        self->timedtransform = luaL_ref(L, LUA_REGISTRYINDEX);
    } else if (!strcmp(k, "traverse")) {
        luaL_unref(L, LUA_REGISTRYINDEX, self->timedtraverse);
        self->timedtraverse = luaL_ref(L, LUA_REGISTRYINDEX);
    } else if (!strcmp(k, "prepare")) {
        luaL_unref(L, LUA_REGISTRYINDEX, self->timedprepare);
        self->timedprepare = luaL_ref(L, LUA_REGISTRYINDEX);
    } else if (!strcmp(k, "cleanup")) {
        luaL_unref(L, LUA_REGISTRYINDEX, self->timedcleanup);
        self->timedcleanup = luaL_ref(L, LUA_REGISTRYINDEX);
    } else {
	[super set: L];	
    }
}

@end
