/* Copyright (C) 2008 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <lua.h>
#include <lauxlib.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <time.h>
#include "cel.h"
#include "lamp.h"
#include "texture.h"

@implementation Cel

-(Cel *) init
{
    [super init];
    
    self->color[0] = 1;
    self->color[1] = 1;
    self->color[2] = 1;
    
    self->outline[0] = 0;
    self->outline[1] = 0;
    self->outline[2] = 0;

    self->thickness = 1;
    
    return self;
}

-(void) get
{
    const char *k;
    int i;

    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "color")) {
	lua_getmetatable (_L, 1);
	lua_replace (_L, 1);
	lua_gettable (_L, 1);
    } else if (!xstrcmp(k, "outline")) {
        lua_newtable (_L);
        
        for(i = 0; i < 3; i += 1) {
            lua_pushnumber (_L, self->outline[i]);
            lua_rawseti (_L, -2, i + 1);
        }
    } else if (!xstrcmp(k, "thickness")) {
	lua_pushnumber (_L, self->thickness);
    } else {
	[super get];
    }
}

-(void) set
{    
    const char *k;
    int i;

    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "color")) {
        if(lua_istable (_L, 3)) {
            for(i = 0 ; i < 3 ; i += 1) {
                lua_rawgeti (_L, 3, i + 1);
                self->color[i] = lua_tonumber (_L, -1);
                
                lua_pop (_L, 1);
            }
	    
	    self->texture = nil;
        } else if(lua_isnumber (_L, 3)) {
	    self->color[0] = lua_tonumber (_L, 3);
	    
            for(i = 1 ; i < 3 ; i += 1) {
                self->color[i] = 0;
            }
	    
	    self->texture = nil;
        } else if(lua_isuserdata (_L, 3)) {
            for(i = 0 ; i < 3 ; i += 1) {
                self->color[i] = 0;
            }
	    
	    self->texture = *(id *)lua_touserdata (_L, 3);
	}

	lua_getmetatable (_L, 1);
	lua_replace (_L, 1);
	lua_settable (_L, 1);
    } else if (!xstrcmp(k, "outline")) {
        if(lua_istable (_L, 3)) {
            for(i = 0 ; i < 3 ; i += 1) {
                lua_rawgeti (_L, 3, i + 1);
                self->outline[i] = lua_tonumber (_L, -1);
                
                lua_pop (_L, 1);
            }
        }
    } else if (!xstrcmp(k, "thickness")) {
	self->thickness = lua_tonumber (_L, -1);
    } else {
	[super set];
    }
}

-(void)traverse
{
    id parent, child;
    GLfloat M[16] = {0, 0, 0, 0,
		     0, 0, 0, 0,
		     0, 0, 0, 0,
		     0, 0, 0, 1};

    glUseProgramObjectARB (0);
    glEnable (GL_DEPTH_TEST);
    glEnable (GL_CULL_FACE);

    /* Bind the intensity texture. */
    
    glActiveTexture (GL_TEXTURE1);
    glEnable (GL_TEXTURE_1D);

    for (parent = [self parent] ; parent ; parent = [parent parent]) {
        for (child = [parent children] ; child ; child = [child sister]) {
            if ([child isMemberOf: [Lamp class]]) {
		M[0] = -[child rotation][2];
		M[4] = -[child rotation][5];
		M[8] = -[child rotation][8];
		M[12] = [child ambience];
		glBindTexture (GL_TEXTURE_1D, [child intensity]);
	    }
	}
    }

    /* Set up the texture matrix to perform the dot product. */
    
    glMatrixMode (GL_TEXTURE);
    glPushMatrix();
    glLoadMatrixf(M);

    /* Set up texture generation.  */
    
    glTexGeni (GL_S, GL_TEXTURE_GEN_MODE, GL_NORMAL_MAP);
    glTexGeni (GL_T, GL_TEXTURE_GEN_MODE, GL_NORMAL_MAP);
    glTexGeni (GL_R, GL_TEXTURE_GEN_MODE, GL_NORMAL_MAP);

    glEnable (GL_TEXTURE_GEN_S);
    glEnable (GL_TEXTURE_GEN_T);
    glEnable (GL_TEXTURE_GEN_R);

    /* Bind the color texture. */
    
    glActiveTexture (GL_TEXTURE0);
    if (self->texture) {
	glBindTexture (GL_TEXTURE_2D, [self->texture index]);
	glEnable (GL_TEXTURE_2D);

	glColor3f (1, 1, 1);
    } else {
	glDisable (GL_TEXTURE_2D);
	glColor3fv (self->color);
    }

    /* Draw and clean up. */
    
    [super traverse];
    [super deferTraversal];

    glMatrixMode (GL_TEXTURE);
    glPopMatrix();

    glDisable (GL_TEXTURE_GEN_S);
    glDisable (GL_TEXTURE_GEN_T);
    glDisable (GL_TEXTURE_GEN_R);

    glDisable (GL_CULL_FACE);
    glDisable (GL_DEPTH_TEST);

    if (self->texture) {
	glDisable (GL_TEXTURE_2D);
    }

    glActiveTexture (GL_TEXTURE0);
    glDisable (GL_TEXTURE_1D);
}

-(void) traverseDeferred
{
    /* Set up the state and draw the outlines. */
    
    glDepthMask(GL_FALSE);
    
    glEnable (GL_DEPTH_TEST);
    glEnable (GL_CULL_FACE);
    glEnable (GL_LINE_SMOOTH);
    glEnable (GL_BLEND);
    
    glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glPolygonMode (GL_FRONT_AND_BACK, GL_LINE);
    glCullFace (GL_FRONT);

    glLineWidth (self->thickness);
    glColor3fv (self->outline);

    [super traverse];

    glCullFace (GL_BACK);
    glPolygonMode (GL_FRONT_AND_BACK, GL_FILL);

    glDisable (GL_BLEND);
    glDisable (GL_POLYGON_OFFSET_LINE);
    glDisable (GL_LINE_SMOOTH);
    glDisable (GL_CULL_FACE);
    glDisable (GL_DEPTH_TEST);

    glDepthMask(GL_TRUE);
}

@end

