-- Copyright (C) 2008 Papavasileiou Dimitris                             
--                                                                      
-- This program is free software: you can redistribute it and/or modify 
-- it under the terms of the GNU General Public License as published by 
-- the Free Software Foundation, either version 3 of the License, or    
-- (at your option) any later version.                                  
--                                                                      
-- This program is distributed in the hope that it will be useful,      
-- but WITHOUT ANY WARRANTY; without even the implied warranty of       
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
-- GNU General Public License for more details.                         
--                                                                      
-- You should have received a copy of the GNU General Public License    
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

require "transforms"
require "transitions"
require "moremath"
require "moremeshes"
require "resources"
require (options.toon and "toon" or "shading")

require "textures"
require "accoustics"
require "switches"

local function respotball (ball, offset)
   if not offset then 
      offset = 0.25 * billiards.tablewidth
   end

   ball.position = {offset, 0, billiards.ballradius}
   ball.velocity = {0, 0, 0}
   ball.spin = {0, 0, 0}

   for _, other in pairs (bodies.balls) do
      local p = ball.position
      local q = other.position
      local r = billiards.ballradius

      if math.distance (p, q) <= 2 * r and other ~= ball then
	 if options.pool then
	    respotball (ball, q[1] - math.sqrt (4 * r * r - q[2] * q[2]) - 1e-3)
	 else
	    respotball (ball, offset > 0 and -offset or 0)
	 end
	 
	 break
      end
   end
end

meshes.ball = function ()
   local mesh = meshes.sphere (billiards.ballradius, 32, 16)
		 
   for i = 7, #mesh.vertices, 8 do
      mesh.vertices[i] = mesh.vertices[i] * 4 - 1.5
      mesh.vertices[i + 1] = mesh.vertices[i + 1] * 2 - 0.5
   end
   
   return meshes.static(mesh)
end

if options.eightball then
   textures.decals = {
      {0.97, 0.97, 0.82},
      resources.patched "billiards/imagery/diffuse/one.lc",
      resources.patched "billiards/imagery/diffuse/two.lc",
      resources.patched "billiards/imagery/diffuse/three.lc",
      resources.patched "billiards/imagery/diffuse/four.lc",
      resources.patched "billiards/imagery/diffuse/five.lc",
      resources.patched "billiards/imagery/diffuse/six.lc",
      resources.patched "billiards/imagery/diffuse/seven.lc",
      resources.patched "billiards/imagery/diffuse/eight.lc",
      resources.patched "billiards/imagery/diffuse/nine.lc",
      resources.patched "billiards/imagery/diffuse/ten.lc",
      resources.patched "billiards/imagery/diffuse/eleven.lc",
      resources.patched "billiards/imagery/diffuse/twelve.lc",
      resources.patched "billiards/imagery/diffuse/thirteen.lc",
      resources.patched "billiards/imagery/diffuse/fourteen.lc",
      resources.patched "billiards/imagery/diffuse/fifteen.lc",
   }
elseif options.nineball then
   textures.decals = {
      {0.97, 0.97, 0.82},
      resources.patched "billiards/imagery/diffuse/one.lc",
      resources.patched "billiards/imagery/diffuse/two.lc",
      resources.patched "billiards/imagery/diffuse/three.lc",
      resources.patched "billiards/imagery/diffuse/four.lc",
      resources.patched "billiards/imagery/diffuse/five.lc",
      resources.patched "billiards/imagery/diffuse/six.lc",
      resources.patched "billiards/imagery/diffuse/seven.lc",
      resources.patched "billiards/imagery/diffuse/eight.lc",
      resources.patched "billiards/imagery/diffuse/nine.lc",
   }
elseif options.experiment then
   textures.decals = {
      {0.97, 0.97, 0.82}
   }
else
   textures.decals = {
      {0.97, 0.97, 0.82},
      {0.92, 0.66, 0.18},
      {0.4, 0.12, 0.1},
   }
end

math.randomseed (os.clock())

-- Create the balls.

bodies.balls = {}

for i, decal in ipairs (textures.decals) do
   bodies.balls[i] = bodies.ball {
      position = billiards.opening[i],

      radius = billiards.ballradius,
      mass = physics.spheremass (3 * billiards.ballmass /
				 (4 * math.pi *
				  billiards.ballradius^3),
			         billiards.ballradius),

      lethargy = {0.01, 0.1, 2, 0},

      isball = true,

      shading = switches.button {
	 surface = options.toon and toon.cel {
	    color = decal,
	    mesh = meshes.ball()
	 } or shading.fresnel {
	    diffuse = decal,
	    specular = {0.7, 0.7, 0.7},
	    parameter = {96, 0.1},
      
	    mesh = meshes.ball()
	 },

	 selected = frames.event {
	    outline = shapes.halo {
	       color = {0.7, 0.6, 0.1},
	       opacity = 0.55,
	       width = 5.0,
	       geometry = meshes.ball()
	    },

	    buttonpress = function (self, button, x, y)
	       self.zero = {x, y}

	       if button == bindings.move and
		  self.ancestry[2].position[3] > 0 and
		  not bodies.observer.isaiming then
		  self.ancestry[2].home = self.ancestry[2].position
	       end

	       if button == bindings.ready then
		  if self.ancestry[2].position[3] < 0 then
		     respotball (self.ancestry[2])
		  elseif self.ancestry[2] == bodies.cueball then
		     if bodies.observer.islooking and not
			(bodies.cueball.ispocketed or
			 bodies.cueball.isout) then
			bodies.observer.isaiming = true
		     elseif bodies.observer.isaiming then
			bodies.observer.islooking = true
		     end
		  elseif bodies.observer.islooking then
		     bodies.cueball = self.ancestry[2]
		  end

		  bodies.observer.longitude = bodies.cueball.position[1]
		  bodies.observer.latitude = bodies.cueball.position[2]
	       end
	    end,

	    buttonrelease = function (self, button)
	       if self.ancestry[2].home then
		  self.ancestry[2].home = nil
	       end
	    end,

	    motion = function (self, button, x, y)
	       if self.ancestry[2].home then
		  local w, h, u, v, dx, dy

		  dx = x - self.zero[1]
		  dy = y - self.zero[2]
			
		  w = billiards.tablewidth / 2 - billiards.ballradius - 1e-3
		  h = billiards.tableheight / 2 - billiards.ballradius - 1e-3

		  u = transforms.fromnode(bodies.observer.eye, {0, 1, 0})
		  v = transforms.fromnode(bodies.observer.eye, {0, 0, -1})

		  u = math.normalize(math.project(u, {0, 0, 1}))
		  v = math.normalize(math.project(v, {0, 0, 1}))

		  self.ancestry[2].home = {
		     math.clamp(self.ancestry[2].home[1] +
				0.003 * (dx * u[1] - dy * v[1]),
			        -w, w),
		     math.clamp(self.ancestry[2].home[2] +
				0.003 * (dx * u[2] - dy * v[2]),
			        -h, h),
		     billiards.ballradius
		  }

		  bodies.observer.longitude = self.ancestry[2].home[1]
		  bodies.observer.latitude = self.ancestry[2].home[2]
	       end
	       
	       self.zero = {x, y}
	    end,
	 }
      },

      shadow = options.toon and frames.gimbal {
	 surface = toon.flat {
	    color = options.carom and {0.1, 0.1, 0} or {0, 0.1, 0},

	    toon.shadow {
	       position = {0, 0, -billiards.ballradius + 0.001},
	       mesh = meshes.ball(){
		  position = {0, 0, billiards.ballradius}
	       }
	    }
	 }
      },
   }
end

bodies.cueball = bodies.balls[1]

-- Add them to the scene.

graph.gear = bodies.system {

   ambience = not options.toon and shading.ambient {
      intensity = resources.clamped(options.pool and
				    "billiards/imagery/light/poolambience.lc" or
				    "billiards/imagery/light/caromambience.lc")
   },

   prepare = function (self)
		local isstable = true
		
		-- Check whether the shot is finished. A ball is
		-- stable if it is either resting or off the table
		
		for _, ball in pairs (bodies.balls) do
		   if not (ball.isout or
			   ball.isresting or
			   ball.ispocketed) then
		      isstable = false
		   end
		end

		-- Finished?

		if not isstable and
		   not bodies.observer.iswaiting then
		   bodies.observer.iswaiting = true
		end

		if bodies.observer.iswaiting and isstable then
		   bodies.observer.islooking = true
		end
	     end,

   unpack (bodies.balls)
}

-- Perturb the opening position slightly.

for _, ball in ipairs (bodies.balls) do
   physics.addforce (ball, {2 * (math.random() - 0.5),
			    2 * (math.random() - 0.5),
			    0})
end

-- Now for the simulation and control logic.

billiards.looking.newshot = function ()
   graph.transition = transitions.dissolve {
      duration = 0.7
   }

   -- Replace any balls that have fallen
   -- off the table to their opening spots

   for _, ball in pairs (bodies.balls) do
      if ball.isout then
	 respotball (ball)
      end
   end

   -- Set-up the camera.

   bodies.observer.radius = 1.5

   if bodies.cueball.ispocketed then
      bodies.observer.azimuth = math.rad(0)
      bodies.observer.elevation = math.rad(80)
      bodies.observer.longitude = 0.5 * billiards.tablewidth + 0.4
      bodies.observer.latitude = 0
   else
      bodies.observer.longitude = bodies.cueball.position[1]
      bodies.observer.latitude = bodies.cueball.position[2]
   end
end

billiards.aiming.reset = function ()
   graph.transition = transitions.dissolve {
      duration = 0.7
   }

   bodies.cue.elevation = 0
   bodies.cue.sidespin = 0
   bodies.cue.follow = 0

   bodies.observer.longitude = bodies.cueball.position[1]
   bodies.observer.latitude = bodies.cueball.position[2]
   bodies.observer.radius = 0.4
   bodies.observer.elevation = math.pi / 2 -
			       bodies.cue.elevation -
			       math.rad (5)
end

billiards.turning.aim = function ()
   bodies.cue.azimuth = bodies.observer.azimuth
end

billiards.striking.zoom = function ()
   bodies.observer.radius = 1.5
end

graph.controls = frames.event {
   keypress = function (self, key)
      if key == bindings.quit then
	 common.iterate = false
      elseif key == bindings.slower then
	 dynamics.timescale = dynamics.timescale / 2
      elseif key == bindings.faster then
	 dynamics.timescale = dynamics.timescale * 2
      elseif key == bindings.pause then
	 if dynamics.oldtimescale then
	    dynamics.timescale = dynamics.oldtimescale
	    dynamics.oldtimescale = nil
	 else
	    dynamics.oldtimescale = dynamics.timescale
	    dynamics.timescale = 0
	 end
      -- elseif key == 'w' then
      -- 	 for _, ball in pairs(bodies.balls) do
      -- 	    if ball.position[3] > 0 then
      -- 	       ball.position = {0, billiards.tableheight / 2 + 0.05, 0.1}
      -- 	       physics.wake (ball)
      -- 	       break
      -- 	    end
      -- 	 end
      else
	 print ("('" .. key .. "' pressed)")
      end
   end,

   keyrelease = function (self, key)
      print ("('" .. key .. "' released)")
   end
}
