/*
 * BMovieReviewer Copyright (C) 2009 Michael J. Beer
 * 
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package data;

import java.util.Iterator;
import java.util.LinkedList;

import javax.swing.ListModel;
import javax.swing.event.ListDataListener;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.TableModel;
import javax.swing.text.AttributeSet;
import javax.swing.text.BadLocationException;
import javax.swing.text.PlainDocument;

public class Wrappers {

	
	/**
	 * Verwaltet einen String
	 * Ersetzt dabei Zeilenumbrueche durch Leerzeichen
	 * @author mibeer
	 *
	 */
	public static class StringWrapper extends PlainDocument {

		private static final long serialVersionUID = 1L;

		protected int maxLen = 0;
		
		public StringWrapper(String s) {
			super();
			setS(s);
		}

		public StringWrapper(String s, int m) {
			super();
			setS(s);
			setMaxLen(m);
		}
		
		public String getS() {
			String eintrag;
			try {
				eintrag = getText(0, getLength());
			} catch (BadLocationException ex) {
				throw new RuntimeException();
			}
			return eintrag;
		}

		public void setS(String s) {
			if (s == null) {
				throw new IllegalArgumentException();
			}
			if(maxLen > 0 && s.length() > maxLen) {
				return;
			}
			try {
				remove(0, this.getLength());
				insertString(0, s.replaceAll("[\r\n]", " "), null);
			} catch (BadLocationException e) {
				throw new RuntimeException();
			}
		}

		public String toString() {
			return getS();
		}
		
		public void setMaxLen(int m) {
			if(m < 0) {
				throw new IllegalArgumentException();
			}
			maxLen = m;
		}
		
		public void insertString(int offs, String str, AttributeSet a)
		throws BadLocationException {
			if(maxLen > 0 && offs + str.length() > maxLen) {
				return;
			}
			super.insertString(offs, str.replaceAll("[\r\n]", " "), a);
		}
	}
	
	
	public static class TextWrapper extends StringWrapper {
		
		private static final long serialVersionUID = 1L;
		
		public TextWrapper(String s) {
			super(s);
		}
		
		public TextWrapper(String s, int mlen) {
			super(s, mlen);
		}
		
		public String getS() {
			String eintrag;
			try {
				eintrag = getText(0, getLength());
			} catch (BadLocationException ex) {
				throw new RuntimeException();
			}
			
			if(eintrag.length() > 0) {
				char sig = eintrag.charAt(eintrag.length() - 1);
				if(sig == '.' || sig == '!' || sig == '?') {
					return eintrag;
				}
			}
			return eintrag + '.';
		}	
	}
	

	public static class IntWrapper extends StringWrapper {

		private static final long serialVersionUID = 1L;

		public IntWrapper(int i) {
			super(Integer.toString(i));
		}
		
		public IntWrapper(int i, int m) {
			super(Integer.toString(i), m);
		}

		public IntWrapper(String s) {
			super(s);
			// int i;
			try {
				Integer.parseInt(s);
			} catch (NumberFormatException e) {
				throw new IllegalArgumentException();
			}
		}

		public void setS(String s) {
			// int i;
			try {
				Integer.parseInt(s);
				super.setS(s);
			} catch (NumberFormatException e) {
				return;
			}
		}

		public int getInt() {
			return Integer.parseInt(getS());
		}

		public void setInt(int i) {
			setS(Integer.toString(i));
		}

		public void insertString(int offs, String str, AttributeSet a)
				throws BadLocationException {
			try {
				Integer.parseInt(str);
			} catch (NumberFormatException e) {
				return;
			}
			super.insertString(offs, str, a);
		}
	}
	
	
	

	
	public static class ListWrapper<T> 
	extends LinkedList<T>
	implements ListModel{

		private static final long serialVersionUID = 1L;
		protected LinkedList<ListDataListener> listeners;
		
		
		public ListWrapper() {
			super();
			listeners = new LinkedList<ListDataListener>();
		}
		
		public void addListDataListener(ListDataListener l) {
			listeners.add(l);
		}

		public Object getElementAt(int index) {
			return get(index);
		}

		public void removeListDataListener(ListDataListener l) {
			listeners.remove(l);
		}

		public int getSize() {
			return size();
		}
		
		public LinkedList<ListDataListener> getListeners(){
			return listeners;
		}	
	}


	public static class Link{
		
		protected int typ;
		protected String link;
		protected LinkedList<TableModelListener> listeners;
		
		public Link(int typ, String link) {
			setLink(link);
			setTyp(typ);
			listeners = new LinkedList<TableModelListener>();
		}

		public Link(String link) {
			if(link == null) {
				throw new IllegalArgumentException();
			}
			setLink(link);
			setTyp(0);
		}

		
		public String getLink() {
			return link;
		}

		public void setLink(String link) {
			if(link == null) {
				throw new IllegalArgumentException();
			}
			this.link = link;
		}

		public int getTyp() {
			return typ;
		}

		public void setTyp(int typ) {
			if(typ < 0 || typ > Bogen.LINK_TYPES.length - 1) {
				throw new IllegalArgumentException();
			}
			this.typ = typ;
		}
		
		public void setTyp(String str) {
			if(str == null) {
				throw new IllegalArgumentException();
			}
			for(int i = 0; i < Bogen.LINK_TYPES.length; i++) {
				if(Bogen.LINK_TYPES[i].equals(str)){
					setTyp(i);
					return;
				}
			}
			throw new IllegalArgumentException();
		}
		
		public String toString() {
			return link;
		}		
	}
	
	
	public static class LinkList 
		extends ListWrapper<Link>
		implements TableModel {

		private static final long serialVersionUID = 1L;
		
		LinkedList<TableModelListener> listeners;
			
		public LinkList() {
			listeners = new LinkedList<TableModelListener>();
		}
		
		public void addTableModelListener(TableModelListener l) {
			listeners.add(l);			
		}

		public Class<?> getColumnClass(int columnIndex) {
			return Link.class;
		}

		public int getColumnCount() {
			return 2;
		}

		public String getColumnName(int columnIndex) {
			switch(columnIndex){
				case 0: return "Typ"; 
				case 1: return "Link";
				default: throw new IllegalArgumentException();
			}
		}

		public int getRowCount() {
			return size();
		}

		public Object getValueAt(int rowIndex, int columnIndex) {
			if(rowIndex >= size() || columnIndex > 1) {
				throw new IllegalArgumentException();
			}
			Link link = get(rowIndex);
			return (columnIndex == 0) ? Bogen.LINK_TYPES[link.getTyp()] : link.getLink();
		}

		public boolean isCellEditable(int rowIndex, int columnIndex) {
			return true;
		}

		public void removeTableModelListener(TableModelListener l) {
			listeners.remove(l);
		}

		public void setValueAt(Object aValue, int rowIndex, int columnIndex) {
			switch(columnIndex) {
			case 0: 
				try{
					get(rowIndex).setTyp(aValue.toString());
				}catch(NumberFormatException e){
					throw new IllegalArgumentException();
				}
				break;
			case 1:
				String str;
				if(aValue == null) {
					str = "";
				} else {
					str = aValue.toString();
				}
				get(rowIndex).setLink(str);
				break;
			default:
				throw new IllegalArgumentException();
			}
			if(columnIndex > 1) {
				throw new IllegalArgumentException();
			}
			
		}		

		public boolean add(Link l) { 
			contentChanged();
			return super.add(l);
		}
		
		public void remove(Link l) {
			super.remove();
			contentChanged();
		}
		
		public void contentChanged() {
			Iterator<TableModelListener> it = listeners.iterator();
			TableModelEvent ev = new TableModelEvent(this);
			while (it.hasNext()) {
				it.next().tableChanged(ev);
			}
		}
		
		public LinkedList<TableModelListener> getTableListeners(){
			return listeners;
		}
	}
}

