/************************************************************************\
 * C2py converts cpy code (with a C-like syntax) to Python code.        *
 * Copyright (C) 2019  Asher Gordon <AsDaGo@protonmail.ch>              *
 *                                                                      *
 * This file is part of c2py.                                           *
 *                                                                      *
 * C2py is free software: you can redistribute it and/or modify         *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation, either version 3 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * C2py is distributed in the hope that it will be useful,              *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with c2py.  If not, see <https://www.gnu.org/licenses/>.       *
\************************************************************************/

/* input.c -- getc() and ungetc() frontend to get characters from
   `command_string' or interface with GNU Readline. */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#include <string.h>
#include <stdlib.h>

#include "readline.h"
#include "c2py.h"
#include "input.h"

/* Default prompts

   FIXME: When Python is run in interactive mode, sys.ps1 and sys.ps2
   are set. When c2py is run in interactive mode they are not. This is
   why we need the default prompts. Figure out how Python sets sys.ps*
   when in interactive mode. */
#define PS1	">>> "
#define PS2	"... "
#define PROMPT	(wait_string ? PS2 : PS1)

static unsigned int input_string_index;
static char *input_string = NULL;
static unsigned int command_string_index = 0;

int get_input(void) {
  if (command_string) {
    /* Make sure we're not at the end of `command_string'. */
    if (!command_string[command_string_index])
      return EOF;

    /* Get a charcter from `command_string'. */
    return command_string[command_string_index++];
  }

  if (interactive) {
    /* Get a character from `input_string' or get a new line if that's
       empty. */
    do {
#ifndef HAVE_LIBREADLINE
      size_t len = 0;
#endif
      char *prompt = PROMPT;
      
      if (input_string && input_string[input_string_index])
	return input_string[input_string_index++];

      if (input_string) {
	char return_newline =
	  !*input_string || input_string[input_string_index - 1] != '\n';
	
	free(input_string);
	input_string = NULL;

	if (return_newline) {
	  input_string_index = 0;
	  return '\n';
	}
      }

      input_string_index = 0;

#ifdef HAVE_LIBPYTHON
      /* Get the prompt if we're in shebang mode */
      if (shebang) {
	PyObject *pysys, *pyprompt;

	if ((pysys = PyImport_AddModule("sys")) &&
	    (pyprompt =
	     PyObject_GetAttrString(pysys, wait_string ? "ps2" : "ps1"))) {
	  PyObject *pyprompt_str;

	  /* Handle non-str objects */
	  if ((pyprompt_str = PyObject_Str(pyprompt))) {
	    PyObject *pybytes;

	    /* We're done with this */
	    Py_DECREF(pyprompt);

	    /* Convert the string object to a bytes object */
	    if ((pybytes =
		 PyUnicode_AsEncodedString(pyprompt_str, NULL, NULL))) {
	      /* Convert the bytes object to a char * */
	      if (!(prompt = PyBytes_AsString(pybytes))) {
		PyErr_Print();
		prompt = PROMPT;
	      }

	      Py_DECREF(pybytes);
	    }
	    else {
	      PyErr_Print();
	    }
	  }
	  else {
	    Py_DECREF(pyprompt);
	    PyErr_Print();
	  }
	}
	else {
	  PyErr_Clear();
	}
      }
#endif

#ifdef HAVE_LIBREADLINE
      if (editing) {
	input_string = readline(prompt);
      }
      else {
	size_t len = 0;
#endif
	fputs(prompt, stderr);

	/* Indent the prompt */
	for (unsigned int i = 0; i < indent_count * indent_size; i++)
	  putc(' ', stderr);

	if (getline(&input_string, &len, *input_files) == -1) {
	  free(input_string);
	  input_string = NULL;
	}
#ifdef HAVE_LIBREADLINE
      }
#endif
      
      if (!input_string)
	return EOF;

#ifdef HAVE_READLINE_HISTORY
      /* Add the line to history if it's not empty. */
      if (
# ifdef HAVE_LIBREADLINE
	  editing ? *input_string :
# endif
	  *input_string != '\n')
	add_history(input_string);
      /* Otherwise, if it is empty, end the conditional (similar to
	 Python's behavior). I know, it brings in a little whitespace
	 dependency in interactive mode, but it is needed because
	 otherwise you couldn't type things like `if (1);
	 print("foo");' (Python doesn't like things immediately after
	 an if or similar in Py_single_input (which is used for
	 interactive input) mode.) */
# ifdef HAVE_LIBPYTHON
      else
# endif
#else
# ifdef HAVE_LIBPYTHON
      if (
#  ifdef HAVE_LIBREADLINE
	  editing ? !*input_string :
#  endif
	  *input_string == '\n')
# endif /* HAVE_LIBPYTHON */
#endif /* HAVE_READLINE_HISTORY */
#ifdef HAVE_LIBPYTHON
	end_conditional();
#endif
    } while (!*input_string);

    return input_string[input_string_index++];
  }

  /* Read a character from `*input_files'. */
  return getc(*input_files);
}

int unget_input(int c) {
  if (command_string) {
    /* Unget a character to `command_string'. */
    if (!command_string_index) {
      char *new_command_string;
      size_t size =
	sizeof(*new_command_string) * (strlen(command_string) + 2);

      new_command_string = malloc(size);
      snprintf(new_command_string, size, "%c%s",
	       (unsigned char)c, command_string);

      if (command_string_free)
	free(command_string);
      
      command_string = new_command_string;
      command_string_free = 1;
    }
    else {
      command_string[--command_string_index] = (unsigned char)c;
    }

    return (unsigned char)c;
  }

  if (interactive) {
    if (!input_string_index) {
      /* Unget a character to `input_string'. */
      char *new_input_string;
      size_t size = sizeof(*new_input_string) *
	((input_string ? strlen(input_string) : 0) + 2);

      new_input_string = malloc(size);
      snprintf(new_input_string, size, "%c%s",
	       (unsigned char)c, input_string ? input_string : "");

      free(input_string);
      input_string = new_input_string;
    }
    else {
      input_string[--input_string_index] = (unsigned char)c;
    }

    return (unsigned char)c;
  }

  /* Unget a character from `*input_files'. */
  return ungetc(c, *input_files);
}

#ifdef HAVE_LIBREADLINE
/* The readline completer. */
char * c2py_complete(const char *text, int state) {
  const char *const completion_tokens[] = {"import", "from", "as",
					   "for", "while", "in",
					   "if", "elif", "else",
					   "switch", "case", "default",
					   "try", "except", "finally",
					   "class",
					   "return"};
  static const char *matches
    [sizeof(completion_tokens) / sizeof(*completion_tokens)];
  static int index;
  static size_t size;

  if (!state) {
    index = size = 0;
    
    for (int i = 0;
	 i < sizeof(completion_tokens) / sizeof(*completion_tokens); i++)
      if (!strncmp(text, completion_tokens[i], strlen(text)))
	matches[size++] = completion_tokens[i];
  }

  if (index >= size)
    return NULL;

  /* We have to strdup() the string because readline expects a
     malloc()'d string. */
  return strdup(matches[index++]);
}

/* A hook to indent the text. */
int indent_text(void) {
  char *indent_string;
  size_t indent_string_size;

  indent_string_size = sizeof(*indent_string) *
    (((indent_count
       /* The way switch statements are implemented, each `case' is a
	  Python if statement, so the indent count is wrong for the
	  prompt. This line is to fix that. */
       - switch_count
       ) * indent_size) + 1);
  indent_string = malloc(indent_string_size);
  memset(indent_string, ' ', indent_string_size - 1);
  indent_string[indent_string_size - 1] = '\0';

  /* Insert the text. */
  rl_insert_text(indent_string);

  free(indent_string);

  return 0;
}
#endif /* HAVE_LIBREADLINE */
