//  Copyright (C) 2011, 2014 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <config.h>

#include <glibmm.h>
#include "main.h"
#include "cbc-osd-connector.h"
#include <libnotify/notify.h>
#include "timing.h"
#include <curlpp/Options.hpp>
#include <curlpp/Infos.hpp>
#include "ucompose.hpp"

CBCOSDConnector* CBCOSDConnector::create(bool news, bool sports)
{
  return new CBCOSDConnector(news, sports);
}

CBCOSDConnector::CBCOSDConnector(bool news, bool sports)
{
  d_watch_news = news;
  d_watch_sports = sports;
}

void open_browser(NotifyNotification *notification, char *action, gpointer user_data)
{
  Glib::ustring url =std::string((gchar*)user_data);
  Glib::ustring cmd = Glib::find_program_in_path("gvfs-open") + " '" + url +"'";
  Glib::spawn_command_line_async(cmd);
}

void CBCOSDConnector::notify(Glib::ustring show_name, Glib::ustring show_description, Glib::ustring show_id, Glib::ustring url)
{
  Glib::ustring text = show_name +"\n" + show_description;
  NotifyNotification *notification = notify_notification_new("Streaming Live",Glib::locale_from_utf8(text).c_str(), NULL);
  Glib::RefPtr<Gdk::Pixbuf> pixbuf = Gdk::Pixbuf::create_from_file
    (Main::get_data_path() + "/data/cbc-osd.png");
  notify_notification_set_icon_from_pixbuf  (notification, pixbuf->gobj());
  gpointer urlp = (gpointer) g_strdup(Glib::locale_from_utf8(url).c_str());
  notify_notification_add_action(notification, Glib::locale_from_utf8(show_id).c_str(), "Start Streaming", open_browser, urlp, g_free);
  notify_notification_set_timeout(notification, 20 * 1000);
  notify_notification_show(notification,NULL);
}

CBCOSDConnector::~CBCOSDConnector()
{
}

bool CBCOSDConnector::get(Glib::ustring url, Slot slot)
{
  std::string tmp;
  int fd = Glib::file_open_tmp(tmp);
  close (fd);
  FILE *fileptr = fopen (tmp.c_str(), "wb");
  curlpp::Easy *easy = new curlpp::Easy();
  easy->setOpt(curlpp::options::HttpGet(true));
  easy->setOpt(curlpp::options::Url(url));
  easy->setOpt(curlpp::options::WriteFile(fileptr));
  try
    {
      easy->perform();
    } catch (cURLpp::LogicError& exception) {
      remove (tmp.c_str());
      return false;
    } catch (cURLpp::RuntimeError& exception) {
      remove (tmp.c_str());
      return false;
    }
  fflush (fileptr);
  fsync (fileno (fileptr));
  fclose (fileptr);
  std::string data = Glib::file_get_contents(tmp);
  int response_code =  curlpp::infos::ResponseCode().get(*easy);
  size_t len = data.length();
  slot(easy, response_code, data.c_str(), len);
  delete easy;
  remove (tmp.c_str());
  return true;
}

void CBCOSDConnector::connect()
{
  if (!notify_is_initted ())
    if (!notify_init (PACKAGE))
      {
        connect_succeeded.emit(false);
        return;
      }

  connect_succeeded.emit(true);

}

bool CBCOSDConnector::start_polling(int msecs)
{
  bool failed = false;
  poll(failed);
  timer = Timing::instance().register_timer
    (sigc::bind(sigc::mem_fun(this, &CBCOSDConnector::poll), failed), msecs);
  return failed;
}

bool CBCOSDConnector::poll_news()
{
  bool success = get (CBC_VIDEO_URL, sigc::mem_fun (*this, &CBCOSDConnector::on_news_polled));
  if (success)
    success = get (CBC_VIDEO_URL "Politics/", sigc::mem_fun (*this, &CBCOSDConnector::on_news_polled));
  if (success)
    success = get (CBC_VIDEO_URL "Canada/", sigc::mem_fun (*this, &CBCOSDConnector::on_news_polled));
  if (success)
    success = get (CBC_VIDEO_URL "Local%20News%20Shows/", sigc::mem_fun (*this, &CBCOSDConnector::on_news_polled));
  return success;
}

bool CBCOSDConnector::poll_sports()
{
  return get (String::ucompose("%1%2", CBC_VIDEO_BASE,"/Sports/Live%20Streaming/"), 
       sigc::mem_fun (*this, &CBCOSDConnector::on_sports_polled));
}

bool CBCOSDConnector::poll(bool &failed)
{
  failed = false;
  if (d_watch_news)
    failed = poll_news();
  if (d_watch_sports && !failed)
    failed = poll_sports();
  return Timing::CONTINUE;
}

void CBCOSDConnector::on_sports_polled(curlpp::Easy *easy, int response, const char *data, size_t len)
{
  if (response == 200)
    {
      if (len != 0)
        sports_polled.emit(data, len);
    }
}

void CBCOSDConnector::on_news_polled(curlpp::Easy *easy, int response, const char *data, size_t len)
{
  if (response == 200)
    news_polled.emit(data, len);
}

void CBCOSDConnector::stop_polling()
{
  timer.disconnect();
}

