#include <iostream>
#include <fstream>
#include "flowfield.h"
#include "vector.h"
#include "chebyshev.h"
#include "tausolver.h"

const Real EPSILON=1e-12;

// An example program that does DNS integration on OrrSommerfeld eigenfunctions
// and compares to the analytic solution for linear OrrSommerfeld growth.
//
// Unlike the other examples, this example performs time-stepping explicitly,
// rather than using the NSIntegrator as a DNS black-box. As such it might 
// be easier to understand for someone familiar with spectral channel flow 
// algorithms, but not familiar with C++. 

int main() {
  fftw_loadwisdom(); 

  const int Nx=8;
  const int Ny=65;
  const int Nz=8;
  const int Nd=3;
  const Real Lx=2*pi;
  const Real Lz=2*pi;
  const Real Ly=2.0;
  const Real a=-1.0;
  const Real b=1.0;
  const Real R = 7500.0;
  const Real nu = 1.0/R;
  const Real dPdx = -2.0*nu;    // mean streamwise pressure gradient.
  const Real dt = 0.1;
  const Real T = 50.0;
  Real scale =  0.01;
  const int plotmod = 1;
  const bool pause = true;
  const int nSteps = int(T/dt);
  const int Ny2 = Ny/2 + 1;

  ChebyTransform trans(Ny);

  cout << "Nx Ny Nz Nd == " << Nx << ' ' << Ny << ' ' << Nz << ' ' << Nd<<endl;
  cout << "Lx Ly Lz == " << Lx << ' ' << Ly << ' '  << Lz << ' ' << endl;

  Vector y = chebypoints(Ny,a,b);
  y.save("y");
  
  ChebyCoeff zero(Ny,a,b,Physical);
  ChebyCoeff parabola(Ny,a,b,Physical);
  for (int ny=0; ny<Ny; ++ny) 
    parabola[ny] = 1.0 - square(y[ny]);
  parabola.makeSpectral(trans);

  ChebyCoeff Ubase = parabola;
  ChebyCoeff Ubasey = diff(Ubase);
  ChebyCoeff Ubaseyy = diff(Ubasey);
  ChebyCoeff UbaseyT = Ubase;
  Ubase.makePhysical(trans);
  Ubasey.makePhysical(trans);
  
  FlowField un(Nx,Ny,Nz,Nd,Lx,Lz,a,b); // u numerical, i.e. DNS velocity field
  FlowField ut(Nx,Ny,Nz,Nd,Lx,Lz,a,b); // u true, i.e. analytic linear OS field
  FlowField Pn(Nx,Ny,Nz,1,Lx,Lz,a,b);  // Pressure numerical
  FlowField Pt(Nx,Ny,Nz,1,Lx,Lz,a,b);  // Pressure true, 
  FlowField vortn(Nx,Ny,Nz,Nd,Lx,Lz,a,b); // etc.
  FlowField fn1(Nx,Ny,Nz,Nd,Lx,Lz,a,b);
  FlowField fn(Nx,Ny,Nz,Nd,Lx,Lz,a,b);
  FlowField divn(Nx,Ny,Nz,Nd,Lx,Lz,a,b);

  // Calculate L2Norm of poisseuille flow.
  un.addProfile(ComplexChebyCoeff(parabola, zero), 0, 0, 0, false);
  Real uPoissNorm = L2Norm(un);
  un.setToZero();

  BasisFunc u_oseig("oseig10_65");
  ComplexChebyCoeff u_peig("ospeig10_65");
  Complex omega = 0.24989153647208251 + I*0.0022349757548207664;
  Complex lambda = -1.0*I*omega;

  u_peig.makeSpectral(trans);
  u_oseig.makeSpectral(trans);
  u_peig  *=  1/L2Norm(u_oseig);
  u_oseig *=  1/L2Norm(u_oseig);

  BasisFunc s_oseig(u_oseig);  // "scaled OS eigenfunc", scale = 1%, say.
  ComplexChebyCoeff s_peig(u_peig);

  un.addProfile(s_oseig, true);
  scale *= uPoissNorm/L2Norm(un);
  un.setToZero();
  s_oseig *= scale;
  s_peig *= scale;
  
  Real cfl;

  // Set data for t = -dt, so as to calculate nonlinear term at t = -dt.
  s_oseig *= exp(lambda*(-dt));
  un.addProfile(s_oseig, true);
  un.nonlinearityRotational(Ubase, Ubasey, UbaseyT, vortn, fn);

  // Reset data for t = 0.
  un.setToZero();
  s_oseig *= exp(lambda*dt);
  un.addProfile(s_oseig, true);
  ut.addProfile(s_oseig, true);
  Pn.addProfile(s_peig,1,0,0, true);
  Pt.addProfile(s_peig,1,0,0, true);

  const Real four_pisq    = 4*square(pi);
  const Real four_pisq_nu = four_pisq*nu;

  // Construct a TauSolver for each (rkstep,kx,kz) triplet. Each TauSolver 
  // construction allocates space for pressure and velocity tridiag equations,
  // and does LU decomp on the tridiags. Thesis notes 4/18/2000.
  cout << "Constructing TauSolvers..." << flush;
  TauSolver** tausolver = new TauSolver*[un.numXmodes()];
  for (int nx=0; nx<un.numXmodes(); ++nx) {
    int kx = un.kx(nx);
    tausolver[nx] = new TauSolver[un.numZmodes()];
    for (int nz=0; nz<un.numZmodes(); ++nz) {
      int kz = un.kz(nz);
      Real lambda = 2.0/dt + four_pisq_nu*(square(kx/Lx) + square(kz/Lz));
      tausolver[nx][nz] = TauSolver(kx, kz, Lx, Lz, a, b, lambda, nu, Ny);
    }
  }
  cout << " done." << endl;

  ComplexChebyCoeff un_k(Ny,a,b,Spectral);
  ComplexChebyCoeff vn_k(Ny,a,b,Spectral);
  ComplexChebyCoeff wn_k(Ny,a,b,Spectral);
  ComplexChebyCoeff Rx_k(Ny,a,b,Spectral);
  ComplexChebyCoeff Ry_k(Ny,a,b,Spectral);
  ComplexChebyCoeff Rz_k(Ny,a,b,Spectral);
  ComplexChebyCoeff fx_k(Ny,a,b,Spectral);  // stores x comp of 3fn - fn1
  ComplexChebyCoeff fy_k(Ny,a,b,Spectral);  // etc
  ComplexChebyCoeff fz_k(Ny,a,b,Spectral);  
  ComplexChebyCoeff Pn_k(Ny,a,b,Spectral);
  ComplexChebyCoeff Pny_k(Ny,a,b,Spectral);

  cout << "======================================================" << endl;
  cout << "Starting time stepping..." << endl;
  ofstream ns("osnorm.asc");
  
  bool debug=true;
  
  Complex L2IP0 = L2InnerProduct(un, u_oseig);
  Real u0norm = L2Norm(un);

  for (int step=0; step<nSteps; ++step) {
    Real t = dt*step;
    cout << "step " << step << "  t == " << t << endl;

    if (step % plotmod == 0) {
      if (debug) {
	Real unnorm = L2Norm(un);
	cout << "================================================" << endl;
	cout << "       div(un)/L2Norm(un)          == " << un.divergence()/L2Norm(un) << endl;
	cout << "L2Dist(un, ut)/L2Norm(ut)          == " << L2Dist(un, ut)/L2Norm(ut) << endl;
	cout << "L2Dist(Pn, Pt)/L2Norm(Pt)          == " << L2Dist(Pn,Pt)/L2Norm(Pt) << endl;
	
	Complex an = L2InnerProduct(un, u_oseig); // project DNS field onto OS eig
	Complex at = L2InnerProduct(ut, u_oseig); // project linear field into OS
	cout << "at  == " << at << endl;
	cout << "an  == " << an << endl;
	cout << "phase at  == " << phase(at) << endl;
	cout << "phase an  == " << phase(an) << endl;
	cout << "(phase(an)-phase(at))/phase(at) == " << (phase(an)-phase(at))/phase(at) << endl;
	cout << "norm at  == " << norm(at) << endl;
	cout << "norm an  == " << norm(an) << endl;
	cout << "(norm(an)-norm(at))/norm(at) == " << (norm(an)-norm(at))/norm(at) << endl;
	Complex L2IP = L2InnerProduct(un, u_oseig);
	cout << "L2IP(u,os)/L2(u0,os) == " << L2IP/L2IP0  << endl;
	cout << "exp(lambda*t)        == " << exp(lambda*t) << endl;
	cout << "lambda      == " << lambda <<endl;
	cout << "lambda appr == " << log(L2IP/L2IP0)/t <<endl;
	cout << "growth rate ==  "<< log(unnorm/u0norm)/t << endl;

	ns << norm(an) << endl;
	ns.flush();

	un.saveSpectrum("unspec");
	ut.saveSpectrum("utspec");
	fn.saveSpectrum("fnspec");
	un.saveSlice(Ny2, 0, "unslice");
	ut.saveSlice(Ny2, 0, "utslice");
	fn.saveSlice(Ny2, 0, "fnslice");
      
	un.saveProfile(0,0,"un00", trans);
	un.saveProfile(1,0,"un10", trans);
	ut.saveProfile(0,0, "ut00", trans);
	ut.saveProfile(1,0, "ut10", trans);
	fn.saveProfile(0,0, "fn00", trans);
	fn.saveProfile(1,0, "fn10", trans);

	if (pause) {
	  cout << "hit return to continue... ";
	  char buff[10];
	  cin.getline(buff, 10);
	}
      }
    }

    // Take a time step. Current time is t=n*dt. Calculate data at t=(n+1)*dt.
    // Store it in un. I.e. map 
    // (u(n), f(n), f(n-1)) ->  (u(n+1), f(n+1), fn). 
    // ( un,   fn,   fn1)   
    // shift nonlinear term in time and calculate new nonlinear term for
    // current value of un. 
    // Update nonlinear terms
    swap(fn1,fn);  
    un.nonlinearityRotational(Ubase, Ubasey, UbaseyT, vortn, fn); 

    // Update with the time-stepping algortihm
    // Each (kx,kz) pair is an independent calculation, so parallelism
    // can be implemented at this level in the code.
    for (int nx=0; nx<un.numXmodes(); ++nx) {
      int kx = un.kx(nx);
      Complex pi2IkxLx = 2.0*kx*pi*I/Lx;
      for (int nz=0; nz<un.numZmodes(); ++nz) {
	int kz = un.kz(nz);
	Complex pi2IkzLz = 2.0*kz*pi*I/Lz;

	for (int ny=0; ny<un.numYmodes(); ++ny) {
	  un_k.set(ny, nu*un.cmplx(nx,ny,nz,0));
	  vn_k.set(ny, nu*un.cmplx(nx,ny,nz,1));
	  wn_k.set(ny, nu*un.cmplx(nx,ny,nz,2));
	  Pn_k.set(ny, Pn.cmplx(nx,ny,nz,0));
	  fx_k.set(ny, -3.0*(fn.cmplx(nx,ny,nz,0)) 
   		         +(fn1.cmplx(nx,ny,nz,0)));
	  fy_k.set(ny, -3.0*(fn.cmplx(nx,ny,nz,1)) 
		         +(fn1.cmplx(nx,ny,nz,1)));
	  fz_k.set(ny, -3.0*(fn.cmplx(nx,ny,nz,2)) 
		         +(fn1.cmplx(nx,ny,nz,2)));
	}
	  
	// Put abnu un" into in Rx, etc. 
	diff2(un_k, Rx_k);
	diff2(vn_k, Ry_k);
	diff2(wn_k, Rz_k);

	// Compute y-comp of pressure gradient
	diff(Pn_k, Pny_k); 
      
	// Add remainder of RHS terms to R. After this step,
	// R = ab nu un" + [1/(b dt)-4pi^2kappa2] un - ab grad P + gb fn + zb fn1
	Real kappa2 = 4*pi*pi*square(kx/Lx) + square(kz/Lz);
	Real c  = 2.0/(dt*nu) - kappa2;
	for (int ny=0; ny<un.numYmodes(); ++ny) {
	  Rx_k.add(ny, c*un_k[ny] + fx_k[ny] - pi2IkxLx*Pn_k[ny]);
	  Ry_k.add(ny, c*vn_k[ny] + fy_k[ny] - Pny_k[ny]);
	  Rz_k.add(ny, c*wn_k[ny] + fz_k[ny] - pi2IkzLz*Pn_k[ny]);
	}

	// The x comp of R has other terms from mean pressure gradient
	// and curvature of base flow.
	if (kx==0 && kz == 0) {
	  Rx_k.add(0, Complex(-2*dPdx));
	  for (int ny=0; ny<Ny; ++ny)
	    Rx_k.re[ny] += nu*Ubaseyy[ny];
	}

	// Solve the tau eqns (influence-matrix solve on (P,v), adjust with
	// tau correction, then calculate (u,w).
	tausolver[nx][nz].solve(un_k, vn_k, wn_k, Pn_k, Rx_k, Ry_k, Rz_k);

	// Load solutions back into the full 3d data arrays.
	for (int ny=0; ny<un.numYmodes(); ++ny) {
	  un.cmplx(nx,ny,nz,0) = un_k[ny];
	  un.cmplx(nx,ny,nz,1) = vn_k[ny];
	  un.cmplx(nx,ny,nz,2) = wn_k[ny];
	  Pn.cmplx(nx,ny,nz,0) = Pn_k[ny];
	}
      }
    }
    un.divergence(divn);

    // And advance the analytic solution analytically
    ut.setToZero();
    Pt.setToZero();

    BasisFunc e_oseig(s_oseig);
    e_oseig *= exp(lambda*(t+dt*plotmod));
    ut.addProfile(e_oseig, true);
      
    ComplexChebyCoeff e_peig(s_peig);
    e_peig *= exp(lambda*(t+dt*plotmod));
    Pt.addProfile(e_peig, 1, 0, 0, true);

  }
  fftw_savewisdom();
}
