/* orrsommfunc.cpp: calculate Orr-Sommerfeld eigenfunctions of Poisseuille flow
 * Channelflow-0.9
 *
 * Copyright (C) 2001  John F. Gibson  
 *  
 * jgibson@mail.sjcsf.edu  
 * John F. Gibson 
 * St. John's College
 * 1160 Camino de la Cruz Blanca
 * Santa Fe, NM 87501
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, U
 */

#include <fstream>
#include <iomanip>
//#include <octave/oct.h>
//#include <octave/CMatrix.h>
//#include <octave/EIG.h>
//#include <octave/CmplxSVD.h>
#include "orrsommfunc.h"
#include "quicksort.h"
#include "helmholtz.h"

typedef ComplexColumnVector ComplexVector;
OrrSommFunc::OrrSommFunc(const string& filebase) 
  :
  BasisFunc(0,0,0,1,1,-1,1,Spectral),
  reynolds_(0.0),
  omega_(0.0, 0.0),
  pressure_(0, -1, 1, Spectral)
{
  string filename = filebase + string(".osf");
  ifstream is(filename.c_str());
  if (!is.good()) {
    cerr << "OrrSommFunc::OrrSommFunc(filebase) : can't open file " << filename << '\n';
    abort();
  }
  char c;
  is >> c;
  if (c != '%') {
    cerr << "OrrSommFunc::OrrSommFunc(filebase): bad header in file " << filename << endl;
    assert(false);
  }
  int newNy;
  is >> newNy
     >> kx_
     >> kz_
     >> Lx_
     >> Lz_
     >> a_
     >> b_
     >> state_ 
     >> lambda_
     >> reynolds_
     >> omega_;
  u_.setBounds(a_,b_);
  v_.setBounds(a_,b_);
  w_.setBounds(a_,b_);
  pressure_.setBounds(a_,b_);
  u_.setState(state_);
  v_.setState(state_);
  w_.setState(state_);
  pressure_.setState(state_);
  BasisFunc::resize(newNy);
  pressure_.resize(newNy);
  
  Real r;
  Real i;
  for (int n=0; n<Ny_; ++n) {
    is >> r >> i; u_.set(n,r+I*i);
    is >> r >> i; v_.set(n,r+I*i);
    is >> r >> i; w_.set(n,r+I*i);
    is >> r >> i; pressure_.set(n,r+I*i);
  }
}
   

// Need octave libs for this 
/************************************************************************
OrrSommFunc::OrrSommFunc(Real Reynolds, int Ny, int kx, int kz, 
			 Real Lx, Real Lz, Real a, Real b, bool manual)
  :
  BasisFunc(Ny, kx, kz, Lx, Lz, a, b, Spectral),
  reynolds_(Reynolds),
  omega_(0.0, 0.0),
  pressure_(Ny, a, b, Spectral)
{
  solve_for_v_omega(manual);
  solve_for_p();
  solve_for_u(); // from momentum
  solve_for_w(); // from momentum

  //w_.setToZero();
  
  //if (kz !=0) {
  //ComplexChebyCoeff vy = diff(v_);
  //ComplexChebyCoeff ux = u_;
  //ux *= (2*pi*kx_/Lx_)*I;
  //w_ = ux;
  //w_ += vy;
  //w_ *= -Lz_/(2*pi*I*kz_);
  //}
  
  (*this) *= 1.0/L2Norm(*this); // set to unit normalization
}
***********************************************************************/

OrrSommFunc& OrrSommFunc::operator=(const OrrSommFunc& os) {
  u_ = os.u_;
  v_ = os.v_;
  w_ = os.w_;
  pressure_ = os.pressure_;
  reynolds_ = reynolds_;
  omega_ = omega_;
  return (*this);
}

bool OrrSommFunc::operator==(const OrrSommFunc& os) const {
  bool equal;
  equal = ((const BasisFunc&) *this == (const BasisFunc&) os);
  equal |= (os.pressure_ == pressure_);
  equal |= (os.reynolds_ == reynolds_);
  equal |= (os.omega_ == omega_);
  return equal;
}

bool OrrSommFunc::operator!=(const OrrSommFunc& os) const {
  return !(*this == os);
}
 
Real OrrSommFunc::reynolds() const {return reynolds_;}
Complex OrrSommFunc::eigenvalue() const {return omega_;}
Complex OrrSommFunc::omega() const {return omega_;}
Complex OrrSommFunc::lambda() const {return -(1*omega_)*I;}
const ComplexChebyCoeff& OrrSommFunc::pressure() const {return pressure_;}
const ComplexChebyCoeff& OrrSommFunc::p() const {return pressure_;}

void OrrSommFunc::chebyfft(const ChebyTransform& t) {
  BasisFunc::chebyfft(t);
  pressure_.chebyfft(t);
}

void OrrSommFunc::ichebyfft(const ChebyTransform& t) {
  BasisFunc::ichebyfft(t);
  pressure_.ichebyfft(t);
}
void OrrSommFunc::makeSpectral(const ChebyTransform& t)  {
  BasisFunc::makeSpectral(t);
  pressure_.makeSpectral(t);
}
void OrrSommFunc::makePhysical(const ChebyTransform& t) {
  BasisFunc::makePhysical(t);
  pressure_.makePhysical(t);
}
//ComplexChebyCoeff OrrSommFunc::totalPressure() const;

OrrSommFunc& OrrSommFunc::operator *= (Real c) {
  BasisFunc::operator*=(c);
  pressure_ *= c;
  return *this;
}

OrrSommFunc& OrrSommFunc::operator *= (Complex c) {
  BasisFunc::operator*=(c);
  pressure_ *= c;
  return *this;
}

void OrrSommFunc::bfsave(const string& filebase) {
  BasisFunc::save(filebase);
}

void OrrSommFunc::save(const string& filebase) {
  string filename = filebase + string(".osf");
  ofstream os(filename.c_str());
  os << setprecision(REAL_DIGITS);
  char s=' ';
  //if (MATLABVERSION > 4)
  os << '%'
     <<s<< Ny_
     <<s<< kx_
     <<s<< kz_
     <<s<< Lx_
     <<s<< Lz_
     <<s<< a_
     <<s<< b_
     <<s<< state_ 
     <<s<< lambda_
     <<s<< reynolds_
     <<s<< omega_
     << '\n';

  for (int i=0; i<Ny_; ++i) 
    os << Re(u_[i]) <<s<< Im(u_[i]) <<s
       << Re(v_[i]) <<s<< Im(v_[i]) <<s
       << Re(w_[i]) <<s<< Im(w_[i]) <<s
       << Re(pressure_[i]) <<s<< Im(pressure_[i]) << '\n';
  os.close();
}

    
void OrrSommFunc::solve_for_v_omega(bool manual) {
  
  const Real al = 2*pi*kx_/Lx_;
  const Real be = 2*pi*kz_/Lz_;
  const Real ga2 = square(al) + square(be);
  const Real ga4 = square(ga2);
  
  const Complex iaR = (al*reynolds_)*I;
  //const Complex ia3R = I*cube(al)*reynolds_;
  const int N = Ny_; // typographical convenience
  ComplexMatrix A(N,N);
  ComplexMatrix B(N,N);
 
  // Fill in ODE elems of matrices, leaving four rows for BCs.
  for (int n=0; n<N-4; ++n) {
    Real cn = chebc(n);
    Real cn1 = chebc(n-1);
    Real cn2 = chebc(n-2);
    Real dn2 = chebd(n-2);

    if (n-2 >= 0)
      A(n,n-2) = -0.25*iaR*ga2*cn2;
    
    A(n,n) = iaR*n*(n-1) + (ga4 + iaR*(ga2-2))*cn - 0.25*iaR*ga2*cn*(cn+cn1);
    B(n,n) = iaR*ga2*cn;
     
    if (n+2 < N)
      A(n,n+2) = -0.25*iaR*ga2*cn;
    
    for (int p=n+2; p<N; p+=2) {
      /**********
      Real nm2 = (Real) n-2;
      Real np2 = (Real) n+2;
      A(n,p) += 
	- (2*al2 + 0.25*iaR*(4-cn-cn1))*p*(p+n)*(p-n)
	+ 0.25*iaR*p*(cn*(p+np2)*(p-np2) + dn2*(p+nm2)*(p-nm2));
      **************/
      Real p2 = square((Real) p);
      Real n2 = square((Real) n);
      Real np2_2 = square((Real) n+2);
      Real nm2_2 = square((Real) n-2);
      A(n,p) += 
	- (2*ga2 + 0.25*iaR*(4-cn-cn1))*p*(p2-n2)
	+ 0.25*iaR*p*(cn*(p2-np2_2) + dn2*(p2-nm2_2));

      B(n,p) = -iaR*p*(p+n)*(p-n);
    }

    for (int p=n+4; p<N; p+=2) {
      Real q  = (Real) p-n;
      Real q2 = q*q;
      Real q3 = q2*q;
      Real q4 = q2*q2;
      Real n2 = n*n;
      Real n3 = n2*n;
      A(n,p) = A(n,p) + p/24.0*(q4*(q2-8) + n*q3*(6*q2-32) + 
				(12*n2*q2 + 8*n3*q)*(q2-4) + 
				16*q2 + 32*q*n);
    }
  }

  // Enforce the BCs for the tau method.
  Complex z(0.0, 0.0);
  Complex l(1.0, 0.0);
  for (int n=N-4; n<N; ++n)
    for (int p=0; p<N; ++p) {
      A(n,p) = z;
      B(n,p) = z;
    }

  // Even-mode BC eqns. p and n are even, 
  for (int p=0; p<N; p+=2) {
    int n = N-3;
    A(n,p) = l;  // v(-1)           
    B(n,p) = z;  // v(-1) == zero

    n = N-1;
    A(n,p) = (Real) p*p; // v'(-1)
    B(n,p) = z;  // v'(-1) == zero
  }

  // Odd-mode BC eqns. p and n are odd, 
  for (int p=1; p<N; p+=2) {
    int n = N-4;
    A(n,p) = l;   // v(1) 
    B(n,p) = z;   // v(1) == zero
    n = N-2;
    A(n,p) = (Real) p*p;   // v'(1) 
    B(n,p) = z;  // v'(1) == zero
  }

  //msave(A, "A");
  //msave(B, "B");

  // A and B are set. But B is rank-deficient, making for an unpleasant
  // generalized eigenvalue problem A x = lambda B x. Transform this to
  // E y = lambda F y, by eliminating the null space of B, via x = Qs y.
  // Qs is found from the SVD of the portion of A that projects onto 
  // the non-nullspace of B. See thesis notes 5/6/00.
  
  // b == bar, t == tilde, s == star

  // Copy last four rows of A into first four rows of C.
  int r=4;  // number of BC eqns, rank deficiency of B.
  ComplexMatrix C(N,N);
  for (int n=0; n<r; ++n) 
    for (int p=0; p<N; ++p) 
      C(n,p) = A(N-4+n,p);

  //msave(C, "C");
  
  ComplexSVD Csvd(C); // C = U D V^herm
  
  // Extract the rectangular Qs that projects onto non-nullspace of B.
  //ComplexMatrix U = Csvd.left_singular_matrix();
  DiagMatrix D    = Csvd.singular_values();
  ComplexMatrix Q = Csvd.right_singular_matrix();
  ComplexMatrix Qs = Q.extract(0,r, N-1, N-1);
  //msave(C, "C");
  //msave(U, "U");
  //msave(D, "D");
  //msave(Q, "Q");
  
  
  // Now project E = A(0:N-r-1,:) Qs, F = B(0:N-r-1,:) Qs
  ComplexMatrix Ab = A.extract(0,0, N-r-1, N-1);
  ComplexMatrix Bb = B.extract(0,0, N-r-1, N-1);
  ComplexMatrix E = Ab * Qs;
  ComplexMatrix F = Bb * Qs;
  //msave(Ab, "Ab");
  //msave(Bb, "Bb");
  //msave(E, "E");
  //msave(F, "F");
  

  // Hooray! We are now out of the null space of B, with a 
  // generalized eigenvalue problem E y = lambda F y. Transform this
  // (F^-1 E) y = lambda y and solve.
  ComplexMatrix Finv = F.inverse();
  ComplexMatrix FinvE = Finv*E;
  
  EIG eig(FinvE);
  ComplexVector eigvals = eig.eigenvalues();
  
  ComplexMatrix Y = eig.eigenvectors();
  ComplexMatrix X = Qs*Y;

  //cout << "Checking that first two eigenvalues are absurdly large..." << endl;
  //if (Im(eigvals(0)) > 2 && Im(eigvals(1)) > 2) 
  //cout << "Yes, they are. Throwing them out." << endl;
  //  else {
  //cout << "No, they're not! Confusion reigns... exiting...." << endl;
  //exit(1);
  //}

  int k=0;
  if (manual) {
    cout << "Choose a growth rate from " << endl;
    for (int n=0; n<eigvals.length(); ++n)
      cout << "n=" << n << " eigenval == " << Im(eigvals(n)) << endl;
    cout << "Which one do you want? (enter <n>)..." << flush;
    cin >> k;
  }
  else {
    array<Real> imag_part(eigvals.length()-2);
    for (int i=2; i<imag_part.length(); ++i) 
      imag_part[i-2] = Im(eigvals(i));
    array<int> index = quicksort(imag_part);
    reverse(index); // put in decreasing order (largest first)

    k=index[0]+2;
  }

  for (int n=0; n<N; ++n) 
    v_.set(n, X(n,k));
  omega_ = eigvals(k);
}


void OrrSommFunc::solve_for_p() {
  
  Vector y = chebypoints(Ny_, a_, b_);
  ChebyTransform trans(Ny_);
  ComplexChebyCoeff v_yy = diff2(v_);
  v_.makePhysical(trans);
  v_yy.makePhysical(trans);

  ComplexChebyCoeff Ubase(Ny_, a_, b_, Physical);
  for (int n=0; n<Ny_; ++n) 
    Ubase.set(n, Complex(1.0 - square(y[n]), 0.0));

  const Real nu = 1/reynolds_;
  const Real al = 2*pi*kx_/Lx_;
  const Real be = 2*pi*kz_/Lz_;
  const Real al2 = square(al);
  const Real be2 = square(be);
  const Complex ia = I*al;
  ComplexChebyCoeff peigy(Ny_, a_, b_, Physical);
  for (int n=0; n<Ny_; ++n) 
    peigy.set(n, (I*omega_ - nu*(al2+be2) - ia*Ubase[n])*v_[n] + nu*v_yy[n]);
  trans.makeSpectral(peigy);
  trans.makeSpectral(v_);

  integrate(peigy, pressure_);
}  



// Thesis notes 4/26/01 and 5/23/01.
void OrrSommFunc::solve_for_u() {
  const Real al = 2*pi*kx_/Lx_;
  const Real be = 2*pi*kz_/Lz_;
  const Real ga2 = square(al) + square(be);
  const Complex ia = I*al; 
  const Complex iaR = I*al*reynolds_;
  const Complex mu = I*reynolds_*(omega_-al) - ga2;

  Vector y = chebypoints(Ny_, a_, b_);
  ChebyTransform trans(Ny_);

  // Build a matrix A for solving u'' + mu u + i alpha y^2 u = f.
  // with BCs u(+-1)=0. See thesis notes 4/26/01.
  ComplexMatrix A(Ny_,Ny_);

  // Fill in ODE elems of matrices, leaving two rows rows for BCs.
  for (int n=0; n<=Ny_-3; ++n) {

    Real cn = chebc(n);
    Real cn1 = chebc(n-1);
    Real cn2 = chebc(n-2);
    
    if (n-2 >= 0)
      A(n,n-2) = 0.25*iaR*cn2;
   
    A(n,n) = mu  + 0.25*iaR*(cn+cn1);

    if (n+2 < Ny_)
      A(n,n+2) = 0.25*iaR;

    for (int p=n+2; p<Ny_; p+=2) 
      A(n,p) += cn*p*(p+n)*(p-n);
  }

  // Enforce u(+-1)=0 BCs
  Complex one(1.0, 0.0);
  for (int p=0; p<Ny_; p+=2)
    A(Ny_-2,p) = one;
  for (int p=1; p<Ny_; p+=2)
    A(Ny_-1,p) = one;
    
  // Construct RHS vector
  v_.makePhysical(trans);
  pressure_.makePhysical(trans);
  ComplexChebyCoeff f(Ny_, a_, b_, Physical);
  for (int n=0; n<Ny_; ++n)
    f.set(n, reynolds_*(ia*pressure_[n] + v_[n]*(-2*y[n])));
  f.makeSpectral(trans);
  
  // revert v_ and peig
  v_.makeSpectral(trans);
  pressure_.makeSpectral(trans);

  ComplexVector bb(Ny_);
  for (int n=0; n<Ny_-2; ++n)
    bb(n) = f[n];
  bb(Ny_-2) = Complex(0.0, 0.0);
  bb(Ny_-1) = Complex(0.0, 0.0);
  
  ComplexVector  x = A.solve(bb);
  for (int n=0; n<Ny_; ++n)
    u_.set(n, x(n));
}

// Thesis notes 5/23/01.
void OrrSommFunc::solve_for_w() {
  const Real al = 2*pi*kx_/Lx_;
  const Real be = 2*pi*kz_/Lz_;
  const Real ga2 = square(al) + square(be);
  const Complex iaR = I*al*reynolds_;
  const Complex ibR = I*be*reynolds_;
  const Complex mu = (I*reynolds_*(omega_-al) - ga2);

  Vector y = chebypoints(Ny_, a_, b_);
  ChebyTransform trans(Ny_);

  // Build a matrix A for solving u'' + mu u = f.
  // with BCs u(+-1)=0. See thesis notes 4/26/01.
  ComplexMatrix A(Ny_,Ny_);

  // Fill in ODE elems of matrices, leaving two rows rows for BCs.
  for (int n=0; n<=Ny_-3; ++n) {

    Real cn = chebc(n);
    Real cn1 = chebc(n-1);
    Real cn2 = chebc(n-2);
    
    if (n-2 >= 0)
      A(n,n-2) = 0.25*iaR*cn2;
   
    A(n,n) = mu  + 0.25*iaR*(cn+cn1);

    if (n+2 < Ny_)
      A(n,n+2) = 0.25*iaR;

    for (int p=n+2; p<Ny_; p+=2) 
      A(n,p) += cn*p*(p+n)*(p-n);
  }

  // Enforce u(+-1)=0 BCs
  Complex one(1.0, 0.0);
  for (int p=0; p<Ny_; p+=2)
    A(Ny_-2,p) = one;
  for (int p=1; p<Ny_; p+=2)
    A(Ny_-1,p) = one;
    
  // Construct RHS vector
  pressure_.makePhysical(trans);
  ComplexChebyCoeff f(Ny_, a_, b_, Physical);
  for (int n=0; n<Ny_; ++n)
    f.set(n, ibR*pressure_[n]);
  f.makeSpectral(trans);
  pressure_.makeSpectral(trans);

  ComplexVector bb(Ny_);
  for (int n=0; n<Ny_-2; ++n)
    bb(n) = f[n];
  bb(Ny_-2) = Complex(0.0, 0.0);
  bb(Ny_-1) = Complex(0.0, 0.0);
  
  ComplexVector  x = A.solve(bb);
  for (int n=0; n<Ny_; ++n)
    w_.set(n, x(n));
}
