#include "turbstats.h"
#include <fstream>
#include <iomanip>

TurbStats::TurbStats() 
  :
  count_(0),
  nu_(0)
{}

/***********************************
TurbStats(const string& filebase) 
  :
  count_(0),
  nu_(0)
{
  string filename(filebase);
  filename += string(".asc");
  ifstream is(filename.c_str());
  if (!is.good()) {
    cerr << "ChebyCoeff::ChebyCoeff(filebase) : can't open file " << filename << endl;
    abort();
  }
  // Read in header. Form is "%N a b nu count wallunits"
  char c;
  int N;
  Real a;
  Real b;
  bool wallunits;
  is >> c;
  if (c != '%') {
    string message("TurbStats(filebase): bad header in file ");
    message += filename;
    cerr << message << endl;
    abort();
  }
  is >> N >> a >> b >> nu_ >> count_ >> wallunits;
  ChebyCoeff shape(N,a,b,Physical);
  Ubase_ = shape;
  Umean_ = shape;
  ubase_ = shape;
  uu_ = shape;
  uv_ = shape;
  uw_ = shape;
  vv_ = shape;
  vw_ = shape;
  ww_ = shape;
  assert(is.good());
  // Read in mean values, in wall units or physical
  for (int i=0; i<N; ++i) {
    is >> uu_[i] >> uv_[i] >> uw_[i] >> vv_[i] >> vw_[i] >> ww_[i] 
       >> Ubase_[i] >> ubase_[i] >> Umean_[i];
    assert(is.good());
  }
  is.close();

  // Now convert from means (and wall units) back to accumulation sums.
}
***************/
  
TurbStats::TurbStats(const ChebyCoeff& Ubase, Real nu)
  :
  count_(0),
  nu_(nu),
  Ubase_(Ubase),
  Umean_(Ubase.numModes(),Ubase.a(),Ubase.b(),Physical),
  ubase_(Ubase.numModes(),Ubase.a(),Ubase.b(),Physical),
  uu_(Ubase.numModes(),Ubase.a(),Ubase.b(),Physical),
  uv_(Ubase.numModes(),Ubase.a(),Ubase.b(),Physical),
  uw_(Ubase.numModes(),Ubase.a(),Ubase.b(),Physical),
  vv_(Ubase.numModes(),Ubase.a(),Ubase.b(),Physical),
  vw_(Ubase.numModes(),Ubase.a(),Ubase.b(),Physical),
  ww_(Ubase.numModes(),Ubase.a(),Ubase.b(),Physical)
{
  if (Ubase_.state() == Spectral) {
    ChebyTransform t(Ubase_.numModes());
    t.makePhysical(Ubase_);
  }
    
}

void TurbStats::addData(FlowField& un, FlowField& tmp) {
  fieldstate xzstate = un.xzstate();
  fieldstate ystate = un.ystate();
  int Nx = un.numXgridpts();
  int Ny = un.numYgridpts();
  int Nz = un.numZgridpts();

  // Add 0,0 profile to mean velocity
  un.makeSpectral_xz();
  un.makePhysical_y();
  
  for (int ny=0; ny<Ny; ++ny) {
    ubase_[ny] += Re(un.cmplx(0,ny,0,0));
    Umean_[ny] += Re(un.cmplx(0,ny,0,0)) + Ubase_[ny];
  }
  
  // Compute uu(y).
  un.makePhysical_xz();
  tmp.setToZero();
  tmp.setState(Physical, Physical);
  
  // tmp[0]=uu, tmp[1]=vv, tmp[2]=ww
  for (int nx=0; nx<Nx; ++nx)
    for (int nz=0; nz<Nz; ++nz) 
      for (int ny=0; ny<Ny; ++ny) {
	// cache thrash
	Real u = un(nx,ny,nz,0) + Ubase_[ny];
	Real v = un(nx,ny,nz,1);
	Real w = un(nx,ny,nz,2);
	tmp(nx,ny,nz,0) = u*u;
	tmp(nx,ny,nz,1) = u*v;
	tmp(nx,ny,nz,2) = u*w;
	tmp(nx,ny,nz,3) = v*v;
	tmp(nx,ny,nz,4) = v*w;
	tmp(nx,ny,nz,5) = w*w;
      }
  tmp.makeSpectral_xz();
  for (int ny=0; ny<Ny; ++ny) {
    uu_[ny] += Re(tmp.cmplx(0,ny,0,0));
    uv_[ny] += Re(tmp.cmplx(0,ny,0,1));
    uw_[ny] += Re(tmp.cmplx(0,ny,0,2));
    vv_[ny] += Re(tmp.cmplx(0,ny,0,3));
    vw_[ny] += Re(tmp.cmplx(0,ny,0,4));
    ww_[ny] += Re(tmp.cmplx(0,ny,0,5));
  } 
  ++count_;
  un.makeState(xzstate,ystate);
}

Real TurbStats::ustar() const {
  ChebyCoeff U = Umean_;
  U *= 1.0/count_;
  ChebyTransform trans(U.numModes());
  trans.chebyfft(U);
  ChebyCoeff dUdy = diff(U);
  trans.ichebyfft(dUdy);
  return sqrt(nu_/2*(abs(dUdy.eval_a() + abs(dUdy.eval_b()))));
}   


Real TurbStats::bulkReynolds() const {
  // calculate bulk velocity
  ChebyCoeff U = Umean_;
  U *= 1.0/count_;
  ChebyTransform trans(U.numModes());
  trans.chebyfft(U);
  Real Ubulk = U.mean();
  return 0.5*(U.b() - U.a())*Ubulk/nu_;
}   

Real TurbStats::parabolicReynolds() const {
  return 1.5*bulkReynolds();
}

Real TurbStats::centerlineReynolds() const {
  ChebyCoeff U = Umean_;
  U *= 1.0/count_;
  ChebyTransform trans(U.numModes());
  trans.chebyfft(U);
  Real center = 0.5*(U.a() + U.b());
  Real Ucenter = U.eval(center);
  return 0.5*(U.b() - U.a())*Ucenter/nu_;
}

Vector TurbStats::yplus() const {
  int Ny = Umean_.numModes();
  Real a = Umean_.a();
  Real b = Umean_.b();
  Real c = ustar()/nu_;
  
  Vector y = chebypoints(Ny, a, b);
  Vector yp(Ny);
  for (int ny=0; ny<Ny; ++ny)
    yp[ny] = c*(y[ny] - a);
  return yp;
}

ChebyCoeff TurbStats::Umean() const {
  ChebyCoeff rtn(Umean_);
  rtn *= 1.0/count_;
  return rtn;
}

ChebyCoeff TurbStats::ubase() const {
  ChebyCoeff rtn(ubase_);
  rtn *= 1.0/count_;
  return rtn;
}

ChebyCoeff TurbStats::uu() const {
  int Ny = uu_.numModes();
  ChebyCoeff rtn(Ny, uu_.a(), uu_.b(), Physical);
  Real c = 1.0/count_;
  for (int ny=0; ny<Ny; ++ny)
    rtn[ny] = c*uu_[ny] - square(c*Umean_[ny]);
  
  return rtn;
}

ChebyCoeff TurbStats::uv() const {
  ChebyCoeff rtn = uv_;
  rtn *= 1.0/count_;
  return rtn;
}

ChebyCoeff TurbStats::uw() const {
  ChebyCoeff rtn = uw_;
  rtn *= 1.0/count_;
  return rtn;
}
ChebyCoeff TurbStats::vv() const {
  ChebyCoeff rtn = vv_;
  rtn *= 1.0/count_;
  return rtn;
}
ChebyCoeff TurbStats::vw() const {
  ChebyCoeff rtn = vw_;
  rtn *= 1.0/count_;
  return rtn;
}
ChebyCoeff TurbStats::ww() const {
  ChebyCoeff rtn = ww_;
  rtn *= 1.0/count_;
  return rtn;
}

void TurbStats::msave(const string& filebase, bool wallunits) const {
  string filename(filebase);
  filename += string(".asc");
  ofstream os(filename.c_str());
  os << setprecision(REAL_DIGITS);
  char s = ' ';
  int Ny = uu_.numModes();
  Real u_star = ustar();

  os << '%' << Ny <<s<< uu_.a() <<s<< uu_.b() <<s<< nu_ <<s<< count_ 
     <<s<< wallunits <<s<< u_star << '\n';

  Real c = (wallunits) ? 1.0/square(u_star) : 1.0;
  Real d = (wallunits) ? 1.0/u_star : 1.0;
  c /= count_;
  d /= count_;
  for (int ny=0; ny<Ny; ++ny) {
    os << c*(uu_[ny] - square(Umean_[ny])/count_) << s
       << c*uv_[ny] << s
       << c*uw_[ny] << s
       << c*vv_[ny] << s
       << c*vw_[ny] << s
       << c*ww_[ny] << s
       << d*Umean_[ny] << s
       << d*ubase_[ny] << s
       << d*Ubase_[ny] << '\n';
  }
}
