#include <iostream>
#include <fstream>
#include <sstream>
#include <iomanip>
#include "time.h"
#include "flowfield.h"
#include "cfvector.h"
#include "chebyshev.h"
#include "tausolver.h"
#include "nsintegrator.h"
#include "complexdefs.h"

// This program compares the growth of OrrSommerfeld eigenfuncs integrated 
// with three methods:
// (1) direct numerical simulation (fully 3d with nonlinearity)
// (2) analytic expression exp(-i omega t) * OS eigenfunction 
// (3) an ODE integration of the linearized equations

// The DNS is represented by  FlowField un (u nonlinear) 
// The analytic expression by FlowField ul (u linear)
// The ODE is represented by a Complex number ark (a runge-kutta)
// Admittedly the names of these variables are poorly chosen.

// To compare the DNS and analytic expression to the ODE integration,
// we compute the inner product of un and ul with a unit OS eigenfunction:
// Complex an = L2InnerProduct(un, unit_useig);
// Complex al = L2InnerProduct(ul, unit_useig);
// If the three integration methods are identical, an == al == ark.

int closest(Real x) {
  int n0 = int(x); // rounds down;
  int n1 = n0 + 1;
  return (abs(x-n0) < abs(n1-x)) ? n0 : n1;
}
  

int main() {
  fftw_loadwisdom();

  const int Nx=16;
  const int Ny=65;
  const int Nz=4;
  const int Nd=3;
  const Real Lx=2*pi;
  const Real Lz=pi;
  const Real a= -1.0;
  const Real b=  1.0;
  const Real Reynolds = 7500.0;
  const Real nu = 1.0/Reynolds;
  const Real dPdx = -2.0*nu;    // mean streamwise pressure gradient.

  const Real T = 1.0;

  DNSFlags flags;
  flags.nonlinearity = Linearized;
  flags.timestepping = SBDF3;
  flags.initstepping = SMRK2;

  Real scale =  0.01;
  const char s = ' ';
  
  ChebyTransform trans(Ny);
  Vector y = chebypoints(Ny,a,b);

  // Get Orr-Sommerfeld eigenfunction from file. The data is the 
  // y-dependence of the eigenfunction's (u,v) components. x dependence
  // is exp(2 pi i x / Lx). Eigfunction is const in z. Reconstruct
  // w from velocity field, pressure from OS eqns. 
  BasisFunc ueig("os_ueig10_65");
  ComplexChebyCoeff peig("os_peig10_65");
  ueig.makeSpectral(trans);
  peig.makeSpectral(trans);  

  // The temporal variation of the eigenfunction is exp(lambda t)
  Complex omega = 0.24989153647208251 + I*0.0022349757548207664;
  Complex lambda = -1.0*I*omega;

  // lambda_v == the component of the eigenvalue due to viscosity
  // lambda_b == the component of the eigenvalue due to the base flow
  // This is useful for integrating the eigenfunction numerically with 
  // the same operator-splitting algorithm as the DNS.
  Complex lambda_v(-0.006564724662891, 0.0);
  Complex lambda_b = lambda - lambda_v; 

  cout << "   omega = " << omega << endl;
  cout << "  lambda = " << lambda << endl;

  ChebyCoeff Ubase(Ny,a,b,Physical);
  for (int ny=0; ny<Ny; ++ny) 
    Ubase[ny] = 1.0 - square(y[ny]);

  FlowField un(Nx,Ny,Nz,Nd,Lx,Lz,a,b);  // numerical velocity field
  FlowField pn(Nx,Ny,Nz,1,Lx,Lz,a,b);   // numerical pressure field
  FlowField ul(Nx,Ny,Nz,Nd,Lx,Lz,a,b);  // linear-solution velocity
  FlowField pl(Nx,Ny,Nz,1,Lx,Lz,a,b);   // linear-solution pressure
  
  // Calculate L2Norm of poisseuille base flow and perturbation field 
  // and rescale so that perturb/poisseuille is equal to the given scale.
  un.setState(Spectral, Physical);
  un += Ubase;
  un.makeSpectral();
  Real uPoissNorm = L2Norm(un);
  un.setToZero();
  un.addProfile(ueig, true);  
  pn.setToZero();
  //pn.addProfile(peig, pn.mx(1), pn.mz(0), 0, true);  


  Real uPerturbNorm = L2Norm(un);
  ueig *= scale*uPoissNorm/uPerturbNorm;
  peig *= scale*uPoissNorm/uPerturbNorm;
  
  
  ofstream errs("errscaling.asc");

  for (int N=4; N<=512; N *= 2) {

    Real dt = T/N;

    // Set un to perturbation at t=-dt
    un.setToZero();
    un.addProfile(ueig, true);

    FlowField zero = un;
    zero.setToZero();

    cout << "building DNS..." << flush;
    NSIntegrator dns(zero, Ubase, nu, dt, flags);
    cout << "done" << endl;

    {
    
      FlowField tmp  = zero;

      dns.u_[3] = zero;
      dns.f_[3] = zero;
      ueig *= exp(lambda*(-3*dt)); // set for t = -4 dt
      dns.u_[2].addProfile(ueig, true);
      navierstokesNL(dns.u_[2], Ubase, dns.f_[2],tmp,flags.nonlinearity);

      dns.u_[2] = zero;
      dns.f_[2] = zero;
      ueig *= exp(lambda*(dt)); // set for t = -3 dt
      dns.u_[2].addProfile(ueig, true);
      navierstokesNL(dns.u_[2], Ubase, dns.f_[2],tmp,flags.nonlinearity);

      dns.u_[1] = zero;
      dns.f_[1] = zero;
      ueig *= exp(lambda*dt);    // set for t = -2 dt
      dns.u_[1].addProfile(ueig, true);
      navierstokesNL(dns.u_[1], Ubase, dns.f_[1],tmp,flags.nonlinearity);
    
      dns.u_[0] = zero;
      dns.f_[0] = zero;
      ueig *= exp(lambda*dt);    // set for t = -dt
      dns.u_[0].addProfile(ueig, true);
      navierstokesNL(dns.u_[0], Ubase, dns.f_[0],tmp,flags.nonlinearity);
    
      //ueig *= exp(lambda*dt);    // set for t = 0
    } 
    // Reset perturbation velocity and pressure for t=0, numerical and linear
    un.setToZero();
    pn.setToZero();
    un.addProfile(ueig, true);
    pn.addProfile(peig,1,0,0, true);

    // Advance DNS to t = T = N*dt    
    clock_t start = clock();
    dns.advance(un, pn, N); 
    clock_t stop = clock();
    Real cputime = Real(stop-start)/ CLOCKS_PER_SEC;

    // Advance analytic solution same N*dt
    BasisFunc exp_ueig(ueig);  
    ComplexChebyCoeff exp_peig(peig);
    exp_ueig *= exp(lambda*(N*dt));
    exp_peig *= exp(lambda*(N*dt));
    ul.setToZero();
    pl.setToZero();
    ul.addProfile(exp_ueig, true);
    pl.addProfile(exp_peig, 1, 0, 0, true);

    cout << endl;
    errs << dt << s << L2Dist(un,ul)/L2Norm(ul) << s << L2Dist(pn,pl)/L2Norm(pl) << s << cputime << endl;
    cout << dt << s << L2Dist(un,ul)/L2Norm(ul) << s << L2Dist(pn,pl)/L2Norm(pl) << s << cputime << endl;
  }    
  fftw_savewisdom();
}

    


  
