/* basisfunc.h: a Complex-Vector-valued Chebyshev expansion class.
 * Channelflow-0.9
 *
 * Copyright (C) 2001-2005  John F. Gibson
 *
 * John F. Gibson
 * Center for Nonlinear Sciences
 * School of Physics
 * Georgia Institute of Technology
 * Atlanta, GA 30332-0430
 *
 * gibson@cns.physics.gatech.edu
 * jfg@member.fsf.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, U
 */

#ifndef CHANNELFLOW_BASISFUNC_H
#define CHANNELFLOW_BASISFUNC_H

#include "channelflow/mathdefs.h"
#include "channelflow/chebyshev.h"
#include "channelflow/array.h"

class BasisFunc {
public:
  BasisFunc();
  BasisFunc(const std::string& filebase);
  //BasisFunc(std::istream& is);
  BasisFunc(int Nd, int Ny, int kx, int kz, Real Lx, Real Lz, Real a, Real b, 
	    fieldstate s=Spectral);
  BasisFunc(int Ny, int kx, int kz, Real Lx, Real Lz, Real a, Real b, 
	    fieldstate s=Spectral);    // defaults to Nd=3
  BasisFunc(int Ny, const BasisFunc& f);

  BasisFunc(const ComplexChebyCoeff& u,
	    const ComplexChebyCoeff& v,
	    const ComplexChebyCoeff& w,
	    Real lambda, int kx, int kz, Real Lx, Real Lz);

  void save(const std::string& filebase, fieldstate s=Physical) const; 
  void binaryDump(std::ostream& os) const; 
  void binaryLoad(std::istream& is); 
  
  void randomize(bool adiri, bool bdiri, Real decay=0.7);
  void interpolate(const BasisFunc& f);
  void reflect(const BasisFunc& f);

  inline int Nd() const;
  inline int Ny() const;
  inline int kx() const;
  inline int kz() const;
  inline Real Lx() const;
  inline Real Lz() const;
  inline Real a() const;
  inline Real b() const;
  inline Real rmsval() const; // root mean square value == POD eigenval
  inline Real lambda() const; // root mean square value == POD eigenval
  inline fieldstate state() const;
  
  void resize(int Ny);
  void setBounds(Real Lx, Real Lz, Real a, Real b);
  void setkxkz(int kx, int kz);

  void setState(fieldstate s);
  void setRmsval(Real lamda);
  void setToZero();
  void conjugate();
  void fill(const BasisFunc& f);

  void chebyfft();
  void ichebyfft();
  void makeSpectral();
  void makePhysical();
  void makeState(fieldstate s);

  void chebyfft(const ChebyTransform& t);
  void ichebyfft(const ChebyTransform& t);
  void makeSpectral(const ChebyTransform& t);
  void makePhysical(const ChebyTransform& t);
  void makeState(fieldstate s, const ChebyTransform& t);

  const ComplexChebyCoeff& u() const;
  const ComplexChebyCoeff& v() const;
  const ComplexChebyCoeff& w() const;
  const ComplexChebyCoeff& component(int i) const;
  const ComplexChebyCoeff& operator[](int i) const;
  ComplexChebyCoeff& u();
  ComplexChebyCoeff& v();
  ComplexChebyCoeff& w();
  ComplexChebyCoeff& component(int i);
  ComplexChebyCoeff& operator[](int i);

  Real bcNorm(bool a_dirichlet=true, bool b_dirichlet=true) const;  
  //Real divNorm() const; 
  bool geometryCongruent(const BasisFunc& f) const;
  bool congruent(const BasisFunc& f) const; // geomCongruent & kx,kz equality

  BasisFunc& operator *= (const BasisFunc& g);
  BasisFunc& operator += (const BasisFunc& g);
  BasisFunc& operator -= (const BasisFunc& g);
  BasisFunc& operator *= (Real c);
  BasisFunc& operator *= (Complex c);   // gcc-2.95 can't handle this

  
protected:
  int Nd_;
  int Ny_;
  int kx_;
  int kz_;
  Real Lx_;
  Real Lz_;
  Real a_;
  Real b_;
  fieldstate state_;
  Real lambda_;         // eigenvalue of basis function
  
  array<ComplexChebyCoeff> u_;  
};

bool operator==(const BasisFunc& f, const BasisFunc& g);
bool operator!=(const BasisFunc& f, const BasisFunc& g);
// L2Norm2(f)  == Int ||f||^2     dx dy dz (/(Lx Ly Lz) if normalize==true)
// bcNorm2(f)  == Int ||f||^2     dx dz at y=a,b (/(Lx Lz) if normalize==true)
// divNorm2(f) == Int ||div f||^2 dx dy dz (/(Lx Ly Lz) if normalize==true)

// innerProduct(f,g) == Int f g*  dx dy dz (/(Lx Ly Lz) if normalize==true)

// L2Norm(f)    == sqrt(L2Norm2(f))
// L2Dist(f,g)  == sqrt(L2Dist2(f,g))
// L2Dist2(f,g) == L2Norm2(f-g)
// etc.

// These are not so simple to compute for individual Fourier modes.X
//Real L1Norm(const BasisFunc& f, bool normalize=true);
//Real L1Dist(const BasisFunc& f, const BasisFunc& g, bool normalize=true);
//Real LinfNorm(const BasisFunc& f);
//Real LinfDist(const BasisFunc& f, const BasisFunc& g);

Real L2Norm(const BasisFunc& f, bool normalize=true);
Real L2Norm2(const BasisFunc& f, bool normalize=true);
Real L2Dist(const BasisFunc& f, const BasisFunc& g, bool normalize=true);
Real L2Dist2(const BasisFunc& f, const BasisFunc& g, bool normalize=true);
Complex L2InnerProduct(const BasisFunc& f0, const BasisFunc& f1, 
		       bool normalize=true);

Real chebyNorm(const BasisFunc& f, bool normalize=true);
Real chebyNorm2(const BasisFunc& f, bool normalize=true);
Real chebyDist(const BasisFunc& f, const BasisFunc& g, bool normalize=true);
Real chebyDist2(const BasisFunc& f, const BasisFunc& g, bool normalize=true);
Complex chebyInnerProduct(const BasisFunc& f0, const BasisFunc& f1, 
		       bool normalize=true);

Real norm(const BasisFunc& f, NormType n, bool normalize=true);
Real norm2(const BasisFunc& f, NormType n, bool normalize=true);
Real dist(const BasisFunc& f, const BasisFunc& g, NormType n, bool nrmlz=true);
Real dist2(const BasisFunc& f, const BasisFunc& g, NormType n,bool nrmlz=true);
Complex innerProduct(const BasisFunc& f0, const BasisFunc& f1, 
		     NormType n, bool normalize=true);

Real bcNorm(const BasisFunc& f, bool normalize=true);
Real bcNorm2(const BasisFunc& f, bool normalize=true);
Real bcDist(const BasisFunc& f, const BasisFunc& g, bool normalize=true);
Real bcDist2(const BasisFunc& f, const BasisFunc& g, bool normalize=true);

Real divNorm(const BasisFunc& f, bool normalize=true);
Real divNorm2(const BasisFunc& f, bool normalize=true);
Real divDist(const BasisFunc& f, const BasisFunc& g, bool normalize=true);
Real divDist2(const BasisFunc& f, const BasisFunc& g, bool normalize=true);

BasisFunc xdiff(const BasisFunc& f);
BasisFunc ydiff(const BasisFunc& f);
BasisFunc zdiff(const BasisFunc& f);
BasisFunc laplacian(const BasisFunc& f);
BasisFunc gradient(const BasisFunc& f, int i); // grad of ith comp of f
BasisFunc gradient(const ComplexChebyCoeff& f, int kx,int kz, Real Lx,Real Lz);
ComplexChebyCoeff divergence(const BasisFunc& f);

void dotgrad(const BasisFunc& f_n, const BasisFunc& grad_f_pu,
	     const BasisFunc& grad_f_pv, const BasisFunc& grad_f_pw,
	     ComplexChebyCoeff& tmp, BasisFunc& rtn);

// An expensive convenience.
BasisFunc dotgrad(const BasisFunc& f_n, const BasisFunc& f_p);
ComplexChebyCoeff dot(const BasisFunc& f, const BasisFunc& g);

BasisFunc cross(const BasisFunc& f, const BasisFunc& g);
BasisFunc curl(const BasisFunc& f);


void ubcFix(ChebyCoeff& u, bool a_dirichlet, bool b_dirichlet);
void ubcFix(ComplexChebyCoeff& u, bool a_dirichlet, bool b_dirichlet);
void vbcFix(ChebyCoeff& v, bool a_dirichlet, bool b_dirichlet);
void vbcFix(ComplexChebyCoeff& v, bool a_dirichlet, bool b_dirichlet);

inline int BasisFunc::Nd() const {return Nd_;}
inline int BasisFunc::Ny() const {return Ny_;}
inline int BasisFunc::kx() const {return kx_;}
inline int BasisFunc::kz() const {return kz_;}
inline Real BasisFunc::Lx() const {return Lx_;}
inline Real BasisFunc::Lz() const {return Lz_;}
inline Real BasisFunc::a() const {return a_;}
inline Real BasisFunc::b() const {return b_;}
inline Real BasisFunc::rmsval() const {return lambda_;}
inline Real BasisFunc::lambda() const {return lambda_;}
inline fieldstate BasisFunc::state() const {return state_;}
  

#endif
