<?php
/******************************************************************************
  A library for manipulating path-like category strings
  Copyright (C) 2007  Sylvain Hall
  
  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
  MA  02110-1301, USA.
******************************************************************************/

/*
An absolute path begins with the root character "/" and
A path *always* ends by a "/".
*/

/**
 * Definition of the category separator
 */
define("CAT_SEPARATOR", ",");

/**
 * Definition of the subcategory delimitor.  Note: it is assumed that
 * this delimiter is a single character (this is necessary for the
 * implementation of some functions).  Multi-character delimitors are not
 * supported.
 */
define("CAT_PATH", "/");

/**
 * Definition of the "go up" symbol
 */
define ("CAT_GOUP", "..");

/**
 * Returns the category prefix of the current category.
 * For example, the prefix of "a/b/c/" is "a/b/".
 * @param $path The path string to process
 * @return The (possibly empty) prefix of the path string, FALSE if
 * the string has a bad format (for example, contains more than one path)
 */
function cat_prefix($path)
{
  if (strstr($path, CAT_SEPARATOR) !== FALSE)
  {
    // String contains multiple paths
    return FALSE;
  }
  // If path is root, error (no suffix possible)
  if ($path === CAT_PATH)
  {
    return FALSE;
  }
  // Finds next-to-last occurrence of the path delimiter
  $subs = strrchr(substr($path, 0, strlen($path) - 1), CAT_PATH);
  if ($subs === FALSE)
  {
    // No path delimiter
    return $path;
  }
  return substr($path, 0, strlen($path) - strlen($subs));
}

/**
 * Returns the category suffix of the current category.
 * For example, the suffix of "/a/b/c/" is "c".  It is not a path, so the
 * path delimiter is not present at the end.
 * @param $path The path string to process
 * @return The suffix of the path string, FALSE if
 * the string has a bad format (for example, contains more than one path)
 */
function cat_suffix($path)
{
  if (strstr($path, CAT_SEPARATOR) !== FALSE)
  {
    // String contains multiple paths
    return FALSE;
  }
  // If path is root, it is its own suffix
  if ($path === CAT_PATH)
  {
    return $path;
  }
  // Finds next-to-last occurrence of the path delimiter
  $subs = strrchr(substr($path, 0, strlen($path) - 1), CAT_PATH);
  if ($subs === FALSE)
  {
    // No path delimiter
    return $path;
  }
  return substr($subs, 1, strlen($subs) - 1);
}

/**
 * Merges two paths, i.e. appends the second path at the end of the first.
 * For example, if "/a/b/c" is path1 and "d/e/f" is path2, the resulting
 * path is "/a/b/c/d/e/f".
 * This function resolves eventual "go up characters" that might be placed
 * in path2, i.e. if "/a/b/c" is path1 and "../d/e/f" is path2, the
 * resulting path is "/a/b/d/e/f".
 * This function also resolves root characters, i.e.
 * if "/a/b/c" is path1 and "/d/e/f" is path2, the resulting
 * path is "/d/e/f".
 * Root and "go up" characters may appear anywhere in path2; thus screwed-up
 * things like the fusion of "/a/b/c" to "../d/e/../f/g/../h/" results in
 * "/a/b/d/f/h/" as expected.
 * @param $path1 The original path
 * @param $path2 The path to merge
 * @return The merged path
 */
function cat_merge_paths($path1, $path2)
{
  if (strstr($path1, CAT_SEPARATOR) !== FALSE || strstr($path2, CAT_SEPARATOR) !== FALSE)
  {
    // Either string contains multiple paths
    return FALSE;
  }
  $result = $path1.$path2;
  // Checks last occurrence of two successive delimiters and truncates
  // everything before
  $pos = 0;
  while (strpos($result, CAT_PATH.CAT_PATH) !== FALSE)
  {
    $pos = strpos($result, CAT_PATH.CAT_PATH);
    $result = substr($result, $pos + strlen(CAT_PATH), strlen($result) - $pos - strlen(CAT_PATH));
  }
  // Now resolves "go up" symbols one by one
  while (strpos($result, CAT_GOUP.CAT_PATH) !== FALSE)
  {
    // Checks first occurrence of the "go up" symbol
    $pos = strpos($result, CAT_GOUP);
    $begin = substr($result, 0, $pos);
    $end = substr($result, $pos + strlen(CAT_GOUP.CAT_PATH), strlen($result)
      - ($pos + strlen(CAT_GOUP.CAT_PATH)));
    // Checks next-to-last occurrence of the path symbol
    // Note: it is assumed here that CAT_PATH is a single character (use of
    // strrchr())
    $dir_to_cut = strrchr(substr($begin, 0, strlen($begin) - 1), CAT_PATH);
    if ($dir_to_cut === FALSE) // Cannot go higher than root
      return FALSE;
    $consider = substr($begin, 0, strlen($begin)
      - strlen($dir_to_cut));
    $result = $consider.$end;
  }
  return $result;
}

/**
 * In a multi-path string, returns the complete paths that match a given
 * prefix.
 * @param $multi_path A string containing one or more paths
 * @param $prefix The prefix to look for
 * @return All the paths in $multi_path, if any, that match the prefix.
 */
function cat_branches($multi_path, $prefix)
{
  $paths = explode(CAT_SEPARATOR, $multi_path);
  $out = "";
  foreach ($paths as $p)
  {
    if (strpos($p, $prefix) === 0)
    {
      $out .= ",".$p;
    }
  }
  if (strlen($out) > 1)
    return substr($out, 1, strlen($out) - 1);
  return "";
}

/**
 * Gets the top category of a category string.  For example, the top
 * category of "/a/b/c/" is "a"; the top category of the root is the empty
 * string.
 * @param $s A single path
 * @return The top category of that path
 */
function cat_topcat($path)
{
  if (strstr($path, CAT_SEPARATOR) !== FALSE)
  {
    // String contains multiple paths
    return FALSE;
  }
  // The top category of the root is the empty string
  if ($path === CAT_PATH)
    return "";
  $pos = 0;
  $pos = strpos(substr($path, 1, strlen($path) - 1), CAT_PATH);
  if ($pos === FALSE)
  {
    // Incorrectly formatted path
    return FALSE;
  }
  return substr($path, 1, $pos);
}

/**
 * In a multi-path string, returns the list of possible immediate
 * descendants for a given prefix.  For example, if the string is
 * "/a/b/c/,/a/b/d,/e/f" and the prefix is "/a/b/", the immediate
 * descendants are "c,d".
 * @param $multi_path A string containing one or more paths
 * @param $prefix The prefix to look for
 * @return The list of immediate descendants, each separated by the
 * defined separator
 */
function cat_subcats($multi_path, $prefix)
{
  $branches = cat_branches($multi_path, $prefix);
  $branches_a = explode(CAT_SEPARATOR, $branches);
  $poscat = "";
  foreach ($branches_a as $b)
  {
    $suffix = substr($b, strlen($prefix), strlen($b) - strlen($prefix));
    $poscat .= CAT_SEPARATOR.cat_topcat(CAT_PATH.$suffix);
  }
  if (strlen($poscat) > 1)
    return substr($poscat, 1, strlen($poscat) - 1);
  return "";
}
?>
