// dzn-runtime -- Dezyne runtime library
//
// Copyright © 2014, 2015, 2016, 2017, 2019, 2020, 2021 Jan (janneke) Nieuwenhuizen <janneke@gnu.org>
// Copyright © 2015, 2016, 2017, 2019, 2020, 2021, 2022 Rutger van Beusekom <rutger@dezyne.org>
// Copyright © 2015 Paul Hoogendijk <paul@dezyne.org>
//
// This file is part of dzn-runtime.
//
// dzn-runtime is free software: you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation, either version 3 of the
// License, or (at your option) any later version.
//
// dzn-runtime is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with dzn-runtime.  If not, see <http://www.gnu.org/licenses/>.
//
// Commentary:
//
// Code:

#include <dzn/runtime.hh>
#include <dzn/coroutine.hh>

#include <algorithm>
#include <iostream>

namespace dzn
{
  std::ostream debug(nullptr);
  runtime::runtime(){}

  void trace(std::ostream& os, port::meta const& m, const char* e)
  {
    if (!os.rdbuf())
      return;
    os << path(m.require.meta, m.require.name) << "." << e << " -> "
       << path(m.provide.meta, m.provide.name) << "." << e << std::endl;
  }

  void trace_out(std::ostream& os, port::meta const& m, const char* e)
  {
    if (!os.rdbuf())
      return;
    os << path(m.require.meta, m.require.name) << "." << e << " <- "
       << path(m.provide.meta, m.provide.name) << "." << e << std::endl;
  }

  void trace_qin(std::ostream& os, port::meta const& m, const char* e)
  {
    if (!os.rdbuf())
      return;

    if (path(m.provide.meta) == "<external>")
      os << path(m.require.meta, "<q>") << " <- "
         << path(m.provide.meta, m.provide.name) << "." << e << std::endl;
    else
      os <<  path(m.provide.meta, m.provide.name) << ".<q> <- "
         <<  path(m.require.meta, m.require.name) << "." << e << std::endl;
  }

  void trace_qout(std::ostream& os, port::meta const& m, const char* e)
  {
    if (!os.rdbuf())
      return;
    os << path(m.require.meta, m.require.name) << "." << e << " <- "
       << path(m.require.meta, "<q>") << std::endl;
  }

  bool runtime::external(void* scope) {
    return (states.find(scope) == states.end());
  }

  size_t& runtime::handling(void* scope)
  {
    return states[scope].handling;
  }

  void*& runtime::deferred(void* scope)
  {
    return states[scope].deferred;
  }

  std::queue<std::function<void()> >& runtime::queue(void* scope)
  {
    return states[scope].queue;
  }

  bool& runtime::performs_flush(void* scope)
  {
    return states[scope].performs_flush;
  }

  bool& runtime::skip_block(void* port)
  {
    return skip_port[port];
  }

  void runtime::flush(void* scope, size_t coroutine_id)
  {
    handling(scope) = 0;
    if(!external(scope))
    {
      std::queue<std::function<void()> >& q = queue(scope);
      while(! q.empty())
      {
        std::function<void()> event = q.front();
        q.pop();
        handle(scope, event, coroutine_id);
        handling(scope) = 0;
      }
      if (deferred(scope)) {
        void* tgt = deferred(scope);
        deferred(scope) = nullptr;
        if (!handling(tgt)) {
          runtime::flush(tgt, coroutine_id);
        }
      }
    }
  }

  void runtime::defer(void* src, void* tgt, const std::function<void()>& event, size_t coroutine_id)
  {
    if(!(src && performs_flush(src)) && !handling(tgt))
    {
      handle(tgt, event, coroutine_id);
      flush(tgt, coroutine_id);
    }
    else
    {
      deferred(src) = tgt;
      queue(tgt).push(event);
    }
  }
}
