/*
 * MouseMovedEventArgs.cs - Arguments for curses mouse movement events.
 *
 * Copyright (C) 2001  Southern Storm Software, Pty Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

namespace Curses
{

using System;

/// <summary>
/// <para>Event arguments for <see cref="T:Curses.MouseMovedEventHandler"/>
/// events.</para>
/// </summary>
public sealed class MouseMovedEventArgs : EventArgs
{
	private int x, y;
	private MouseState state;

	/// <summary>
	/// <para>Construct a new <see cref="T:Curses.MouseMovedEventArgs"/>
	/// object.</para>
	/// </summary>
	///
	/// <param name="x">
	/// <para>The X co-ordinate at which the event occurred.</para>
	/// </param>
	///
	/// <param name="y">
	/// <para>The Y co-ordinate at which the event occurred.</para>
	/// </param>
	///
	/// <param name="state">
	/// <para>Additional mouse state flags that accompanied the event.</para>
	/// </param>
	public MouseMovedEventArgs(int x, int y, MouseState state)
			{
				this.x = x;
				this.y = y;
				this.state = state;
			}

	/// <summary>
	/// <para>Get the X co-ordinate where the event occurred.</para>
	/// </summary>
	///
	/// <value>
	/// <para>Returns the X co-ordinate.</para>
	/// </value>
	public int X
			{
				get
				{
					return x;
				}
			}

	/// <summary>
	/// <para>Get the Y co-ordinate where the event occurred.</para>
	/// </summary>
	///
	/// <value>
	/// <para>Returns the Y co-ordinate.</para>
	/// </value>
	public int Y
			{
				get
				{
					return y;
				}
			}

	/// <summary>
	/// <para>Get the additional mouse state flags that
	/// accompanied the event.</para>
	/// </summary>
	///
	/// <value>
	/// <para>Returns the mouse state flags.</para>
	/// </value>
	public MouseState State
			{
				get
				{
					return state;
				}
			}

} // class MouseMovedEventArgs

} // namespace Curses
