/*
 * FloatBenchmark.cs - Implementation of the "FloatBenchmark" class.
 *
 * Copyright (C) 2001  Southern Storm Software, Pty Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

namespace PNetMark
{

using System;

/// <summary>
/// <para>The <see cref="T:PNetMark.FloatBenchmark"/> class implements
/// the <see cref="T:PNetMark.IBenchmark"/> interface, and runs number
/// of floating-point tests.</para>
/// </summary>
public class FloatBenchmark : IBenchmark
{
	// Internal state.
	public int maxDegrees;
	public double[,] xA;
	public double[,] vA;
	public double[,] vB;

	/// <summary>
	/// <para>Construct a new <see cref="T:PNetMark.FloatBenchmark"/>
	/// instance.</para>
	/// </summary>
	public FloatBenchmark()
	{
		maxDegrees = 185;
	}

	/// <summary>
	/// <para>Initialize the benchmark to run on this system.</para>
	/// </summary>
	///
	/// <param name="size">
	/// <para>A value that indicates the size of the benchmark to run.
	/// This should be zero to use the default value.</para>
	/// </param>
	///
	/// <returns>
	/// <para>Returns <see langword="true"/> if the benchmark could be
	/// initialized, or <see langword="false"/> otherwise.</para>
	/// </returns>
	public bool Initialize(int size)
	{
		int posn;
		if(size != 0)
		{
			maxDegrees = size;
		}
		xA = new double [3, 3];
		vA = new double [3, 20];
		vB = new double [3, 20];
		for(posn = 0; posn < 20; ++posn)
		{
			vA[0, posn] = (double)posn;
			vA[1, posn] = (double)-posn;
			vA[2, posn] = ((double)posn) * Math.PI;
		}
		return true;
	}

	/// <summary>
	/// <para>Get the name of the benchmark.</para>
	/// </summary>
	///
	/// <value>
	/// <para>A <see cref="T:System.String"/> that contains the name
	/// of the benchmark.</para>
	/// </value>
	public String Name
	{
		get
		{
			return "Float ";
		}
	}

	/// <summary>
	/// <para>Get the magnification factor for test scoring.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The magnification factor.</para>
	/// </value>
	public int Magnification
	{
		get
		{
			return 4449;
		}
	}

	/// <summary>
	/// <para>Run the benchmark.</para>
	/// </summary>
	public void Run()
	{
		int degrees;
		double radians;
		double sine, cosine;
		int posn, i, j;

		for(degrees = 0; degrees < maxDegrees; ++degrees)
		{
			// Compute the sin and cos of "degrees".
			radians = ((double)degrees) * Math.PI / 180.0;
			sine = Math.Sin(radians);
			cosine = Math.Cos(radians);

			// Initialize the xA array with a rotation matrix.
			xA[0, 0] = cosine;
			xA[1, 0] = sine;
			xA[2, 0] = 0.0;
			xA[0, 1] = -sine;
			xA[1, 1] = cosine;
			xA[2, 1] = 0.0;
			xA[0, 2] = 0.0;
			xA[1, 2] = 0.0;
			xA[2, 2] = 1.0;

			// Rotate the vectors.
			for(posn = 0; posn < 20; ++posn)
			{
				vB[0, posn] = 0.0;
				vB[1, posn] = 0.0;
				vB[2, posn] = 0.0;
				for(i = 0; i < 3; ++i)
				{
					for(j = 0; j < 3; ++j)
					{
						vB[i, posn] += xA[j, i] * vA[i, posn];
					}
				}
			}
		}
	}

	/// <summary>
	/// <para>Clean up after the benchmark has been run.</para>
	/// </summary>
	public void CleanUp()
	{
		// Nothing needs to be done here.
	}

} // class FloatBenchmark

} // namespace PNetMark
