/*
 * LoopBenchmark.cs - Implementation of the "LoopBenchmark" class.
 *
 * Copyright (C) 2001  Southern Storm Software, Pty Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

namespace PNetMark
{

using System;

/// <summary>
/// <para>The <see cref="T:PNetMark.LoopBenchmark"/> class implements
/// the <see cref="T:PNetMark.IBenchmark"/> interface to test the
/// looping facilities of the runtime engine.</para>
/// </summary>
public class LoopBenchmark : IBenchmark
{
	// Internal state.
	public int dummy;
	public int count;
	public int[] fibs;

	/// <summary>
	/// <para>Construct a new <see cref="T:PNetMark.LoopBenchmark"/>
	/// instance.</para>
	/// </summary>
	public LoopBenchmark()
	{
		dummy = 12;
		count = 64;
	}

	/// <summary>
	/// <para>Initialize the benchmark to run on this system.</para>
	/// </summary>
	///
	/// <param name="size">
	/// <para>A value that indicates the size of the benchmark to run.
	/// This should be zero to use the default value.</para>
	/// </param>
	///
	/// <returns>
	/// <para>Returns <see langword="true"/> if the benchmark could be
	/// initialized, or <see langword="false"/> otherwise.</para>
	/// </returns>
	public bool Initialize(int size)
	{
		if(size != 0)
		{
			count = size;
		}
		fibs = new int [count];
		return true;
	}

	/// <summary>
	/// <para>Get the name of the benchmark.</para>
	/// </summary>
	///
	/// <value>
	/// <para>A <see cref="T:System.String"/> that contains the name
	/// of the benchmark.</para>
	/// </value>
	public String Name
	{
		get
		{
			return "Loop  ";
		}
	}

	/// <summary>
	/// <para>Get the magnification factor for test scoring.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The magnification factor.</para>
	/// </value>
	public int Magnification
	{
		get
		{
			return 2692;
		}
	}

	/// <summary>
	/// <para>Run the benchmark.</para>
	/// </summary>
	public void Run()
	{
		int index;
		int index2;
		int value1;
		int sum1;
		int sum2;
		int temp;

		// Initialize the "fibs" array.
		fibs[0] = 1;
		for(index = 1; index < count; ++index)
		{
			fibs[index] = fibs[index - 1] + index;
		}

		// Sort the contents of the "fibs" array into descending order.
		sum1 = 0;
		sum2 = 0;
		for(index = 0; index < count; ++index)
		{
			for(index2 = 1; index2 < count; ++index2)
			{
				value1 = count + dummy;
				sum1 += value1;
				sum2 += 2;
				if(fibs[index2 - 1] < fibs[index2])
				{
					temp = fibs[index2 - 1];
					fibs[index2 - 1] = fibs[index2];
					fibs[index2] = temp;
				}
			}
		}
	}

	/// <summary>
	/// <para>Clean up after the benchmark has been run.</para>
	/// </summary>
	public void CleanUp()
	{
		// Nothing needs to be done here.
	}

} // class LoopBenchmark

} // namespace PNetMark
