/*
 * PNetMark.cs - Implementation of the "PNetMark" class.
 *
 * Copyright (C) 2001, 2002  Southern Storm Software, Pty Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

namespace PNetMark
{

using System;
using System.IO;

/// <summary>
/// <para>The <see cref="T:PNetMark.PNetMark"/> class is the main entry
/// point for the Portable.NET benchmark.</para>
/// </summary>
public class PNetMark
{
	// Internal state.
	private bool printHeader;
	private double score1;
	private double score2;
	private bool hasFloat;

	/// <summary>
	/// <para>Run the Portable.NET benchmark.</para>
	/// </summary>
	///
	/// <returns>
	/// <para>Returns the exit status for the process, which is non-zero if
	/// an error occurred.</para>
	/// </returns>
	public int Run(String[] args)
	{
		// Set the default command-line option values.
		printHeader = true;

		// Process the command-line options.
		foreach(String value in args)
		{
			if(value == "--no-header")
			{
				printHeader = false;
			}
			else if(value == "--version" || value == "-v")
			{
				PrintVersion();
				return 0;
			}
			else
			{
				PrintUsage();
				return 1;
			}
		}

		// Print the header.
		if(printHeader)
		{
			PrintHeader(Console.Out);
		}

		// Initialize the rolling score values, and determine if
		// the runtime engine has floating-point support and the
		// "Math" library.
		try
		{
			score1 = Math.Sin(0.0);
			score2 = 0.0;
			hasFloat = true;
		}
		catch(NotImplementedException)
		{
			Console.WriteLine("Warning: floating-point is not fully supported by the runtime engine");
			Console.WriteLine();
			hasFloat = false;
		}

		// Run all benchmarks.
		Run(new SieveBenchmark());
		Run(new LoopBenchmark());
		Run(new LogicBenchmark());
		Run(new StringBenchmark());
		if(hasFloat)
		{
			Run(new FloatBenchmark());
		}
		Run(new MethodBenchmark());

		// Compute the final score and report it.
		ReportScore();

		// Done.
		return 0;
	}

#if !STANDALONE
	/// <summary>
	/// <para>Main entry point for the Portable.NET benchmark.</para>
	/// </summary>
	///
	/// <returns>
	/// <para>Returns the exit status for the process, which is 1 if
	/// an error occurred.</para>
	/// </returns>
	public static int Main(String[] args)
	{
		PNetMark mark = new PNetMark();
		return mark.Run(args);
	}
#endif

	/// <summary>
	/// <para>Print the version header for this program.</para>
	/// </summary>
	///
	/// <param name="output">
	/// <para>The output stream to write the header to.</para>
	/// </param>
	private static void PrintHeader(TextWriter output)
	{
		output.WriteLine("PNetMark " + Version.VersionString + " - Portable.NET Benchmark");
		output.WriteLine("Copyright (c) 2001, 2002 Southern Storm Software, Pty Ltd.");
		output.WriteLine();
	}

	/// <summary>
	/// <para>Print the version information for this program.</para>
	/// </summary>
	private static void PrintVersion()
	{
		PrintHeader(Console.Out);
		Console.Out.WriteLine
		  ("ILSIZE comes with ABSOLUTELY NO WARRANTY.  This is free software,");
		Console.Out.WriteLine
		  ("and you are welcome to redistribute it under the terms of the");
		Console.Out.WriteLine
		  ("GNU General Public License.  See the file COPYING for further details.");
		Console.Out.WriteLine();
		Console.Out.WriteLine
		  ("Use the `--help' option to get help on the command-line options.");
	}

	/// <summary>
	/// <para>Print the usage information for this program.</para>
	/// </summary>
	private static void PrintUsage()
	{
		PrintHeader(Console.Error);
		Console.Error.WriteLine("Usage: pnetmark [options]");
		Console.Error.WriteLine();
		Console.Error.WriteLine
		  ("    --no-header");
		Console.Error.WriteLine
		  ("        Do not display the program header.");
		Console.Error.WriteLine
		  ("    --help");
		Console.Error.WriteLine
		  ("        Display this help message.");
		Console.Error.WriteLine
		  ("    --version or -v");
		Console.Error.WriteLine
		  ("        Display the program version.");
	}

	/// <summary>
	/// <para>Update the rolling score with a new benchmark score</para>
	/// </summary>
	private void UpdateScore(int score)
	{
		if(hasFloat)
		{
			score1 += Math.Log((double)score);
			score2 += 1.0;
		}
	}

	/// <summary>
	/// <para>Run a specific benchmark and report its score.</para>
	/// </summary>
	///
	/// <param name="benchmark">
	/// <para>The benchmark to be run.</para>
	/// </param>
	private void Run(IBenchmark benchmark)
	{
		RunBenchmark runner = new RunBenchmark(benchmark);
		Console.Out.Write(runner.Name);
		Console.Out.Write(" ... ");
		Console.Out.Flush();
		int score = runner.GetScore();
		Console.Out.WriteLine(score);
		if(score != 0)
		{
			UpdateScore(score);
		}
	}

	/// <summary>
	/// <para>Report the total PNetMark score.</para>
	///
	/// <remarks>If the runtime engine does not have support for
	/// floating-point operations, then the total score will not
	/// be reported.</remarks>
	/// </summary>
	private void ReportScore()
	{
		int finalScore;
		if(hasFloat)
		{
			if(score2 >= 0.5)
			{
				finalScore = (int)(Math.Exp(score1 / score2));
			}
			else
			{
				finalScore = 0;
			}
			Console.Out.WriteLine();
			Console.Out.WriteLine("PNetMark rating is " + finalScore);
		}
		Console.Out.WriteLine();
	}

} // class PNetMark

} // namespace PNetMark
