/*
 * Copyright (C) 2003 Daniel Heck
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: d_engine.hh,v 1.14.2.1 2003/10/05 13:09:57 dheck Exp $
 */
#ifndef D_ENGINE_HH
#define D_ENGINE_HH

#include "px/tools.hh"
#include "px/geom.hh"
#include "px/array2.hh"
#include "px/alist.hh"
#include "px/pxfwd.hh"
#include <list>

namespace display
{
    using px::V2;

    class DisplayLayer;
    class StatusBarImpl;

    typedef px::Rect ScreenArea;
    typedef px::Rect WorldArea;
    typedef std::list<Model*>   ModelList;

//----------------------------------------
// Display engine
//----------------------------------------

    class DisplayEngine {
    public:
        DisplayEngine (px::Screen *scr, int tilew, int tileh);
        ~DisplayEngine();

        /*
        ** Class configuration
        */
        void add_layer (DisplayLayer *l);
        void set_tilesize (int w, int h) { m_tilew=w; m_tileh=h; }
        int get_tilew () const { return m_tilew; }
        int get_tileh () const { return m_tileh; }
        void set_screen_area (const px::Rect & r);
        const px::Rect &get_area() const { return m_area; }

        int get_width() const { return m_width; }
        int get_height() const { return m_height; }

        /*
        ** Scrolling / page flipping
        */
        void set_offset (const px::V2 &off);
        void move_offset (const px::V2 &off);
        px::V2 get_offset () const { return m_offset; }

        /*
        ** Game-related stuff
        */
        void new_world (int w, int h);
        void tick (double dtime);

        /*
        ** Coordinate conversion
        */
        void      world_to_screen (const px::V2 & pos, int *x, int *y);
        WorldArea screen_to_world (const ScreenArea &a);
        ScreenArea world_to_screen (const WorldArea &a);

        V2 to_screen (const V2 &pos);
        V2 to_world (const V2 &pos);

        /*
        ** Screen upates
        */
        void mark_redraw_screen();
        void mark_redraw_area (const WorldArea &wa);

        void redraw_screen_area (const ScreenArea &a);
        void redraw_world_area (const WorldArea &a);

        void update_screen();
        void draw_all (px::GC &gc);

    private:
        void update_layer (DisplayLayer *l, WorldArea wa);

        /*
        ** Variables
        */

        std::vector<DisplayLayer *> m_layers;
        int m_tilew, m_tileh;
        px::Screen *m_screen;

        // Offset of screen
        px::V2 m_offset;

        // Screen area occupied by level display
        px::Rect m_area;

        // Width and height of the world in tiles
        int m_width, m_height;

        px::Array2<bool> m_redrawp;
    };

//----------------------------------------
// Display layer
//----------------------------------------

    class DisplayLayer {
    public:
        DisplayLayer() {}
        virtual ~DisplayLayer() {}

        /*
        ** Class configuration
        */
        void set_engine (DisplayEngine *e) { m_engine = e; }
        DisplayEngine *get_engine() const { return m_engine; }

        /*
        ** DisplayLayer interface.
        */
        virtual void draw (px::GC &gc, const WorldArea &a, int x, int y) = 0;
        virtual void draw_onepass (px::GC &/*gc*/) {}
//        virtual void notify_expose (px::Rect &area) {}
        virtual void tick (double /*dtime*/) {}
        virtual void new_world (int /*w*/, int /*h*/) {}

        // Functions.
        void mark_redraw_area (const px::Rect &r)
        {
            get_engine()->mark_redraw_area(r);
        }

    private:
        DisplayEngine *m_engine;
    };

//----------------------------------------
// Base class for all layers that contains Models
//----------------------------------------
    class ModelLayer : public DisplayLayer {
    public:
        ModelLayer() {}

        // DisplayLayer interface
        void tick (double dtime);
        void new_world (int, int);

        // Member functions
        void activate (Model *m);
        void deactivate (Model *m);
        void maybe_redraw_model(Model *m);

        virtual bool is_sprite_layer() const = 0;
    private:

        // Variables
        ModelList m_active_models;
        ModelList m_active_models_new;
    };

//----------------------------------------
// Layer for grid-aligned models
//----------------------------------------

    class DL_Grid : public ModelLayer {
    public:
        DL_Grid();
        ~DL_Grid();

        void set_model (int x, int y, Model *m);
        Model *get_model (int x, int y);
        Model *yield_model (int x, int y);

    private:
        // DisplayLayer interface.
        void new_world (int w, int h);
        void draw (px::GC &gc, const WorldArea &a, int x, int y);

        void mark_redraw (int x, int y);

        // ModelLayer interface
        virtual bool is_sprite_layer() const { return false; }

        // Variables.
        typedef px::Array2<Model*> ModelArray;
        ModelArray m_models;
    };

//----------------------------------------
// Sprite layer
//----------------------------------------

    class Sprite : public px::Nocopy {
    public:
        Model       *model;
        V2           pos;
        SpriteLayer  layer;
        bool         visible;

        Sprite (const V2 & p, SpriteLayer l, Model *m)
            : model(m), pos(p), layer(l)
        {}
        ~Sprite() { delete model; }
    };

    typedef std::vector<Sprite*> SpriteList;

    class DL_Sprites : public ModelLayer {
    public:
        DL_Sprites();
        ~DL_Sprites();

        /*
        ** DisplayLayer interface
        */
        void draw (px::GC &gc, const WorldArea &a, int x, int y);
        void draw_onepass (px::GC &gc);
        void new_world (int, int);

        /*
        ** Member functions
        */
        SpriteId add_sprite (Sprite *sprite);
        void kill_sprite (SpriteId id);
        void move_sprite (SpriteId, const px::V2& newpos);
        void replace_sprite (SpriteId id, Model *m);

        void redraw_sprite_region (SpriteId id);
        void draw_sprites (bool shades, px::GC &gc);

        Model *get_model (SpriteId id) { return sprites[id]->model; }

        void set_maxsprites (unsigned m) { maxsprites = m; }


        static const SpriteId MAGIC_SPRITEID = 1000000;
        SpriteList sprites;

    private:
        // ModelLayer interface
        bool is_sprite_layer() const { return true; }


        // Variables.
        unsigned numsprites;    // Current number of sprites
        unsigned maxsprites;    // Maximum number of sprites
    };

//----------------------------------------
// Shadow layer
//----------------------------------------

    struct StoneShadowCache;

    class DL_Shadows : public DisplayLayer {
    public:
        DL_Shadows(DL_Grid *grid, DL_Sprites *sprites);
        ~DL_Shadows();

        void new_world(int w, int h);
        void draw (px::GC &gc, int xpos, int ypos, int x, int y);

        void draw (px::GC &gc, const WorldArea &a, int x, int y);
    private:
        /*
        ** Private functions
        */
        void shadow_blit (px::Surface *scr, int x, int y,
                          px::Surface *shadows, px::Rect r);

        bool has_actor (int x, int y);

        Model * get_shadow_model(int x, int y);

        /*
        ** Variables
        */

        DL_Grid    *m_grid;     // Stone models
        DL_Sprites *m_sprites;  // Sprite models

        StoneShadowCache *m_cache;

//         px::Surface *shadow_surface;
        Uint32       shadow_ckey; // Color key
        px::Surface *buffer;
    };

//----------------------------------------
// Stone layer
//----------------------------------------

    class DL_Stones : public DL_Grid {
    public:
        DL_Stones (DL_Shadows *shadowlayer);

        void set_model (int x, int y, Model *m);
        Model *yield_model (int x, int y);
    };

//----------------------------------------
// Lines / Rubber Bands
//----------------------------------------
    struct Line {
        V2 start,end;
        V2 oldstart, oldend;

        Line(const V2 &s, const V2 &e) :start(s), end(e) {}
        Line() {}
    };


    typedef px::AssocList<unsigned, Line> LineMap;

    class DL_Lines : public DisplayLayer {
    public:
        DL_Lines() : m_id(1)
        {
        }

        void draw (px::GC &/*gc*/, const WorldArea &/*a*/, int /*x*/, int /*y*/)
        {}
        void draw_onepass (px::GC &gc);

        RubberHandle add_line (const V2 &p1, const V2 &p2);
        void set_startpoint (unsigned id, const V2 &p1);
        void set_endpoint (unsigned id, const V2 &p2);
        void kill_line (unsigned id);

    private:
        // Private methods.
        void mark_redraw_line (const Line &r);

        // Variables.
        unsigned  m_id;
        LineMap   m_rubbers;
    };

//----------------------------------------
// Parts of the display engine that are common to the game and the editor
//----------------------------------------

    class CommonDisplay {
    public:
        CommonDisplay (const ScreenArea &a);
        ~CommonDisplay();

        Model *set_model (const GridLoc &l, Model *m);
        Model *get_model (const GridLoc &l);
        Model *yield_model (const GridLoc &l);

        void set_floor (int x, int y, Model *m);
        void set_item (int x, int y, Model *m);
        void set_stone (int x, int y, Model *m);

        DisplayEngine *get_engine() const { return m_engine; }

        SpriteHandle add_effect (const V2& pos, Model *m);
        SpriteHandle add_sprite (const V2 &pos, Model *m);

        RubberHandle add_line (V2 p1, V2 p2);

        void new_world (int w, int h);
        void redraw();

    protected:
        DL_Grid    *floor_layer;
        DL_Grid    *item_layer;
        DL_Grid    *stone_layer;

        DL_Sprites *effects_layer;

        DL_Lines   *line_layer;
        DL_Sprites *sprite_layer;
        DL_Shadows *shadow_layer;

    private:

        DisplayEngine *m_engine;
    };


//----------------------------------------
// Game Display Engine
//----------------------------------------
    class Follower {
    public:
        Follower (DisplayEngine *e);
        virtual ~Follower() {}
        virtual void tick(double dtime, const px::V2 &point) = 0;
        virtual void center(const px::V2 &point);

    protected:
//         void set_engine(DisplayEngine *e) { m_engine = e; }
        DisplayEngine *get_engine() const { return m_engine; }
        bool set_offset (V2 offs);
        int get_hoff() const { return m_hoff; }
        int get_voff() const { return m_voff; }

    private:
        DisplayEngine *m_engine;
        int m_hoff, m_voff;
    };

    class Follower_Screen : public Follower {
    public:
        Follower_Screen(DisplayEngine *e);
        void tick(double dtime, const px::V2 &point);
    };

    class Follower_Scrolling : public Follower {
    public:
        Follower_Scrolling(DisplayEngine *e);
        void tick(double dtime, const px::V2 &point);
        void center(const px::V2 &point);
    private:
        bool   currently_scrolling;
        V2     curpos, destpos;
        V2     dir;
        double scrollspeed;
        double resttime;
    };

    class GameDisplay : public CommonDisplay {
    public:
        GameDisplay(const ScreenArea &a);
        ~GameDisplay();

        StatusBar * get_status_bar() const;

        void tick(double dtime);
        void new_world (int w, int h);

        void resize_game_area (int w, int h);

        /*
        ** Scrolling
        */
        void set_follow_mode (FollowMode m);
        void follow_center();
        void set_follow_sprite(SpriteId id);
        void set_reference_point (const px::V2 &point);

        // current screen coordinates of reference point
        void get_reference_point_coordinates(int *x, int *y);

        /*
        ** Screen updates
        */
        void redraw (px::Screen *scr);
        void redraw_all (px::Screen *scr);
        void draw_all (px::GC &gc);
    private:
        void set_follower (Follower *f);
        void draw_borders (px::GC &gc);

        /*
        ** Variables
        */
        Uint32         last_frame_time;
        bool           redraw_everything;
        StatusBarImpl *status_bar;

        V2          m_reference_point;
        Follower   *m_follower;

        ScreenArea inventoryarea;
    };

    class ModelHandle {
    public:
        ModelHandle ();
    };
}

#endif
