
//======================================================================
// STATUS BAR
//======================================================================
namespace display
{
    class Window {
    public:
        Window() {}
        Window (const ScreenArea &area) : m_area(area)
        {}

        const ScreenArea &get_area() const { return m_area; }
    private:
        ScreenArea m_area;
    };


    class TextDisplay {
    public:
        TextDisplay(Rect a, Font &f);

        void set_text(const string &t, TextMode m);

        void deactivate() { active = false; }
        bool is_active() const { return active; }

        void tick(double dtime);
        bool has_changed() { return changedp; }

        void draw(px::GC &gc, const Rect &r);

        TextMode get_mode() const { return mode; }
    private:
        string                  text;
        bool                    active, changedp;
        TextMode                mode;
        Rect                    area;
        double                  xoff;
        const double            scrollspeed; // pixels per second
        std::auto_ptr<Surface>  textsurface;
        px::Font               &font;
        double                  time, maxtime;
    };

    class StatusBarImpl : public StatusBar, public Window {
    public:
        StatusBarImpl (const ScreenArea &area);
        ~StatusBarImpl();

        bool has_changed() const { return m_changedp; }
        void redraw (px::GC &gc, const ScreenArea &r);
        void tick (double dtime);
        void new_world();

        // Window interface.

        // StatusBar interface.
        void set_inventory (Inventory *inv);
        void update_inventory (Inventory *inv);

        void show_text (const std::string &str, 
                        TextMode m=TEXT_SCROLLING, 
                        bool may_interrupt=false);
        void hide_text() { m_textview.deactivate(); }

        void set_counter (int new_counter) {
            if (m_showcounter_p && new_counter != m_counter) {
                m_changedp = true;
                m_counter = new_counter;
            }
        }
        void show_move_counter (bool active) {
            if (active != m_showcounter_p) {
                m_showcounter_p = active;
                m_changedp      = true;
            }
        }
        void show_odometer (bool active) {
            if (active != m_showodometer_p) {
                m_showodometer_p = active;
                m_changedp = true;
            }
        }


    private:
        ScreenArea     m_itemarea;
        ScreenArea     m_textarea;
        vector<Model*> m_models;
        bool           m_changedp;
        Inventory *    m_inventory;
        TextDisplay    m_textview;

        double m_leveltime;
        bool   m_showtime_p;
        int    m_counter;
        bool   m_showcounter_p;
        bool   m_showodometer_p;
        bool m_interruptible;   // Current text message may be interrupted
    };
}

StatusBarImpl::StatusBarImpl (const ScreenArea &area)
: Window(area),
  m_itemarea (area.x+142, area.y+area.h-64+5, area.w-147, 64-10),
  m_textarea (area.x+147, area.y+area.h-64+20, area.w-157, 32),
  m_models(),
  m_changedp(false),
  m_inventory(0),
  m_textview (m_textarea, *enigma::GetFont("dreamorp24")),
  m_leveltime (0),
  m_showtime_p (true),
  m_counter (0),
  m_showcounter_p(false),
  m_showodometer_p(false),
  m_interruptible(true)
{}

StatusBarImpl::~StatusBarImpl()
{
    px::delete_sequence(m_models.begin(), m_models.end());
    m_models.clear();
}

void
StatusBarImpl::redraw (px::GC &gc, const ScreenArea &r)
{
    ScreenArea a = get_area();
    clip(gc, intersect(a, r));

    blit(gc, a.x, a.y, enigma::GetImage ("inventory"));

    if (m_showtime_p || m_showcounter_p) {
        const int     BUFSIZE       = 8;
        char          buf[BUFSIZE];
        int           xsize_time    = 0;
        int           xsize_moves   = 0;
        Surface      *s_time        = 0;
        Surface      *s_moves       = 0;

        if (m_showtime_p) {
            Font      *f             = enigma::GetFont ("timefont");
            int        minutes       = static_cast<int>(m_leveltime/60) % 100;
            int        seconds       = static_cast<int>(m_leveltime) % 60;
            const int  SIZE_PER_CHAR = 16; // depends on fontsize (28 for timefont)

            int len    = snprintf(buf, BUFSIZE, "%d:%02d", minutes, seconds);
            s_time     = f->render(buf);
            xsize_time = static_cast<int>((len-0.5)*SIZE_PER_CHAR);
        }

        if (m_showcounter_p) {
            Font      *f             = enigma::GetFont ("smallfont");
            int        len           = snprintf(buf, BUFSIZE, "%d", m_counter);
            const int  SIZE_PER_CHAR = 8; // depends on fontsize (14 for smallfont)

            s_moves     = f->render(buf);
            xsize_moves = static_cast<int>(len*SIZE_PER_CHAR);
        }

        const int YOFF_MOVES = 26;
//         const int YOFF_MOVES = 36;
        const int YOFF_TIME  = 13;
        const int XCENTER    = 66;

        if (m_showtime_p) {
            if (m_showcounter_p) { // time + moves
                const int DISPLAY_WIDTH = 120; // est.
                int       width         = xsize_time + xsize_moves;
                int       left_width    = (DISPLAY_WIDTH*xsize_time)  / width;
                int       right_width   = (DISPLAY_WIDTH*xsize_moves) / width;

                int xoff = XCENTER + (left_width - xsize_time - DISPLAY_WIDTH)/2 - 3 ;
                blit(gc, a.x + xoff, a.y + YOFF_TIME, s_time);

                xoff = XCENTER + (DISPLAY_WIDTH-right_width-xsize_moves)/2 + 3;
                blit(gc, a.x + xoff, a.y + YOFF_MOVES, s_moves);
            }
            else { // only time
                int xoff = XCENTER - xsize_time/2;
                blit(gc, a.x + xoff, a.y + YOFF_TIME, s_time);
            }
        }
        else {                  // only moves
            int xoff = XCENTER - xsize_moves/2;
            blit(gc, a.x + xoff, a.y + YOFF_MOVES, s_moves);
        }

        delete s_moves;
        delete s_time;
    }
    else
        blit(gc, a.x+5, a.y+10, enigma::GetImage ("enigma_logo2_small"));


    if (m_textview.is_active()) {
        m_textview.draw (gc, r);
    }
    else
    {
        for (unsigned i=0; i<m_models.size(); ++i)
        {
            Model *m = m_models[i];
            m->draw(gc, m_itemarea.x + 10+i*40, m_itemarea.y + 11);
        }
    }
    m_changedp = false;
}

void
StatusBarImpl::set_inventory (Inventory *inv)
{
    m_inventory = inv;
    update_inventory (inv);
}

void
StatusBarImpl::update_inventory (Inventory *inv)
{
    if (inv && inv == m_inventory)
    {
        if (m_textview.is_active() && m_interruptible)
        {
            // Interrupt text messages only if unimportant
            hide_text();
        }


        px::delete_sequence(m_models.begin(), m_models.end());
        m_models.clear();

        for (int i=0; i<inv->size(); ++i) {
            world::Item *it = inv->get_item(i);
            m_models.push_back(MakeModel(it->get_inventory_model ()));
        }
        m_changedp = true;
    }
}

void
StatusBarImpl::show_text (const std::string &str, TextMode m, bool may_interrupt)
{
    m_textview.set_text (str, m);
    m_interruptible = may_interrupt;
}

void
StatusBarImpl::tick (double dtime)
{
    // Update text display
    if (m_textview.is_active())
    {
        m_textview.tick (dtime);
        m_changedp |= m_textview.has_changed();
    }

    double oldtime=m_leveltime;
    m_leveltime += dtime;
    if (m_showtime_p && int(m_leveltime) - int(oldtime) >= 1)
        m_changedp = true;      // update clock

    if (m_showcounter_p)
        set_counter (player::GetMoveCounter());
}

void
StatusBarImpl::new_world()
{
    m_leveltime = 0;

    delete_sequence(m_models.begin(), m_models.end());
    m_models.clear();
    m_textview.deactivate();
}



//----------------------------------------
// TextDisplay
//----------------------------------------
TextDisplay::TextDisplay(ScreenArea a, Font &f)
    : text(), active(false), changedp(false),
      mode(TEXT_SCROLLING),
      area(a), xoff(0), scrollspeed(200),
      textsurface(0), font(f)
{
    time = maxtime = 0;
}

void
TextDisplay::set_text(const string &t, TextMode m)
{
    text = t;
    mode = m;
    textsurface.reset(font.render(text.c_str()));

    time = 0;

    switch (mode) {
    case TEXT_2SECONDS: maxtime = 2.0;  break;
    case TEXT_5SECONDS: maxtime = 5.0;  break;
    default :           
        maxtime = 1e20;         // this should be close to ``indefinitely''
        break;
    }

    if (mode == TEXT_SCROLLING) {
        xoff = -area.w;
    }
    else { // all other modes are centered
        xoff = -(area.w - textsurface->width())/2;
    }

    changedp = active = true;
}

void
TextDisplay::tick(double dtime)
{
    time += dtime;
    if (time > maxtime) {
        active = false;
        changedp = true;
    }
    else {
        int oldxoff = int(xoff);
        if (mode == TEXT_SCROLLING) {
            xoff += dtime * scrollspeed;
            changedp = int(xoff) != oldxoff;
            if (xoff >= textsurface->width()) {
                active = false;
                changedp = true;
            }
        }
    }
}

void
TextDisplay::draw(px::GC &gc, const ScreenArea &r)
{
    if (active) {
        clip(gc, intersect(area, r));
        set_color(gc, 0,0,0);
        box(gc, area);
        blit(gc, area.x-int(xoff), area.y, textsurface.get());
    }
}

