/*
 * Copyright (C) 2002,2003 Daniel Heck
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: menus.cc,v 1.64.2.7 2003/10/03 22:22:59 dheck Exp $
 */
#include "video.hh"
#include "menus.hh"
#include "editor.hh"
#include "enigma.hh"
#include "options.hh"
#include "px/px.hh"
#include "sound.hh"
#include "config.h"
#include "help.hh"
#include "display.hh"
#include "oxyd.hh"

#include <map>
#include <cassert>

using namespace enigma;
using namespace options;
using namespace gui;
using namespace px;
using namespace OxydLib;
using namespace std;


#include "menus_internal.hh"

//----------------------------------------------------------------------
// Level pack menu 
//----------------------------------------------------------------------

LevelPackMenu::LevelPackMenu()
: m_selection(-1)
{
    VTableBuilder builder(this, Rect(0,0,150,40), 5,5, 4);

    for (unsigned i=0; i<enigma::LevelPacks.size(); ++i) {
        LevelPack *lp = enigma::LevelPacks[i];
        buttons.push_back(builder.add(new TextButton(lp->get_name(), this)));
    }
    builder.finish();
}

void
LevelPackMenu::on_action(Widget *w)
{
    for (unsigned i=0; i<buttons.size(); ++i)
        if (buttons[i]==w) {
            m_selection = i;
            Menu::quit();
        }
}

void
LevelPackMenu::draw_background(px::GC &gc) 
{
    video::SetCaption("Enigma - Level Pack Menu");
//     sound::PlayMusic("sound/menu.s3m");

    blit(gc, 0,0, enigma::GetImage("menu_bg"));
}


//----------------------------------------------------------------------
// LevelWidget implementation
//----------------------------------------------------------------------
LevelWidget::LevelWidget(LevelPack *lp, int w, int h)
: level_pack(lp),
  ifirst(0), iselected(0),
  width(w), height(h),
  m_areas(),
  listener(0)
{}

LevelMenu *
LevelWidget::get_menu()
{
    if (Menu *m = Widget::get_menu())
        return dynamic_cast<LevelMenu*>(m);
    else {
        assert(0);
        return 0;
    }
}

void
LevelWidget::show_text(const string& text)
{
    get_menu()->show_text(text);
}

void
LevelWidget::scroll_up(int nlines)
{
    for (; nlines; --nlines) {
        if (ifirst+width*height >= (int)level_pack->size())
            break;
        ifirst += width;
        if (iselected < ifirst)
            iselected += width;
    }
    redraw();
}

void
LevelWidget::scroll_down(int nlines)
{
    for (; nlines; --nlines) {
        if (ifirst < width)
            break;
        ifirst -= width;
        if (iselected >= ifirst+width*height)
            iselected -= width;
    }
    redraw();
}

void
LevelWidget::page_up()
{
    set_selected (ifirst - width*height, iselected - width*height);
}
void
LevelWidget::page_down()
{
    set_selected (ifirst + width*height, iselected + width*height);
}

void
LevelWidget::recalc_available()
{
    max_available = HighestAvailableLevel(level_pack);
}

void
LevelWidget::change_levelpack (LevelPack *lp)
{
    iselected  = 0;
    ifirst     = 0;
    level_pack = lp;
    oxyd::ChangeSoundset(options::SoundSet, level_pack->get_default_SoundSet());
    cache.clear();
    preview_cache.clear();
    recalc_available();
    redraw();
    sound::PlaySound("menumove");
}

void
LevelWidget::next_unsolved()
{
    unsigned next = NextLevel(level_pack, iselected, max_available, true, false);

    if (next)
        set_current(next);
    else
        show_text("No further unsolved level available!");
}

void
LevelWidget::draw (px::GC &gc, const px::Rect &r)
{
    const int imgw = 120;       // Size of the preview images
    const int imgh = 78;
    preview_cache.set_size(imgw, imgh);

    const int buttonw = imgw+20;
    const int buttonh = imgh+35;

    const int hgap = Max(0, (get_w() - width*buttonw) / (width-1));
    const int vgap = Max(0, (get_h() - height*buttonh)/ (height-1));

    Font    *smallfnt = enigma::GetFont("levelmenu");

    Surface *img_easy        = enigma::GetImage("completed-easy");
    Surface *img_hard        = enigma::GetImage("completed");
    Surface *img_changed     = enigma::GetImage("changed");
    Surface *img_error       = enigma::GetImage("error");
    Surface *img_unavailable = enigma::GetImage("unavailable");
    Surface *img_unknown     = enigma::GetImage("unknown");

    unsigned i=ifirst;          // level index
    for (int y=0; y<height; y++)
    {
        for (int x=0; x<width; x++, i++)
        {
            if (i >= level_pack->size())
                goto done_painting;

            int xpos = get_x() + x*(buttonw + hgap);
            int ypos = get_y() + y*(buttonh + vgap);

            Rect buttonarea(xpos, ypos, buttonw, buttonh);
            if (r.overlaps(buttonarea)    // This area is requested
                || r.w == 0)    // repaint whole screen
            {
                if( (i-ifirst) >= m_areas.size())
                    m_areas.push_back(buttonarea);
                else
                    m_areas[(i-ifirst)] = buttonarea;

                // Tint background of selected button
                if (i == (unsigned) iselected) {
                    px::TintRect (video::GetScreen()->get_surface(),
                                  intersect(buttonarea, r),
                                  120,120,120,120);
                    set_color(gc, 200,200,200);
                    frame(gc, buttonarea);
                }

                const LevelInfo *levelinfo = level_pack->get_info(i);

                // Draw level preview
                Surface *img = 0;
                if (level_pack->may_have_previews()) {
                    string fname = string("levels/") + levelinfo->filename + ".png";
                    img          = cache.get(enigma::FindDataFile(fname));
                }
                if (!img && !Nozoom) {
                    img     = preview_cache.getPreview(level_pack, i);
                    if (!img)
                        img = img_error;
                }


                int imgx = xpos + 10;
                int imgy = ypos + 10;
                if (img)
                    blit (gc, imgx, imgy, img);
                else
                    blit (gc, imgx, imgy, img_unknown);

                int  finished      = 0;
                bool level_changed = false;
                if (LevelStatus *ls = GetLevelStatus(level_pack->get_name(), levelinfo->filename)) {
                    finished = ls->finished;
                    if (finished)
                        level_changed = level_pack->get_modtime(i) > ls->solved_at;
                }

                // Shade unavailable levels
                if (LevelIsLocked(level_pack, i))
                    blit (gc, imgx, imgy, img_unavailable);

                // Draw solved/changed icons on top of level preview
                if (finished)
                    blit (gc, imgx, imgy, (finished&DIFFICULTY_HARD) ? img_hard : img_easy);

                // Add warning sign if level has been changed since player solved it
                if (level_changed)
                    blit (gc, imgx-3, imgy-3, img_changed);

                // Draw level name
                const char *caption = levelinfo->name.c_str();
                smallfnt->render (gc,
                                  xpos + (buttonw-smallfnt->get_width(caption))/2,
                                  imgy + imgh + 1,
                                  caption);
            }
        }
    }
  done_painting:
    m_areas.resize (i-ifirst); // Remove unused areas (if any) from the list
    return;
}

void
LevelWidget::set_selected (int newfirst, int newsel)
{
    int numlevels = static_cast<int>(level_pack->size());
    newsel = Clamp(newsel, 0, numlevels-1);

    if (newsel < newfirst)
        newfirst = (newsel/width)*width;
    if (newsel >= newfirst+width*height)
        newfirst = (newsel/width-2)*width;

    newfirst = Clamp(newfirst, 0, numlevels-1);

    int oldsel = iselected;
    if (newfirst != ifirst) {
        ifirst    = newfirst;
        iselected = newsel;

        if (!m_areas.empty()) {
            sound::PlaySound("menumove");
            if (oldsel != newsel) sound::PlaySound("menuswitch");
            redraw();
        }
    }
    else if (newsel != iselected) {
        iselected = newsel;

        if (!m_areas.empty()) {
            sound::PlaySound("menuswitch");
            redraw (m_areas[oldsel-ifirst]); // old selection
            redraw (m_areas[iselected-ifirst]); // new selection
        }
    }
}

bool
LevelWidget::on_event(const SDL_Event &e)
{
    bool h=false;

    switch (e.type) {
    case SDL_MOUSEMOTION:
        if (get_area().contains(e.motion.x, e.motion.y)) {
            int newsel=iselected;
            for (unsigned i=0; i<m_areas.size(); ++i)
                if (m_areas[i].contains(e.motion.x, e.motion.y))
                {
                    newsel = ifirst+i;
                    break;
                }
            set_current(newsel);
            h=true;
        }
        break;
    case SDL_MOUSEBUTTONDOWN:
        if (get_area().contains(e.button.x, e.button.y))
            h=handle_mousedown (&e);
        break;
    case SDL_KEYDOWN:
        h= handle_keydown (&e);
        break;
    }
    return h;
}

bool
LevelWidget::handle_mousedown (const SDL_Event *e)
{
    switch (e->button.button) {
    case SDL_BUTTON_LEFT:
        for (unsigned i=0; i<m_areas.size(); ++i)
            if (m_areas[i].contains(e->button.x, e->button.y))
            {
                sound::PlaySound("menuok");
                iselected = ifirst+i;
                trigger_action();
                return true;
            }
        break;
    case SDL_BUTTON_RIGHT:
        break;
    case 4: scroll_down(1); return true;
    case 5: scroll_up(1); return true;
    }
    return false;
}

static const char *helptext_levelmenu[] = {
    "Escape:",                  "Skip to main menu",
    "F1:",                      "Show this help",
    "F5:",                      "Select next unsolved level",
    "Arrows:",                  "Select level",
    "Return:",                  "Play selected level",
    "Back/Space:",              "Previous/next levelpack",
    "Alt+Return:",              "Switch between fullscreen and window",
    0
};

bool
LevelWidget::handle_keydown (const SDL_Event *e)
{
    switch (e->key.keysym.sym) {
    case SDLK_l:                // Reload level index
//        level_pack.init();
        redraw();
        break;

    case SDLK_o:                // Show/hide "todo" levels
        break;

    case SDLK_u: { // set state to "unsolved"
        const LevelInfo *info   = level_pack->get_info(iselected);
        LevelStatus *status = GetLevelStatus(level_pack->get_name(), info->filename);

        if (status && status->finished) {
            status->finished  = 0;
            redraw();
        }
        break;
    }
    case SDLK_LEFT:  set_current (iselected-1); break;
    case SDLK_RIGHT: set_current (iselected+1); break;
    case SDLK_DOWN:  set_current (iselected+width); break;
    case SDLK_UP:    set_current (iselected-width); break;
    case SDLK_PAGEDOWN: page_down(); break;
    case SDLK_PAGEUP: page_up(); break;
    case SDLK_HOME: start(); break;
    case SDLK_END: end(); break;

    case SDLK_F1: {
        px::Screen *screen = video::GetScreen();
        displayHelp(screen, helptext_levelmenu, 200);
        get_menu()->draw_all(screen);
        break;
    }
    case SDLK_F5:
        next_unsolved();
        break;

    case SDLK_SPACE: {
        LevelMenu *lmenu = get_menu();
        if (lmenu) {
            lmenu->next_levelpack();
        }
        break;
    }
    case SDLK_BACKSPACE: {
        LevelMenu *lmenu = get_menu();
        if (lmenu) {
            lmenu->previous_levelpack();
        }
        break;
    }
    case SDLK_RETURN:
        trigger_action();
        break;

    default:
        return false;           // key not handled
    }
    return true;
}


//----------------------------------------
// ValueButton implementation
//----------------------------------------

bool 
ValueButton::update_value(int old_value, int new_value) 
{
    if (new_value<min_value) new_value      = min_value;
    else if (new_value>max_value) new_value = max_value;

    if (new_value != old_value) {
        set_value(new_value);
        set_text(build_text(new_value));
        return true;
    }
    return false;
}

bool 
ValueButton::on_event(const SDL_Event &e) 
{
    // handles button movement and
    bool handled = PushButton::on_event(e);

    switch (e.type) {
    case SDL_KEYDOWN:
        bool keyhandled   = true;
        bool changed = false;

        switch (e.key.keysym.sym) {
        case SDLK_LEFT:  changed = inc_value(-1); break;
        case SDLK_RIGHT: changed = inc_value(1); break;
        default :
            keyhandled = false;
            break;
        }

        if (keyhandled) {
            handled = true;
            sound::PlaySound(changed ? "menuswitch" : "menustop");
        }

        break;
    }

    return handled;
}

void 
ValueButton::on_action(Widget */*w*/) 
{
    if (!inc_value(1))
        update_value(get_value(), min_value);
}


//----------------------------------------------------------------------
// Various buttons for the options menu
//----------------------------------------------------------------------

namespace
{
    class MouseSpeedButton : public ValueButton {
        int get_value() const     { return int(MouseSpeed+.5); }
        void set_value(int value) { MouseSpeed = value; }

        string build_text(int value) const  {
            char msg[50];
            sprintf(msg, "Mouse speed: %d", value);
            return msg;
        }
    public:
        MouseSpeedButton()
        : ValueButton("", minMouseSpeed, maxMouseSpeed)
        { update(); }
    };

    class SoundVolumeButton : public ValueButton {
        int get_value() const     { return int(0.5+SoundVolume*10.0); }
        void set_value(int value) {
            SoundVolume = value/10.0;
            sound::UpdateVolume();
        }

        string build_text(int value) const {
            if (value == 0) {
                return "Sound off";
            }
            else {
                char msg[50];
                sprintf(msg, "Sound vol: %d", value);
                return msg;
            }
        }
    public:
        SoundVolumeButton() : ValueButton("", 0, 10) { update(); }
    };

    class MusicVolumeButton : public ValueButton {
        int get_value() const     { return int(0.5+MusicVolume*10.0); }
        void set_value(int value) {
            MusicVolume = value/10.0;
            sound::UpdateVolume();
        }

        string build_text(int value) const {
            if (value == 0) {
                return "Music off";
            }
            else {
                char msg[50];
                sprintf(msg, "Music vol: %d", value);
                return msg;
            }
        }
    public:
        MusicVolumeButton() : ValueButton("", 0, 10) { update(); }
    };

    class StereoButton : public TextButton {
        void update() {
            if (StereoSeparation == 0)
                set_text("Mono");
            else if (StereoSeparation > 0)
                set_text("Stereo l-r");
            else
                set_text("Stereo r-l");
        }
        // ActionListener interface.
        void on_action(Widget *) {
            if (StereoSeparation == 0)
                StereoSeparation = +10.0;
            else if (StereoSeparation > 0)
                StereoSeparation = -10.0;
            else
                StereoSeparation = 0;
            update();
        }
    public:
        StereoButton() : TextButton("", this) { update(); }
    };

    class InGameMusicButton : public TextButton {
        void update() {
            if (InGameMusic)
                set_text("Music in game");
            else
                set_text("No music in game");
        }
        void on_action(Widget *) {
            InGameMusic = ! InGameMusic;
            if( InGameMusic)
                sound::PlayMusic( LevelMusicFile.c_str());
	    else
		sound::StopMusic( LevelMusicFile.c_str());
            update();
        }
    public:
        InGameMusicButton() : TextButton("", this) { update(); }
    };

    class SoundSetButton : public TextButton {
        void update() {
            string sound_set;
            switch (SoundSet) {
            case 0: sound_set = "Default"; break;
            case 1: sound_set = "Enigma"; break;
            default :
                switch (OxydVersion(SoundSet-2)) {
                case OxydVersion_Oxyd1:          sound_set = "Oxyd"; break;
                case OxydVersion_OxydMagnum:     sound_set = "Magnum"; break;
                case OxydVersion_OxydMagnumGold: sound_set = "Mag.Gold"; break;
                case OxydVersion_OxydExtra:      sound_set = "Extra"; break;
                case OxydVersion_PerOxyd:        sound_set = "Per.Oxyd"; break;
                default :
                    fprintf(stderr, "Invalid soundset %i\n", SoundSet);
                    break;
                }
                break;
            }

            sound_set += " sound";
            set_text(sound_set);
        }
        void on_action(Widget *) {
            ++SoundSet;
            if (SoundSet >= 2) { // test if oxyd pack is installed
                while (1) {
                    if ((SoundSet-2) > OxydVersion_Last) {
                        SoundSet = 0;
                        break;
                    }
                    if (oxyd::FoundOxyd(OxydVersion(SoundSet-2))) {
                        break;
                    }
                    ++SoundSet;
                }
            }
            oxyd::ChangeSoundset(SoundSet, -1);
            update();
        }
    public:
        SoundSetButton() : TextButton("", this) { update(); }
    };

    class SkipSolvedButton : public TextButton {
        void update() {
            if (SkipSolvedLevels)
                set_text("Skip solved levels");
            else
                set_text("Play all levels");
        }
        void on_action(Widget *) {
            SkipSolvedLevels = ! SkipSolvedLevels;
            update();
        }
    public:
        SkipSolvedButton() : TextButton("", this) { update(); }
    };

    class VideoModeButton : public TextButton {
        video::VideoModes get_mode() {
            int mode = Clamp(options::VideoMode, 0, int(video::VM_COUNT));
            return static_cast<video::VideoModes>(mode);
        }
        void update() {
            string txt ("Video: ");
            txt += video::GetInfo(get_mode())->name;
            set_text(txt);
        }
        void on_action(Widget *) {
            int mode = get_mode() + 1;
            if (mode >= video::VM_COUNT) mode = 0;
            options::VideoMode = mode;
            options::MustRestart = true;
            update();
        }
    public:
        VideoModeButton() : TextButton("", this) { update(); }
    };
}


FullscreenButton::FullscreenButton() : TextButton("", this) 
{
    update(); 
}

void FullscreenButton::on_action(Widget *) 
{
    bool old = FullScreen;
    FullScreen = video::ToggleFullscreen();
    if (old != FullScreen)
        update();
    else
        options::MustRestart = true;
}


void FullscreenButton::update() 
{
    if (FullScreen > 0)
        set_text("Fullscreen");
    else
        set_text("Window");
}

DifficultyButton::DifficultyButton() : TextButton("", this) 
{
    update(); 
}

void DifficultyButton::update() 
{
    if (Difficulty == DIFFICULTY_EASY)
        set_text("Difficulty: Easy");
    else
        set_text("Difficulty: Normal");
}

void DifficultyButton::on_action(Widget *) 
{
    Difficulty = (DIFFICULTY_EASY+DIFFICULTY_HARD)-Difficulty;
    options::MustRestartLevel = true;
    update();
}



//----------------------------------------------------------------------
// Options menu
//----------------------------------------------------------------------
OptionsMenu::OptionsMenu(px::Surface *background_)
: back(new TextButton("Back", this)),
  fullscreen(new FullscreenButton),
  m_restartinfo (new Label("")),
  background(background_)
{
    const int spacing     = 5;
    const int big_spacing = 60;
    const int but_width   = 150;
    const int but_height  = 40;

    BuildVList left (this, Rect(0, 0, but_width, but_height), spacing);
    BuildVList right(this, Rect(but_width+big_spacing, 0, but_width, but_height), spacing);

    left.add (fullscreen);
    left.add (new MouseSpeedButton);
    left.add (new SkipSolvedButton);
//    left.add (new DifficultyButton);
    left.add (new VideoModeButton);

    right.add (new SoundVolumeButton);
    right.add (new SoundSetButton);
    right.add (new MusicVolumeButton);
//    right.add (new InGameMusicButton);
    right.add (new StereoButton);

    {
        Rect l = left.pos();
        Rect r = right.pos();

        add (m_restartinfo, Rect (l.x, l.y + 15, 400, 20));
        m_restartinfo->set_alignment (HALIGN_LEFT);
        update_info();

        l.x = (l.x+r.x)/2;
        l.y = Max(l.y, r.y)+big_spacing;

        add(back, l);
    }
}

void OptionsMenu::update_info()
{
    if (options::MustRestart)
        m_restartinfo->set_text ("Please restart Enigma to activate your changes!");
    else
        m_restartinfo->set_text ("");
}

bool OptionsMenu::on_event (const SDL_Event &e)
{
    bool handled=false;
    if (e.type == SDL_MOUSEBUTTONDOWN
        && e.button.button == SDL_BUTTON_RIGHT)
    {
        Menu::quit();
        handled = true;
    }
    else if (e.type == SDL_KEYDOWN) {
        if ((e.key.keysym.sym==SDLK_RETURN) &&
            (e.key.keysym.mod & KMOD_ALT))
        {
            // update state of FullscreenButton :
            dynamic_cast<FullscreenButton*>(fullscreen)->update();
            handled = true;
        }
    }
    return handled;
}

void OptionsMenu::on_action(Widget *w)
{
    if (w == back)
        Menu::quit();
}

void OptionsMenu::tick (double)
{
    update_info();
}

void OptionsMenu::draw_background(px::GC &gc)
{
    video::SetCaption("Enigma - Options Menu");
//     blit(gc, 0,0, enigma::GetImage("menu_bg"));
    blit(gc, 0,0, background);
}


//----------------------------------------------------------------------
// Level menu
//----------------------------------------------------------------------
LevelMenu::LevelMenu(LevelPack *lp, unsigned long pos)
: but_back       (new TextButton("Back", this)),
  but_difficulty (new DifficultyButton),
  but_levelpack  (new TextButton("Level Pack", this)),
  lbl_lpinfo     (new Label("")),
  lbl_statistics (new Label("")),
  lbl_levelname  (new Label("")),
  lbl_levelinfo  (new Label("")),
  levelwidget (new LevelWidget(lp, 4, 3)),
  level_pack  (lp),
  m_ilevelpack ()
{
    const int
        BX=140,                 // button xsize
        BY=35,                  // button ysize
        Y1=60,                  // y position for level preview
        Y2=10,                  // y position for information area
        Y3=430;                 // y position for bottom button row
    {
        pgup     = new ImageButton("ic-up", "ic-up1", this);
        pgdown   = new ImageButton("ic-down", "ic-down1", this);
        start    = new ImageButton("ic-top", "ic-top1", this);
        end      = new ImageButton("ic-bottom", "ic-bottom1", this);
        but_unsolved = new ImageButton("ic-unsolved", "ic-unsolved1", this);
        // new TextButton("Timer on")

        BuildHList hlist1(this, Rect(10, Y3, BX, BY), 10);

        hlist1.add (but_levelpack);
        hlist1.add (but_unsolved);
        hlist1.add (but_difficulty);
//        hlist1.add (newgame);
        hlist1.add (but_back);
    }

    {
        Rect r(10+590+10, Y1, 20, 50);
        r.y = Y1;
        add (pgup, r);
        r.y += 60;
        add (pgdown, r);
        r.y = Y1+240;
        add (start, r);
        r.y += 60;
        add (end, r);
    }

    add (lbl_lpinfo, Rect (325,Y2,305,28));
    lbl_lpinfo->set_alignment (HALIGN_RIGHT, VALIGN_TOP);
    add (lbl_statistics, Rect (325, Y2+20, 305, 28));
    lbl_statistics->set_alignment (HALIGN_RIGHT, VALIGN_TOP);
    
    add (lbl_levelname,  Rect (10,Y2,305, 28));
    lbl_levelname->set_alignment (HALIGN_LEFT, VALIGN_TOP);
    add (lbl_levelinfo,  Rect (10,Y2+20,305, 28));
    lbl_levelinfo->set_alignment (HALIGN_LEFT, VALIGN_TOP);
    
    add (levelwidget,  Rect (10,Y1,590,350));
    levelwidget->set_listener(this);

    set_position(pos);
}

void LevelMenu::tick(double dtime)
{
    // info texts disappear after some time
    if (shown_text_ttl>0.0) {
        shown_text_ttl -= dtime;
        if (shown_text_ttl <= 0.0)
            shown_text = "";
    }
    update_info();
}

bool
LevelMenu::on_event (const SDL_Event &e)
{
    // Pass all events to the level widget first
    bool handled=levelwidget->on_event(e);

    if (!handled) {
        if (e.type == SDL_MOUSEBUTTONDOWN
            && e.button.button == SDL_BUTTON_RIGHT)
        {
            Menu::quit();
            handled=true;
        }
        else
            handled = Menu::on_event (e);
    }
    return handled;
}

void
LevelMenu::on_action(Widget *w)
{
    if (w==levelwidget) {
        int ilevel = levelwidget->selected_level();
        LevelPack *lp = enigma::LevelPacks[m_ilevelpack];

        if ((unsigned)ilevel < lp->size()) {
            if (!LevelIsLocked (lp, ilevel)) 
            {
                ilevel = StartGame(lp, ilevel);
                invalidate_all();
                levelwidget->set_current(ilevel);
            }
            else
                show_text("You are not allowed to play this level yet.");
        }
    } else if (w == but_back) {
        Menu::quit();
    } else if (w == pgup) {
        levelwidget->page_up();
    } else if (w == pgdown) {
        levelwidget->page_down();
    } else if (w == start) {
        levelwidget->start();
    } else if (w == end) {
        levelwidget->end();
    } else if (w == but_unsolved) {
        levelwidget->next_unsolved();
    } else if (w == but_levelpack) {
        LevelPackMenu lpm;
        lpm.center(video::GetScreen());
        if (lpm.manage(video::GetScreen())) {
            set_levelpack(lpm.get_selection());
        }
        invalidate_all();
//        set_levelpack ((m_ilevelpack+1) % enigma::LevelPacks.size());
    }
}

void
LevelMenu::update_info()
{
    char txt[200];
    enigma::LevelPack *lp        = enigma::LevelPacks[m_ilevelpack];

    // Display levelpack statistics (percentage of solved levels)
    {
	int numsolved = enigma::CountSolvedLevels (lp);
	int pct = 0;
	if (lp->size() > 0) {
	    pct = 100*numsolved / lp->size();
	}
	snprintf (txt, sizeof(txt), "%d%% solved", pct);
	lbl_statistics->set_text(txt);
    }
    
    int iselected = levelwidget->selected_level();
    if (const enigma::LevelInfo *li = lp->get_info (iselected)) 
    {
	// Display level name
        if (options::WizardMode) {
            snprintf (txt, sizeof(txt), "#%d: %s (%s)",
                      iselected+1, li->name.c_str(), li->filename.c_str());
        } else {
            snprintf (txt, sizeof(txt), "#%d: %s",
                      iselected+1, li->name.c_str());
        }
        lbl_levelname->set_text(txt);

	// Display best time
        if (shown_text.length()) {
            lbl_levelinfo->set_text(shown_text);
        }
        else {
            int par_time = li->best_time;
            if (LevelStatus *ls= options::GetLevelStatus (lp->get_name(), li->filename)) {
                if (options::Difficulty == DIFFICULTY_HARD) {
                    if (ls->par_hard > 0)
                        par_time = ls->par_hard;
                } else {
                    if (ls->par_easy > 0)
                        par_time = ls->par_easy;
                }
            }
            snprintf (txt, sizeof(txt), "Time to beat: %d:%02d",
                      par_time/60, par_time % 60);

            lbl_levelinfo->set_text(txt);
        }
    }
}

void
LevelMenu::set_levelpack (unsigned index)
{
    if (index < enigma::LevelPacks.size()) {
        m_ilevelpack = index;

        enigma::LevelPack *lp = enigma::LevelPacks[m_ilevelpack];
        levelwidget->change_levelpack (lp);

        char txt[100];
        snprintf (txt, sizeof(txt), "%s: %d levels",
                  lp->get_name().c_str(), lp->size());
        lbl_lpinfo->set_text(txt);

        update_info();
    }
}


void
LevelMenu::draw_background(px::GC &gc)
{
    video::SetCaption("Enigma - Level Menu");
    sound::PlayMusic( MenuMusicFile.c_str());

    blit(gc, 0,0, enigma::GetImage("menu_bg"));
}


//----------------------------------------------------------------------
// Main menu
//----------------------------------------------------------------------

MainMenu::MainMenu()
{
    BuildVList b(this, Rect((640-150)/2,150,150,40), 5);
    m_startgame = b.add(new TextButton("Start Game", this));
//    leveled     = b.add(new TextButton("Editor", this));
    manual      = b.add(new TextButton("Manual", this));
    options     = b.add(new TextButton("Options", this));
    credits     = b.add(new TextButton("Credits", this));
    quit        = b.add(new TextButton("Quit", this));
}

void
MainMenu::draw_background(px::GC &gc)
{
    video::SetCaption("Enigma - Main Menu");
    sound::PlayMusic( MenuMusicFile.c_str());

    blit(gc, 0,0, enigma::GetImage("menu_bg"));

    Font *f = enigma::GetFont("levelmenu");
    Surface * logo(enigma::GetImage("enigma_logo3"));
    int x0=(640-logo->width())/2;
    int y0=30;
    blit(gc, x0, y0, logo);
    f->render (gc, 5, 460, "v" PACKAGE_VERSION);
}

void
MainMenu::on_action(Widget *w)
{
    if (w == m_startgame) {
        LevelMenu m(enigma::LevelPacks[0], options::LevelMenuPosition);
        m.manage(video::GetScreen());
        options::LevelMenuPosition = m.get_position();
        invalidate_all();
    }
    else if (w == manual) {
        show_help ();
    } else if (w == credits) {
        show_credits ();
    } else if (w == options) {
        GUI_OptionsMenu(enigma::GetImage("menu_bg"));
    } else if (w == leveled) {
        editor::Run();
    } else if (w == quit) {
        Menu::quit();
    } else
        return;
    invalidate_all();
}

void
MainMenu::show_text( const char *text[])
{
    Screen *scr = video::GetScreen ();
    GC gc (scr->get_surface());
    blit (gc, 0,0, enigma::GetImage("menu_bg"));


    Font *f = enigma::GetFont("menufont");
    for (int i=0; text[i]; ++i)
    {
        f->render (gc, 40, 20+i*f->get_height(), text[i]);
    }
    scr->update_all ();
    scr->flush_updates();

    SDL_Event e;
    for (;;) {
        SDL_WaitEvent(&e);
        if (e.type == SDL_KEYDOWN || e.type == SDL_MOUSEBUTTONDOWN)
            break;
    }
}

void
MainMenu::show_credits ()
{
    static const char *credit_text[] = {
	"Project maintainer:",
	"  DANIEL HECK",
	"",
        "Main developers:",
        "  SIEGFRIED FENNIG, MARTIN HAWLISCH, DANIEL HECK,",
        "  PETR MACHATA, RALF WESTRAM",
        "",
        "Additional Level Design:",
        "  NAT PRYCE, JACOB SCOTT, SVEN SIGGELKOW, ANDRZEJ SZOMBIERSKI",
        "",
        "Special Thanks To:",
        "  JOHANNES FORTMANN (graphics), JEREMY SAWICKI (oxydlib),",
        "  ANDREW \"NECROS\" SEGA (menu music)",
	"",
	"Please see the manual for more detailed credits.",
        "",
	"Home Page: http://www.nongnu.org/enigma",
	"Contact: enigma-devel@nongnu.org",
        "",
        "Enigma is free software and may be distributed under the",
        "terms of the GNU General Public License, version 2.  See",
        "the accompanying COPYING.GPL for details.",
        0
    };

    show_text(credit_text);
}

void
MainMenu::show_help ()
{
    static const char *screen1[] = {
        "Introduction:",
        "",
        "The idea behind Enigma is simple: In most levels your job is to find",
        "pairs of \"Oxyd\" stones (you will recognize them when you see them)",
        "with matching colors. You have to open all oxyd stones but they only",
        "stay open when opening two stones of the same color one after",
        "another. Just play the first levels in the \"Oxyd & Co\" group and you",
        "will get the idea.",
        "In some other levels, called \"meditation landscapes\" you have a",
        "different job: You control a couple of small white marbles",
        "simultaneously and have to put each of them into a pit on the floor.",
        "",
        "Moving around:",
        "",
        "You control the marble by moving around the mouse into the desired",
        "direction. But be careful, because the marble has some mass and",
        "the floor some friction, the marble needs some time to accelerate or",
        "stop.",
        "Larger levels scroll when you reach the outermost part of the visible",
        "part on the screen.",
        "",
        0
    };
    static const char *screen2[] = {
        "The Floor:",
        "",
        "On most types of floor you can move around riskless but the speed",
        "of your marble may vary.",
        "When moving into abyss or water you will die.",
        "Ice is very slippery.",
        "You cannot control your marble in space.",
        "",
        "Items and Inventory:",
        "",
        "In many levels you can see different items. You can pick them up",
        "by simply rolling over them. The items are then stored in your",
        "inventory, which you control using the left and right mouse button.",
        "The leftmost item will be activated when pressing the left mouse",
        "button and you can rotate the items using the right mouse button.",
        "Some items can mutate when hit by a laser or crushed by a stone.",
        "",
        0
    };
    static const char *screen3[] = {
        "Stones:",
        "",
        "Most stones  are nothing more than walls.",
        "Many stones are movable when hit strong enough.",
        "The wooden stone will build new floor if moved into water, space or",
        "abyss.",
        "Others can be destroyed using a hammer, dynamite or a laser.",
        "Doors can be opened using a switch or trigger hidden somewhere",
        "around in the level.",
        "Some magic stones can be changed when hit using a magic wand.",
        "And some depend on the color of your marble.",
        "",
        0
    };

    show_text( screen1);
    show_text( screen2);
    show_text( screen3);
}

//----------------------------------------------------------------------
// Functions
//----------------------------------------------------------------------

void
enigma::GUI_MainMenu(LevelPack */*lp*/, unsigned /*ilevel_pack*/)
{
    Screen *scr = video::GetScreen();
    MainMenu m;
    m.manage(scr);
}

void
enigma::GUI_OptionsMenu(Surface *background)
{
    Screen *scr = video::GetScreen();
    OptionsMenu m(background);
    m.center(scr);
    m.manage(scr);
}
