/*
 * Copyright (C) 2002,2003 Daniel Heck
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *  
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: d_models.cc,v 1.21 2004/05/27 20:30:42 dheck Exp $
 */

#include "display_internal.hh"
#include "d_models.hh"

#include "enigma.hh"
#include "lua.hh"
#include "options.hh"
#include "display.hh"
#include "d_engine.hh"
#include "enigma.hh"
#include "video.hh"
#include "main.hh"

#include "px/px.hh"
#include "px/cache.hh"

#include "SDL_image.h"

#include <cstdio>
#include <string>
#include <list>
#include <iostream>

using namespace enigma;
using namespace display;
using namespace std;
using namespace px;


extern "C" {
#include "lualib.h"
#include "tolua.h"
}
#include "global-lua.hh"
#include "display-lua.hh"
#include "enigma-lua.hh"
#include "px-lua.hh"

/* -------------------- Types -------------------- */

namespace
{
    class SurfaceCache_Alpha : public PtrCache<Surface>{
    public:
        Surface *acquire(const std::string &name);
    };

    class SurfaceCache : public PtrCache<Surface>{
    public:
        Surface *acquire(const std::string &name);
    };

    class ModelManager {
    public:
        ModelManager();
        ~ModelManager();

        void define (const std::string name, Model *m);

        /* Create new model of type `name'.  Returns 0 if no such
           model exists. */
        Model *create (const std::string &name);
        
        /* Remove model definition for `name'. */
        void remove (const std::string &name);

        bool has_model (const std::string &name) const 
        { return m_templates.has_key (name); }
        
        size_t num_templates() const 
        { return m_templates.size(); }

    private:
        // Variables
        typedef px::Dict<Model*> ModelMap;
        ModelMap m_templates;
    };
}


/* -------------------- SurfaceCache -------------------- */

Surface *SurfaceCache_Alpha::acquire(const std::string &name) 
{
    string filename;
    if (file::FindImageFile (name + ".png", filename))
        return px::LoadImage(filename.c_str());
    else
        return 0;
}


Surface *SurfaceCache::acquire(const std::string &name) 
{
    string filename;
    if (file::FindImageFile (name + ".png", filename)) {
        SDL_Surface *s = IMG_Load(filename.c_str());
        if (s) {
            SDL_Surface *img = 0;
            if (s->flags & SDL_SRCALPHA) {
                img = SDL_DisplayFormatAlpha(s);
            } else {
                SDL_SetColorKey(s, SDL_SRCCOLORKEY, //|SDL_RLEACCEL, 
                                SDL_MapRGB(s->format, 255,0,255));
                img = SDL_DisplayFormat(s);
            }
            if (img) {
                SDL_FreeSurface(s);
                return Surface::make_surface(img);
            }
            return Surface::make_surface(s);
        }
    }
    return 0;
}



/* -------------------- Variables -------------------- */

namespace
{
    SurfaceCache       surface_cache;
    SurfaceCache_Alpha surface_cache_alpha;
    ModelManager *modelmgr = 0;

    vector<Surface *>  image_pile;

    string  anim_templ_name;
    Anim2d *anim_templ = 0;
}



/* -------------------- ModelManager -------------------- */

ModelManager::ModelManager()
: m_templates (1069)
{}

ModelManager::~ModelManager() {
    delete_map (m_templates.begin(), m_templates.end());
    delete_sequence (image_pile.begin(), image_pile.end());
    image_pile.clear();
}

void ModelManager::define (const std::string name, Model *m) {
    m_templates.insert (name, m);
}

Model * ModelManager::create (const std::string &name) {
    ModelMap::iterator i = m_templates.find(name);

    if (i != m_templates.end())
        return i->second->clone();
    else 
        return 0;
}

void ModelManager::remove (const std::string &name) 
{
    ModelMap::iterator i = m_templates.find(name);
    if (i != m_templates.end()) {
    	delete i->second;
    	m_templates.remove (name);
    }
}


/* -------------------- Functions -------------------- */

void display::InitModels() 
{
    modelmgr = new ModelManager;

    lua_State *L = lua_open(0);
    lua_baselibopen (L);
    lua_strlibopen(L);
    lua_register (L, "FindDataFile", lua::FindDataFile);
    tolua_open(L);
    tolua_global_open(L);
    tolua_enigma_open(L);
    tolua_display_open(L);
    tolua_px_open(L);

    string fname;

    const video::VMInfo *vminfo = video::GetInfo();
    switch (vminfo->tile_size) {
    case 32: fname = "models-32.lua"; break;
    case 40: fname = "models-40.lua"; break;
    case 48: fname = "models-48.lua"; break;
    default:
        assert (0);
    }

    fname = enigma::FindDataFile (fname);
    if (lua_dofile (L, fname.c_str()) != 0) {
        fprintf (stderr, "Error loading '%s'\n", fname.c_str());
    }
    enigma::Log << "# models: " << modelmgr->num_templates() << endl;
//     enigma::Log << "# images: " << surface_cache.size() << endl;

    surface_cache_alpha.clear();
}

void display::ShutdownModels() 
{
    delete modelmgr;
    surface_cache.clear();
}

Surface *display::CropSurface (const Surface *s, Rect r) {
    return px::Grab(s, r);
}

/* Register a new model template `m' under the name `name'. */
void display::DefineModel (const char *name, Model *m) 
{
    if (modelmgr->has_model (name)) {
        enigma::Log << "Redefining model '" << name << "'\n";
        modelmgr->remove (name);
    }
    modelmgr->define (name, m);
}

Model * display::MakeModel (const string &name) 
{
    if (Model *m = modelmgr->create (name))
        return m;
    else {
        enigma::Log << "Unknown model " << name << endl;
        return modelmgr->create ("dummy");
    }
}

int display::DefineImage(const char *name, const char *fname,
                          int xoff, int yoff)
{
    px::Surface *sfc = surface_cache.get(fname);
    if (!sfc)
        return 1;

    DefineModel(name, new ImageModel(sfc, xoff, yoff));
    return 0;
}

void display::DefineImageModel (const char *name, px::Surface *s)
{
    DefineModel (name, new ImageModel(s, 0, 0));
}

int display::DefineSubImage(const char *name, const char *fname,
                             int xoff, int yoff, px::Rect subrect)
{
    px::Surface *sfc = surface_cache.get(fname);
    if (!sfc)
        return 1;
    
    DefineModel(name, new ImageModel(sfc, subrect, xoff, yoff));
    return 0;
}

void display::DefineRandModel(const char *name, int n, const char **names)
{
    RandomModel *m = new RandomModel();
    for (int i=0; i<n; i++) 
        m->add_model(names[i]);
    DefineModel(name, m);
}

void display::DefineShadedModel (
    const char *name, 
    const char *model, 
    const char *shadow)
{
    DefineModel(name, new ShadowModel(MakeModel(model),
                                      MakeModel(shadow)));
}


/* Create an image by overlaying several other images.  The first entry in
   `images' is the name of the background image, the following images are
   drawn on top of it. */
void display::DefineOverlayImage (const char *name, int n, 
                                  const char **images)
{
    Surface *sfc = Duplicate(surface_cache.get(images[0]));
    if (sfc) {
        GC gc(sfc);
	for (int i=1; i<n; i++) 
	    blit (gc, 0,0, surface_cache_alpha.get(images[i]));
	DefineModel(name, new ImageModel(sfc, 0,0));
        image_pile.push_back(sfc); // make sure it gets destructed
    }
}

void display::DefineComposite (const char *name, 
                               const char *bgname,
                               const char *fgname)
{
    DefineModel(name, new CompositeModel(MakeModel(bgname),
                                          MakeModel(fgname)));
}

void display::DefineAnim (const char *name, bool loop_p) {
    anim_templ = new Anim2d(loop_p);
    DefineModel(name, anim_templ);
    anim_templ_name = name;
}

void display::AddFrame (const char *name, const char *model, double time)
{
    if (anim_templ_name != name) 
        fprintf(stderr, "AddFrame: Cannot add frames to completed animations.");
    else
        anim_templ->add_frame(MakeModel(model), time / 1000.0);
}

void display::DefineAlias (const char *name, const char *othername)
{
    DefineModel(name, new AliasModel(othername));
}



/* -------------------- Image -------------------- */

Image::Image(px::Surface *sfc)
: surface(sfc), rect(surface->size()), refcount(1)
{}

Image::Image(px::Surface *sfc, const px::Rect &r)
: surface(sfc), rect(r), refcount(1)
{}


void display::incref(Image *i) { 
    ++i->refcount; 
}

void display::decref (Image *i) {
    if (-- i->refcount == 0) {
        delete i;
    }
}


void display::draw_image (Image *i, px::GC &gc, int x, int y) 
{
    blit(gc, x, y, i->surface, i->rect);
}

/* -------------------- RandomModel -------------------- */

Model * RandomModel::clone() {
    if (!modelnames.empty()) {
	int r = enigma::IntegerRand(0, modelnames.size()-1);
	return MakeModel(modelnames[r]);
    } else {
	fprintf(stderr, "display_2d.cc: empty RandomModel\n");
	return 0;
    }
}

/* -------------------- AliasModel -------------------- */

Model * AliasModel::clone() {
    return MakeModel(name);
}

/* -------------------- Anim2d -------------------- */

Anim2d::Anim2d(AnimRep *r) 
: rep(r), curframe(0), frametime(0), 
  finishedp(false), 
  changedp(false), 
  reversep(false),
  is_sprite(false),
  callback(0)
{
    rep->refcount++;
    frametime = enigma::DoubleRand (0, 0.02);
}

Anim2d::~Anim2d() {
    if (--rep->refcount == 0)
        delete rep;
}

void Anim2d::add_frame(Model *m, double duration) {
    rep->frames.push_back(new AnimFrame(m, duration));
}

void Anim2d::draw(px::GC &gc, int x, int y) 
{
    if (!finishedp) {
        AnimFrame *f =rep->frames[curframe];
        f->model->draw(gc,x,y);
        changedp = false;
    }
}
        
void Anim2d::draw_shadow (px::GC &gc, int x, int y) 
{
    if (!finishedp) {
        AnimFrame *f =rep->frames[curframe];
        f->model->draw_shadow(gc,x,y);
    }
}

void Anim2d::expose (ModelLayer *ml, const px::V2 &pos) {
    ml->activate (this);
    posx = pos[0];
    posy = pos[1];
    is_sprite = ml->is_sprite_layer();
}

void Anim2d::remove (ModelLayer *ml) {
    ml->deactivate (this);
}

bool Anim2d::has_changed(Rect &changed_region) {
    if (changedp) {
	if (is_sprite) 
	    changed_region = Rect(round_down<int>(posx-1), round_down<int>(posy-1), 3, 3);
	else //if (get_shadow())
	    changed_region = Rect(round_down<int>(posx), round_down<int>(posy), 2, 2);
// 	else
// 	    changed_region = Rect(static_cast<int>(posx), static_cast<int>(posy), 1, 1);
    }
    return changedp;
}

void Anim2d::tick (double dtime) 
{
    assert(curframe < rep->frames.size());
    frametime += dtime;
    double framedur = rep->frames[curframe]->duration;

    if (frametime >= framedur) {
        frametime -= framedur;
        changedp = true;

        if (reversep) {
            if (curframe >= 1)
                curframe--;
            else if (rep->loop)
                curframe = rep->frames.size()-1;
            else
                finishedp = true;
        } 
        else {
            if (curframe+1 < rep->frames.size())
                curframe++;
            else if (rep->loop)
                curframe = 0;
            else 
                finishedp = true;
        }
        if (finishedp && callback!=0)
            callback->animcb();
    }
}

void Anim2d::update (ModelLayer *ml, double dtime) {
    tick (dtime);
    px::Rect r;
    if (has_changed(r)) {
        ml->mark_redraw_area (r);
    }
}


/* -------------------- Functions -------------------- */

namespace display
{
    Surface *GetSurface (const std::string &filename)
    {
        return surface_cache.get(filename);
    }

    void RegisterImage (const std::string &name, Image *img)
    {
    }

    Image *LookupImage (const std::string &name)
    {
        return 0;
    }

}
