/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2003 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */

#include "library.h"
#include <stdio.h>
#include <gmp.h> /* used for types, *not* for functions or constants! */

/* The following functions are not visible from epsilon: */
static void (*loaded_mpz_init)(mpz_t);
static void (*loaded_mpz_init_set_si)(mpz_t, signed long int);
static signed long int (*loaded_mpz_get_si)(mpz_t);
static void (*loaded_mpz_add)(mpz_t, mpz_t, mpz_t);
static void (*loaded_mpz_sub)(mpz_t, mpz_t, mpz_t);
static void (*loaded_mpz_mul)(mpz_t, mpz_t, mpz_t);
static void (*loaded_mpz_tdiv_q)(mpz_t, mpz_t, mpz_t);
static void (*loaded_mpz_tdiv_r)(mpz_t, mpz_t, mpz_t);
static int (*loaded_mpz_cmp)(mpz_t, mpz_t);
static char* (*loaded_mpz_get_str) (char*, int, mpz_t);
static void (*loaded_mp_set_memory_functions)
      (void *(*ALLOC_FUNC_PTR) (size_t),
      void *(*REALLOC_FUNC_PTR) (void *, size_t, size_t),
      void (*FREE_FUNC_PTR) (void *, size_t));
static void (*loaded_mpz_init_set_str)(mpz_t, char *, int);

/* The following constants are visible from epsilon: */
/* none */

/* Used only in this compilation unit: */
static integer_t SIZEOF_MPZ_T; /* in words */

/* Custom heap managing functions for GMP: */
integer_t size_in_chars_to_size_in_words(size_t size_in_chars){
  if(size_in_chars % sizeof(word_t) == 0)
    return (integer_t) (size_in_chars / sizeof(word_t));
  else
    return (integer_t) (size_in_chars / sizeof(word_t) + 1);
}

void* my_malloc(size_t size){
  /* Allocate in the garbage-collected heap: */
  return allocate_inexact(size_in_chars_to_size_in_words(size));
}

void* my_realloc(void* p, size_t old_size, size_t new_size){
  /* Allocate a new block, copy the old data. Don't free the old block. */
  void* new_block = allocate_inexact(size_in_chars_to_size_in_words(new_size));
  memcpy(new_block,
	 p,
	 (old_size <= new_size) ? old_size : new_size); /* this is in chars */
  return new_block;
}

void my_free(void* p, size_t size){
  /* Do nothing. The GC will free the block when needed. */
}

void initialize_c_library(){
  dynamic_library_handle_t handle;

#ifdef DEBUG
  fprintf(stderr, "Initializing C library gmp... ");
#endif

  /* Open the gmp library: */
  handle = open_dynamic_library("libgmp.so");
  
  /* Load functions from libgmp: */
  loaded_mpz_init = load_symbol_from_handle(handle, "__gmpz_init");
  loaded_mpz_init_set_si = load_symbol_from_handle(handle, "__gmpz_init_set_si");
  loaded_mpz_get_si = load_symbol_from_handle(handle, "__gmpz_get_si");
  loaded_mpz_add = load_symbol_from_handle(handle, "__gmpz_add");
  loaded_mpz_sub = load_symbol_from_handle(handle, "__gmpz_sub");
  loaded_mpz_mul = load_symbol_from_handle(handle, "__gmpz_mul");
  loaded_mpz_tdiv_q = load_symbol_from_handle(handle, "__gmpz_tdiv_q");
  loaded_mpz_tdiv_r = load_symbol_from_handle(handle, "__gmpz_tdiv_r");
  loaded_mpz_cmp = load_symbol_from_handle(handle, "__gmpz_cmp");
  loaded_mpz_get_str = load_symbol_from_handle(handle, "__gmpz_get_str");
  loaded_mpz_init_set_str = load_symbol_from_handle(handle, "__gmpz_init_set_str");
  loaded_mp_set_memory_functions =
    load_symbol_from_handle(handle, "__gmp_set_memory_functions");

  /* Exported constants: */
  /* none */

  /* Compute SIZEOF_MPZ_T (in words, not in chars!): */
  if(sizeof(mpz_t) % sizeof(word_t) == 0)
    SIZEOF_MPZ_T = sizeof(mpz_t) / sizeof(word_t); /* no space wasted */
  else
    SIZEOF_MPZ_T = sizeof(mpz_t) / sizeof(word_t) + 1; /* < 1 word wasted */

  /* Bind the GMP memory functions to the eAM garbage collector primitives: */
  loaded_mp_set_memory_functions(my_malloc, my_realloc, my_free);

#ifdef DEBUG
  fprintf(stderr, "done.\n");
#endif
}

/* Commodity definitions: */

mpz_t* allocate_mpz(){
  /* SIZEOF_MPZ_T should be less MAXIMUM_SMALL_HOMOGENEOUS_SIZE, so we
     can use allocate_exact(): */
  return (mpz_t*)allocate_exact(SIZEOF_MPZ_T);
}

mpz_t* allocate_and_init_mpz(){
  mpz_t* r = allocate_mpz();

  loaded_mpz_init(*r);
  return r;
}

int compare(mpz_t** b1_b2){
  return loaded_mpz_cmp(*(b1_b2[0]), *(b1_b2[1]));
}

/* Here begin definitions of symbols to be exported. All of these are
   visible from epsilon: */

word_t integer_to_bignum(word_t i){
  mpz_t* r = allocate_mpz();
  loaded_mpz_init_set_si (*r, (signed long int)(integer_t)i);
  return (word_t)r;
}
word_t bignum_to_integer(word_t b){
  return (word_t)(integer_t)loaded_mpz_get_si(*(mpz_t*)b);
}

word_t plus(word_t b1_b2){
  mpz_t* r = allocate_and_init_mpz();
  loaded_mpz_add(*r, *((mpz_t**)b1_b2)[0], *((mpz_t**)b1_b2)[1]);
  return r;
}
word_t minus(word_t b1_b2){
  mpz_t* r = allocate_and_init_mpz();
  loaded_mpz_sub(*r, *((mpz_t**)b1_b2)[0], *((mpz_t**)b1_b2)[1]);
  return r;
}
word_t times(word_t b1_b2){
  mpz_t* r = allocate_and_init_mpz();
  loaded_mpz_mul(*r, *((mpz_t**)b1_b2)[0], *((mpz_t**)b1_b2)[1]);
  return r;
}
word_t divided(word_t b1_b2){
  mpz_t* r = allocate_and_init_mpz();
  loaded_mpz_tdiv_q(*r, *((mpz_t**)b1_b2)[0], *((mpz_t**)b1_b2)[1]);
  return r;
}
word_t modulo(word_t b1_b2){
  mpz_t* r = allocate_and_init_mpz();
  loaded_mpz_tdiv_r(*r, *((mpz_t**)b1_b2)[0], *((mpz_t**)b1_b2)[1]);
  return r;
}

word_t equal(word_t b1_b2){
  return (word_t)(integer_t)(compare(b1_b2) == 0);
}
word_t different(word_t b1_b2){
  return (word_t)(integer_t)(compare(b1_b2) != 0);
}
word_t less(word_t b1_b2){
  return (word_t)(integer_t)(compare(b1_b2) < 0);
}
word_t greater(word_t b1_b2){
  return (word_t)(integer_t)(compare(b1_b2) > 0);
}
word_t less_or_equal(word_t b1_b2){
  return (word_t)(integer_t)(compare(b1_b2) <= 0);
}
word_t greater_or_equal(word_t b1_b2){
  return (word_t)(integer_t)(compare(b1_b2) >= 0);
}

word_t bignum_to_string(word_t b){
  char* c_string = loaded_mpz_get_str(NULL, 10, *((mpz_t*)b));

  //fprintf(stderr, ">%s<\n", c_string);
  return c_string_to_epsilon_string(c_string);
}

word_t string_to_bignum(word_t s){
  char* c_string = epsilon_string_to_c_string(s);
  mpz_t* r = allocate_mpz();
  loaded_mpz_init_set_str(*r, c_string, 10);
  free(c_string); /* epsilon_string_to_c_string() uses malloc() */
  
  return (word_t)r;
}
