/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2002, 2003 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */

/* This module provides the powerful GNU-style (i.e. POSIX with GNU extensions)
   command-line parsing facility , but does not necessarily require GNU libc.
   As such it may be preferrable for portability (especially towards the free
   operating systems derived from BSD, which I would like to support as
   secondary targets, right after GNU systems). */

#include <math.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <limits.h>
#include <ctype.h>
#include <strings.h>
#include <string.h>

#include "command_line.h"

// To do: test for unicity of short and long options
// To do: forbid long options beginning with "no-"

typedef enum
{
  toggle, with_optional_parameter, with_nonoptional_parameter
}
type_of_parameter_t;

typedef struct
{
  type_of_parameter_t type;
  char *long_name;
  char short_name;
  char *value;
  char *default_value;
  char *default_value_with_unspecified_parameter;	/* only used with with_optional_parameter */
  char *description;
}
option_t;

int options_number;
int options_size;

option_t *options = NULL;	/* NULL stands for 'not initialized' */
option_t **pointers_to_options_sorted_by_long_name;
option_t **pointers_to_options_sorted_by_short_name;

char *program_name = "";
char *synopsis = "";
char *version = "";
char *copyright = "";
char *general_help = "";
char *license_text = "";
char *bug_reporting = "";

void
set_program_name (char *x)
{
  program_name = x;
}

void
set_synopsis_string (char *x)
{
  synopsis = x;
}

void
set_general_help_message (char *x)
{
  general_help = x;
}

void
set_version_string (char *x)
{
  version = x;
}

void
set_copyright_string (char *x)
{
  copyright = x;
}

void
set_license_message (char *x)
{
  license_text = x;
}

void
set_bug_reporting_message (char *x)
{
  bug_reporting = x;
}

void
show_help_but_dont_exit ()
{				/* To do: this works but is ugly */
  int i;

  if (strcmp (synopsis, ""))
    printf ("Usage: %s\n\n", synopsis);
  if (strcmp (general_help, ""))
    printf ("%s\n\n", general_help);
  printf ("You can mix options and non-option arguments in any order.\n\n");
  printf ("Options:\n");
  for (i = 0; i < options_number; i++)
    {
      char *temp =
	(char *) malloc (sizeof (char) *
			 (strlen (options[i].long_name) + 15));
      if (options[i].short_name != '\0')
	{
	  printf ("  -%c", options[i].short_name);
	  if (options[i].type == with_nonoptional_parameter)
	    printf (" XXXX");
	}
      else
	printf ("    ");
      if ((options[i].short_name != '\0') &&
	  strcmp (options[i].long_name, ""))
	printf (", ");
      else
	printf ("  ");
      if (strcmp (options[i].long_name, ""))
	{
	  printf ("--");
	  if (options[i].type == with_optional_parameter)
	    sprintf (temp, "%s[=XXXX]", options[i].long_name);
	  else if (options[i].type == with_nonoptional_parameter)
	    sprintf (temp, "%s=XXXX  ", options[i].long_name);
	  else
	    sprintf (temp, "%s   ", options[i].long_name);
	}
      else
	{
	  printf ("  ");
	  strcpy (temp, "");
	}
      if ((options[i].short_name != '\0') &&
	  (options[i].type == with_nonoptional_parameter))
	printf ("%-17s", temp);
      else
	printf ("%-22s", temp);
      free (temp);
      printf ("%s ", options[i].description);
      if (options[i].type == with_optional_parameter)
	printf ("[%s, no XXXX means %s]",
		options[i].default_value,
		options[i].default_value_with_unspecified_parameter);
      else if (options[i].type == toggle)
	printf ("[%s]", options[i].default_value ? "yes" : "no");
      printf ("\n");
    }				/* for */
  printf ("\n");
  printf ("You can abbreviate long options with non-ambiguous prefixes.\n");
  printf
    ("Any boolean long option --OPTION has the negative form --no-OPTION.\n");
  printf
    ("The special token -- stops option processing: all parameters after\n");
  printf ("-- are seen as non-options.\n");
  if (strcmp (bug_reporting, ""))
    printf ("\n%s\n", bug_reporting);
}

void show_help(){
  show_help_but_dont_exit();
  exit(EXIT_FAILURE);
}

void
initialize_options ()
{
  options_size = 1;
  options_number = 0;
  options = (option_t *) malloc (sizeof (option_t) * options_size);
  add_toggle_option ("help", '?', 0, "display this help message and exit");
  add_toggle_option ("version", 'V', 0,
		     "display version information and exit");
}

int
add_option ()
{				/* Returns the index of the newly-created option */
  if (options == NULL)
    initialize_options ();
  options_number++;
  if (options_number > options_size)
    {
      options_size *= 2;
      options = (option_t *) realloc (options,
				      sizeof (option_t) * options_size);
    }
  options[options_number - 1].long_name = "";
  options[options_number - 1].value = "";
  options[options_number - 1].default_value = "";
  options[options_number - 1].description = "";
  return options_number - 1;
}

void
add_toggle_option (const char *long_name,
		   const char short_name,
		   const long default_value, const char *description)
{
  int index;

  if((default_value != 0) && (default_value != 1)){
    fprintf(stderr, "About the default value of toggle --%s, -%c:\n",
	    long_name, short_name);
    fatal("Default value must be either 0 or 1");
  }
  if ((short_name == '\0') && !strcmp (long_name, ""))
    {
      fprintf (stderr, "A toggle option has neither long nor short name\n");
      exit (EXIT_FAILURE);
    }
  index = add_option ();
  options[index].type = toggle;
  options[index].short_name = short_name;
  options[index].long_name =
    malloc (sizeof (char) * (strlen (long_name) + 1));
  strcpy (options[index].long_name, long_name);
  options[index].description =
    malloc (sizeof (char) * (strlen (description) + 1));
  strcpy (options[index].description, description);
  options[index].value = (char *) default_value;
  options[index].default_value = (char *) default_value;
}

void
add_option_with_nonoptional_parameter (const char *long_name,
				       const char short_name,
				       const char *default_value,
				       const char *description)
{
  int index;

  if ((short_name == '\0') && !strcmp (long_name, ""))
    {
      fprintf (stderr,
	       "An option with nonoptional parameter has neither long nor short name\n");
      exit (EXIT_FAILURE);
    }
  index = add_option ();
  options[index].type = with_nonoptional_parameter;
  options[index].short_name = short_name;
  options[index].long_name =
    malloc (sizeof (char) * (strlen (long_name) + 1));
  strcpy (options[index].long_name, long_name);
  options[index].description =
    malloc (sizeof (char) * (strlen (description) + 1));
  strcpy (options[index].description, description);
  options[index].value =
    malloc (sizeof (char) * (strlen (default_value) + 1));
  strcpy (options[index].value, default_value);
  options[index].default_value =
    malloc (sizeof (char) * (strlen (default_value) + 1));
  strcpy (options[index].default_value, default_value);
}

void
add_option_with_optional_parameter (const char *long_name,
				    const char *initial_value,
				    const char *value_with_no_parameter,
				    const char *description)
{
  int index;

  if (!strcmp (long_name, ""))
    {
      fprintf (stderr,
	       "An option with optional parameter has no long name\n");
      exit (EXIT_FAILURE);
    }
  index = add_option ();
  options[index].type = with_optional_parameter;
  options[index].short_name = '\0';
  options[index].long_name =
    malloc (sizeof (char) * (strlen (long_name) + 1));
  strcpy (options[index].long_name, long_name);
  options[index].description =
    malloc (sizeof (char) * (strlen (description) + 1));
  strcpy (options[index].description, description);

  options[index].value =
    malloc (sizeof (char) * (strlen (initial_value) + 1));
  strcpy (options[index].value, initial_value);

  options[index].default_value =
    malloc (sizeof (char) * (strlen (initial_value) + 1));
  strcpy (options[index].default_value, initial_value);

  options[index].default_value_with_unspecified_parameter =
    malloc (sizeof (char) * (strlen (value_with_no_parameter) + 1));
  strcpy (options[index].default_value_with_unspecified_parameter,
	  value_with_no_parameter);
}

int
compare_by_long_name (const void *x, const void *y)
{
  /*
     fprintf(stderr,"Comparing --%s and --%s, result is %i\n",
     (*(option_t**)x)->long_name,
     (*(option_t**)y)->long_name,
     strcmp((*(option_t**)x)->long_name,
     (*(option_t**)y)->long_name));
   */
  return strcmp ((*(option_t **) x)->long_name,
		 (*(option_t **) y)->long_name);
}

int
compare_by_short_name (const void *x, const void *y)
{
  int a = (*(option_t **) x)->short_name;
  int b = (*(option_t **) y)->short_name;

  if (a == '\0')
    a = INT_MIN;
  if (b == '\0')
    b = INT_MIN;

  if (a < b)
    return -1;
  else if (a > b)
    return +1;
  else
    return 0;
}

void
sort_options ()
{
  int i;
  /* Allocate storage for arrays of pointers: */
  pointers_to_options_sorted_by_long_name = (option_t **)
    malloc (sizeof (option_t *) * options_number);
  pointers_to_options_sorted_by_short_name = (option_t **)
    malloc (sizeof (option_t *) * options_number);

  /* Initialize arrays of pointers: */
  for (i = 0; i < options_number; i++)
    {
      pointers_to_options_sorted_by_long_name[i] = &options[i];
      pointers_to_options_sorted_by_short_name[i] = &options[i];
    }

  /* Sort arrays of pointers: */
  qsort (pointers_to_options_sorted_by_long_name,
	 options_number, sizeof (option_t *), compare_by_long_name);
  /*
     for(i=0;i<options_number;i++)
     fprintf(stderr," %i.  -%1c, --%s\n",
     i,
     pointers_to_options_sorted_by_long_name[i]->short_name,
     pointers_to_options_sorted_by_long_name[i]->long_name);
     fprintf(stderr,"\n\n");
   */
  qsort (pointers_to_options_sorted_by_short_name,
	 options_number, sizeof (option_t *), compare_by_short_name);
}

int
round_float_to_int (double x)
{				/* I only need it for positive numbers */
  double fractional_part = x - (int) x;

  if (fractional_part < 0.5)
    return (int) x;
  else
    return ((int) x) + 1;
}

int
search_best_matching_option (option_t ** pointers_to_options,
			     option_t * option,
			     int (*compare) (const void *, const void *))
{
  /* Make a binary search: */
  float a = 0, b = options_number - 1, center;
  int compare_result;
  //int iteration=1;
  /*
     fprintf(stderr,"\t[searching for -%c, --%s]\n",
     option->short_name,
     option->long_name);
   */
  do
    {
      center = (a + b) / 2.0f;
      /*
         fprintf(stderr,"a == %lf, b == %lf, c == %lf\n", a, b, center);
         fprintf(stderr,"\tc is -%c, --%s\n\n",
         pointers_to_options[round_float_to_int(center)]->short_name,
         pointers_to_options[round_float_to_int(center)]->long_name);
       */
      compare_result =
	compare (&pointers_to_options[round_float_to_int (center)], &option);

      if (compare_result == 0)
	return round_float_to_int (center);

      if (compare_result < 0)
	a = center;
      else			/* compare_result > 0 */
	b = center;
    }
  while (fabs (a - b) >= 0.5);
  /* No exact match: return the nearest index */
  /*
     fprintf(stderr,"Returning -%c, --%s as a last resort\n\n",
     pointers_to_options[round_float_to_int(center)]->short_name,
     pointers_to_options[round_float_to_int(center)]->long_name);
   */
  return round_float_to_int (center);
}

option_t temporary_option;

option_t *
short_name_to_option (char short_name)
{
  option_t *r;

  temporary_option.short_name = short_name;
  r =
    pointers_to_options_sorted_by_short_name[search_best_matching_option
					     (pointers_to_options_sorted_by_short_name,
					      &temporary_option,
					      compare_by_short_name)];
  if (r->short_name != short_name)
    {
      fprintf (stderr, "Unknown short option -%c. Usage is shown by --help\n", short_name);
      exit (EXIT_FAILURE);
    }
  else
    return r;
}

int
is_a_prefix_of (char *prefix, char *whole)
{
  return ((char*) strstr (whole, prefix)) == whole;
}

option_t *
long_name_to_option (char *long_name)
{
  int index;

  /* Make a binary search: */
  temporary_option.long_name = long_name;
  index =
    search_best_matching_option (pointers_to_options_sorted_by_long_name,
				 &temporary_option, compare_by_long_name);

  /* Test for an exact match: */
  if (strcmp (long_name,
	      pointers_to_options_sorted_by_long_name[index]->long_name) == 0)
    {
      /* fprintf(stderr, "long_name_to_option: exact match\n"); */
      return pointers_to_options_sorted_by_long_name[index];
    }

  /* If long_name is a prefix of one or more options, then index is either:
     i)  the index of the first option whose long_name is a prefix
     or
     ii) the index of the option immediately preceding the first option whose
     long_name is a prefix; ii) does not happen when long_name is a
     prefix of the first option */
  if ((!is_a_prefix_of (long_name, pointers_to_options_sorted_by_long_name[index]->long_name)) && (index + 1 < options_number))	/* Case ii), or no such prefix is in options */
    index++;			/* Now case ii) can be managed like case i) */
  if (is_a_prefix_of (long_name,
		      pointers_to_options_sorted_by_long_name[index]->
		      long_name))
    {
      /* Case i), or case ii) detected as i) because of the above trick */
      if (index + 1 < options_number)
	if (is_a_prefix_of (long_name,
			    pointers_to_options_sorted_by_long_name[index +
								    1]->
			    long_name))
	  {
	    /* Ambigous prefix */
	    fprintf (stderr, "The option prefix --%s is ambiguous.\n",
		     long_name);
	    exit (EXIT_FAILURE);
	  }
      /* Non-ambiguous prefix */
      fprintf (stderr,
	       "The long option prefix --%s stands for --%s. Ok, no ambiguity.\n",
	       long_name,
	       pointers_to_options_sorted_by_long_name[index]->long_name);
      return pointers_to_options_sorted_by_long_name[index];
    }
  else
    {				/* There is no option whose long_name is a prefix */
      fprintf (stderr, "Unknown long option --%s. Usage is shown by --help\n", long_name);
      exit (EXIT_FAILURE);
    }
  /* Unreachable code: this return is just to avoid a GCC warning */
  return pointers_to_options_sorted_by_long_name[0];
}

void
add_non_option (int *argc_before_now, char **argv, char *value)
{
  int i;
  argv[*argc_before_now] = value;
  (*argc_before_now)++;

  /*
     fprintf(stderr,"Current non_options, adding %s:\n[ ", value);
     for(i = 0; i < *argc_before_now; i++)
     fprintf(stderr, "%s ", argv[i]);
     fprintf(stderr,"]\n\n");
   */
}

char *
long_option_to_value (char *name)
{
  option_t *option = long_name_to_option (name);

  if (strcmp (option->long_name, name))
    {
      fprintf (stderr,
	       "Long option --%s is not known (match must be exact)\n", name);
      exit (EXIT_FAILURE);
    }
  return option->value;
}

char *
short_option_to_value (char name)
{
  option_t *option = short_name_to_option (name);

  if (option->short_name != name)
    {
      fprintf (stderr,
	       "Short option -%c is not known (match must be exact)\n", name);
      exit (EXIT_FAILURE);
    }
  return option->value;
}

void
show_version ()
{
  if (strcmp (program_name, "") || strcmp (version, ""))
    printf ("%s %s\n", program_name, version);
  if (strcmp (copyright, ""))
    printf ("%s\n", copyright);
  if (strcmp (license_text, ""))
    printf ("%s\n", license_text);
  exit (EXIT_SUCCESS);
}

/* To do: this works but is ugly, as any custom parser (well, maybe more than
   the average custom parser :-)) */
void
parse_command_line (int *argc, char **argv)
{
  int i, new_argc = 1;
  option_t *option;

  /* Initialize if needed: */
  if (options == NULL)
    initialize_options ();

  /* Make the sorted arrays of pointers to options: */
  sort_options ();

  /* Look at every element of argv: */
  for (i = 1; i < *argc; i++)
    {
      char *this_element = argv[i];
      int length = strlen (this_element);

      if (length == 1)
	{			/* ? cannot be an option */
	  add_non_option (&new_argc, argv, this_element);
	  continue;		/* Jump to the next iteration */
	}
      if (length >= 2)		/* ??* is not an option */
	if (this_element[0] != '-')
	  {
	    add_non_option (&new_argc, argv, this_element);
	    continue;		/* Jump to the next iteration */
	  }
      if (length == 2)
	if (!strcmp (this_element, "--"))
	  {
	    /* Add all following arguments as non-options: */
	    for (i++; i < *argc; i++)
	      add_non_option (&new_argc, argv, argv[i]);
	    break;		/* Exit the loop */
	  }

      /* If we arrived here then this_element is an option */
      if (this_element[1] == '-')
	{			/* long option */
	  int equal_notation_is_used = 0;	/* is the notation --option=XXXX used? */
	  char *equal_notation_parameter;	/* The XXXX, in the case above */
	  char *pointer_to_equals_in_this_element;
	  int position_of_equals_in_this_element;
	  int no_notation_is_used = 0;	/* is the notation --no-OPTION used? */

	  /* Test for equal-notation: */
	  pointer_to_equals_in_this_element = index (this_element, '=');
	  if (pointer_to_equals_in_this_element != NULL)
	    {
	      /* fprintf(stderr, "Equal-notation is used\n"); */
	      equal_notation_is_used = 1;
	      position_of_equals_in_this_element =
		pointer_to_equals_in_this_element - this_element;
	      equal_notation_parameter =
		pointer_to_equals_in_this_element + 1;
	      *pointer_to_equals_in_this_element = '\0';	/* Overwite '=' */
	      /* fprintf(stderr, "XXXX = \"%s\"\n", equal_notation_parameter); */
	    }

	  /* Test for no--notation: */
	  if (length >= 6)	/* 6 characters is the minimum for the --no--notation */
	    if (!strncmp (this_element, "--no-", 5))
	      {
		/* fprintf(stderr, "no--notation is used\n"); */
		no_notation_is_used = 1;
		/* Change --no-OPTION into --OPTION: */
		this_element += 3;
		this_element[0] = '-';
	      }

	  /* fprintf(stderr,"\t\'%s\' is a long option\n", this_element); */

	  /* no--notation and equal-notation together are forbidden: */
	  if (equal_notation_is_used && no_notation_is_used)
	    {
	      fprintf (stderr,
		       "%s: --no-XXXX and --XXXX=YYYY notations can't be used together\n",
		       this_element);
	      exit (EXIT_FAILURE);
	    }

	  /* Lookup the option name to find the option: */
	  option = long_name_to_option (this_element + 2);

	  switch (option->type)
	    {
	    case toggle:
	      {
		if (equal_notation_is_used)
		  {
		    if (!strcmp (equal_notation_parameter, "yes"))
		      ;		/* "yes" is redundant: do nothing */
		    else if (!strcmp (equal_notation_parameter, "no"))
		      no_notation_is_used = 1;	/* equivalent to no--notation */
		    else
		      {		/* parameter is different than "yes" and "no" */
			fprintf (stderr,
				 "%s: --OPT=XXXX notation can't be used with XXXX\n"
				 "different than \"yes\" and \"no\" since OPT is boolean\n",
				 this_element);
			exit (EXIT_FAILURE);
		      }
		  }
		if (no_notation_is_used)
		  option->value = (char *) 0;
		else
		  option->value = (char *) 1;
		break;		/* exit the switch */
	      }			/* case toggle */
	    case with_optional_parameter:
	      {
		if (no_notation_is_used)
		  {
		    fprintf (stderr,
			     "%s: --no-XXXX notation can't be used if option is not boolean\n",
			     this_element);
		    exit (EXIT_FAILURE);
		  }
		if (equal_notation_is_used)
		  option->value = equal_notation_parameter;
		else
		  option->value =
		    option->default_value_with_unspecified_parameter;
		break;		/* exit the switch */
	      }			/* case with_optional_parameter */
	    case with_nonoptional_parameter:
	      {
		if (no_notation_is_used)
		  {
		    fprintf (stderr,
			     "%s: --no-XXXX notation can't be used if option is not boolean\n",
			     this_element);
		    exit (EXIT_FAILURE);
		  }
		if (equal_notation_is_used)
		  option->value = equal_notation_parameter;
		else
		  {
		    if (i + 1 == *argc)
		      {
			fprintf (stderr, "%s requires a parameter\n",
				 this_element);
			exit (EXIT_FAILURE);
		      }
		    option->value = argv[i + 1];
		    i++;	/* Skip the next element of argv */
		  }
		break;		/* exit the switch */
	      }			/* case with_nonoptional_parameter */
	    default:
	      {
		fprintf (stderr, "This cannot happen SDFTYTR\n");
		exit (EXIT_FAILURE);
	      }			/* default */
	    }			/* switch */
	  continue;		/* go on with the next iteration */
	}			/* long option: end */
      else
	{			/* short option */
	  /* fprintf(stderr,"\t\'%s\' is a short option\n", this_element); */
	  option = short_name_to_option (this_element[1]);

	  /* short options can not have optional parameters: they can be
	     either toggles or with nonoptional parameter. */
	  if (option->type == toggle)
	    {			/* toggle */
	      if (length > 2)
		{
		  fprintf (stderr,
			   "The short option -%c should have no parameters.\n",
			   this_element[1]);
		  exit (EXIT_FAILURE);
		}
	      option->value = (char *) 1;
	    }			/* toggle: end */
	  else
	    {			/* with_nonoptional_parameter */
	      if (length > 2)
		{		/* option and parameter are concatenated */
		  option->value = (char *) realloc (option->value,
						    sizeof (char) *
						    (strlen (this_element + 2)
						     + 1));
		  strcpy (option->value, this_element + 2);
		}		/* option and parameter are concatenated: end */
	      else
		{		/* the parameter is in the next element of argv */
		  if (i + 1 == *argc)
		    {
		      fprintf (stderr,
			       "The short option -%c should have a parameter.\n",
			       this_element[1]);
		      exit (EXIT_FAILURE);
		    }
		  option->value = (char *) realloc (option->value,
						    sizeof (char) *
						    (strlen (argv[i + 1]) +
						     1));
		  strcpy (option->value, argv[i + 1]);
		  i++;
		}		/* parameteris next element of argv: end */
	    }			/* with_nonoptional_parameter: end */
	  continue;		/* go to the next iteration */
	}			/* short option: end */
    }				/* for */

  /* Finally update argc, and set argv[argc] to NULL: */
  *argc = new_argc;
  argv[*argc] = NULL;

  ///////////////////////
  //show_options_status();
  ///////////////////////

  /* Execute code for --help or --version if requested: */
  if (short_option_to_value ('V'))
    show_version ();

  if (short_option_to_value ('?'))
    show_help ();
}

/* Very useful for debugging: */
void
show_options_status ()
{
  int i;

  printf ("-------------------------------\n"
	  "Status of command-line options:\n");
  for (i = 0; i < options_number; i++)
    {
      if (options[i].short_name != '\0')
	printf ("-%c ", options[i].short_name);
      else
	printf ("   ");
      if (strcmp (options[i].long_name, ""))
	printf ("--%-20s ", options[i].long_name);
      else
	printf ("  %-20s ", "");
      if (options[i].type == toggle)
	printf ("%s\n", options[i].value ? "yes" : "no");
      else
	printf ("%s\n", options[i].value);
    }
  printf ("-------------------------------\n");
}

int has_extension(const char* s, const char* e){
  int i,length=strlen(s),elength=strlen(e);

  for(i=length;i>=0;i--)
    if(s[i]=='.')
      return !strcmp(s+i+1,e);
    else if((i<length-elength)||(i==0))
      break;
  /* else do nothing and loop again. */

  return 0;
}

int has_some_extension(const char *s){
  int i,length=strlen(s);
  
  for(i = length - 1; i >= 0; i--)
    if(s[i] == '.')
      return 1;
    /* else do nothing and loop again. */
  
  return 0;
}

/* Don't transform XXXX.E1 into XXXX.E2: instead transform
   XXXX.E1 in XXXXE2, WHERE E2 can contain a dot: */
char* replace_extension_with_part_of_filename(const char* s,const char* e){
  static char buffer_for_names_without_extension[MAX_FILENAME_LENGTH];
  static char buffer[MAX_FILENAME_LENGTH];
  int i,length=strlen(s);

  if(length>=MAX_FILENAME_LENGTH-5)
    fatal("file name is too long.\n");

  /* If there is no dot then just append a dot and e: */
  if(! has_some_extension(s)){
    sprintf(buffer_for_names_without_extension, "%s.XXX", s);
    return replace_extension(buffer_for_names_without_extension, e);
  }

  for(i=length;i>=0;i--)
    if(s[i]=='.')
      break;

  if(i==-1){
    strncpy(buffer,s,MAX_FILENAME_LENGTH);
    strcpy(buffer+length+1,e);
  }
  else{
    strncpy(buffer,s,i);
    strcpy(buffer+i,e);
  }

  return buffer;
}


/* Don't transform XXXX.E1 into XXXX.E2: instead transform
   XXXX.E1 in XXXXE2, WHERE E2 can contain a dot: */
char* replace_extension(const char* s,const char* e){
  static char buffer_for_names_without_extension[MAX_FILENAME_LENGTH];
  static char buffer[MAX_FILENAME_LENGTH];
  int i,length=strlen(s);
    
  if(length>=MAX_FILENAME_LENGTH-5)
    fatal("file name is too long.\n");

  /* If there is no dot then just append a dot and e: */
  if(! has_some_extension(s)){
    sprintf(buffer_for_names_without_extension, "%s.XXX", s);
    return replace_extension(buffer_for_names_without_extension, e);
  }
  
  for(i=length;i>=0;i--)
    if(s[i]=='.')
      break;
  
  if(i==-1){
    strncpy(buffer,s,MAX_FILENAME_LENGTH);
    strcpy(buffer+length+1,e);
  }
  else{
    strncpy(buffer,s,i+1);
    strcpy(buffer+i+1,e);
  }
  
  return buffer;
}

/* filename may contain or not contain directories.
   Returns a pointer to a static buffer. Not reentrant. */
char* remove_path_from_filename(char* filename){
  int i;
  int length = strlen(filename);
  static char buffer[MAX_FILENAME_LENGTH];

  for(i = length - 1; i >= 0; i--){
    if(filename[i] == '/'){ /* we found the beginning of the actual file */
      /* Ok, take the last part, make a copy of it and return: */
      strcpy(buffer, filename + i + 1);
      return buffer;
    }      
  } /* for */
  
  /* If we arrived here then filename does not contain directories: */
  strcpy(buffer, filename);
  return buffer;
}

void command_line_fatal(const char* s){
  fprintf(stderr, "%s: FATAL: %s\n", program_name, s);
  fprintf(stderr, "--help shows usage information\n");
  exit(EXIT_FAILURE);
}

void fatal(const char* s){
  fprintf(stderr, "%s: FATAL: %s\n", program_name, s);
  exit(EXIT_FAILURE);
}

/*
int main(int argc, char **argv){
  set_program_name("test");
  set_synopsis_string("a.out [OPTION_1] ... [OPTION_N] FILE1 FILE2");
  set_general_help_message("Run a.out and find out what happens.");
  set_version_string("1.0");
  set_copyright_string("No copyright");
  set_license_message("Do anything you want");
  set_bug_reporting_message("Please report bugs to bug@foo.com .");

  add_toggle_option(
    "long-name",
    'n',
    1,
    "help for long-name");

  add_option_with_optional_parameter(
    "test-one",
    "default_value_for_1",
    "NO_PARAM",
    "first test option");
  add_option_with_nonoptional_parameter(
    "test-two",
    '2',
    "default_value_for_2",
    "second test option");
  add_option_with_nonoptional_parameter(
    "",
    '3',
    "default_value_for_3",
    "third test option");

  parse_command_line(&argc, argv);

  show_options_status();
  
  printf("\n\nAfter parse_command_line() argv is [ ");
  int i;
  for(i = 0; i < argc; i++)
    printf("%s ", argv[i]);
  printf("]\n");

  return EXIT_SUCCESS;
}
*/
