/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2002, 2003 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */

/* This module provides the powerful GNU-style (i.e. POSIX with GNU extensions)
   command-line parsing facility , but does not necessarily require GNU libc.
   As such it may be preferrable for portability (especially towards the free
   operating systems derived from BSD, which I would like to support as
   secondary targets, right after GNU systems). */

#ifndef COMMAND_LINE_H_
#define COMMAND_LINE_H_


/* Dirty and hackish, shame on myself. To do: use a dynamic buffer */
#define MAX_FILENAME_LENGTH 10000

/* To do: test for unicity of short and long options
   To do: forbid long options beginning with "no-" */

/* Functions to set strings to be shown on --help or --version.
   By default all strings are set to "". */
void set_program_name (char *string);
void set_synopsis_string (char *string);
void set_general_help_message (char *string);
void set_version_string (char *string);
void set_copyright_string (char *string);
void set_license_message (char *string);
void set_bug_reporting_message (char *string);

/* Functions to define options. All string parameters must be non-NULL. 
   The options -h / --help and -V / --version are automatically defined. 
   For long options "" means 'no long name'.
   For short options '\0' means 'no short name'.
   The description parameter is used for --help.
   Don't write the "--" for long options or the "-" for short options;
   they are automatically inserted at the beginning.

   Toggle options have always either value (char*)0 or (char*)1. */
void add_toggle_option (const char *long_option,
			const char short_name,
			const long initial_value,
			const char *description);
void add_option_with_nonoptional_parameter (const char *long_name,
					    const char short_name,
					    const char *initial_value,
					    const char *description);
void add_option_with_optional_parameter (const char *long_name,
					 const char *initial_value,
					 const char *value_with_no_parameter,
					 const char *description);

/* Call this one to parse the arguments in argv. Note that it modifies both
   argc and argv, eliminating all options and leaving only non-options, in
   the same order they were passed in. --help and --version are managed here.*/
void parse_command_line (int *argc, char **argv);

/* Functions to access the values passed via options. These can be called
   only after parse_command_line(). */
char *short_option_to_value (char name);
char *long_option_to_value (char *name);

/* This function shows the value of all options. It's useful for debugging. */
void show_options_status ();

/* These utility functions are tipically useful when working with the command
   line, although they are not directly related with it: */

/* Returns nonzero iff the filename 'name' has the extension 'extension': */
int has_extension(const char *name, const char *extension);
int has_some_extension(const char *name); /* if have *any* extension */

/* filename may contain or not contain directories.
   Returns a pointer to a static buffer. Not reentrant. */
char* remove_path_from_filename(char* filename);

/* Replace the extension in the filename 'name' with 'new_extension'.
   Warning: this uses a static buffer of fixed size MAX_FILENAME_LENGTH.
   If the filename is too long then the program aborts with an error
   message. Not reentrant. */
char* replace_extension(const char *name, const char *new_extension);

/* Replaces also the dot with e. */
char* replace_extension_with_part_of_filename(const char* s,const char* e);

/* Replace the dot and extension in the filename 'name' with
   'new_terminal_part'. If a dot is desired in the new name it must be
   explicitly written in 'new_terminal_part'.
   Warning: this uses a static buffer of fixed size MAX_FILENAME_LENGTH.
   If the filename is too long then the program aborts with an error
   message. Not reentrant. */
char* replace_dot_and_extension(const char *name, const char *new_terminal_part);

void fatal(const char* message);
/* Show the message, suggest to use --help and exit with failure. */
void command_line_fatal(const char* message);

/* Usually you don't need to call these directly: */
void show_help ();		/* The effect of passing --help */
void show_version ();		/* The effect of passing --version */

#endif /* #ifndef COMMAND_LINE_H_ */
