/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2003 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */


#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include "search_path.h"
#include "malloc.h"


search_path_t create_search_path(){
  search_path_t r;

  /* Create structure: */
  r = xmalloc(sizeof(struct search_path));
  
  /* Initialize fields: */
  r->directories_no = 0;
  r->directories_size = INITIAL_DIRECTORIES_NO;
  
  /* Allocate buffer: */
  r->directories = (char**)
    xmalloc(sizeof(char*) * r->directories_size);

  return r;
}

void append_directory_to_search_path(char* directory, search_path_t p){
  int length = strlen(directory);

  /* Increment p->directories_no and allocate more space if needed: */
  if(p->directories_no == p->directories_size){
    p->directories_size *= 2;
    p->directories = xrealloc(p->directories,
			      sizeof(char*) * p->directories_size);
  }
  p->directories_no++;

  /* Insert directory into the structure: */
  p->directories[p->directories_no - 1] = (char*)
    xmalloc(sizeof(char) * (length + 1));
  strcpy(p->directories[p->directories_no - 1],
	 directory);
}

void append_path_to_search_path(char* path, search_path_t p){
  char* copy_of_path; 
  char* directory;
  /* Make a writable copy of path: */
  copy_of_path = (char*) xmalloc(sizeof(char) * (strlen(path) + 1));
  strcpy(copy_of_path, path);
  
  /* Tokenize copy_of_path into directories, appending each directory to p: */
  directory = strtok(copy_of_path, DIRECTORY_SEPARATORS);
  while(directory != NULL){
    if(strcmp(directory, "")) /* 'empty' directories are ignored */
      append_directory_to_search_path(directory, p);
    directory = strtok(NULL, DIRECTORY_SEPARATORS);    
  } /* while */
  
  free(copy_of_path);
}

//#define DEBUG

/* If file_name already contains a path then return a copy of file_name,
   without looking at path; else sarche for the first readable file (actual
   file, not directory) in each directory of the path, following the order 
   in the structure.
   Returns a malloc()'d string holding the full path which the caller
   must free(), or NULL if no suitable file exists. */
char* search_for_file_in_search_path(char* file_name,
				     search_path_t path){
  int length_of_file_name = strlen(file_name);
  int i;
  FILE* f;

  /* If file_name contains a path then just return a copy of file_name: */
  if(strchr(file_name, '/') != NULL){
    char* r = (char*) 
      xmalloc(sizeof(char) * (length_of_file_name + 1));
    strcpy(r, file_name);
    return r;
  }

#ifdef DEBUG
  fprintf(stderr, "search_for_file_in_search_path(): searching for %s\n", file_name);
#endif /* #ifdef DEBUG */

  /* For each directory D in the path try to open D/file_name; return a copy
     of the full path to the first existing one: */
  for(i = 0; i < path->directories_no; i++){
    char* directory = path->directories[i];
    int length_of_directory = strlen(directory);
    char* full_path;

    /* Allocate and fill full_path: */
    full_path = (char*)
      xmalloc(sizeof(char) * (length_of_directory + length_of_file_name + 2));
    sprintf(full_path, "%s/%s", directory, file_name);
    
#ifdef DEBUG
    fprintf(stderr, "search_for_file_in_search_path(): trying %s\n", full_path);
#endif /* #ifdef DEBUG */
    
    /* Try and open the file; return full_path on success: */
    errno = 0; /* Reset errno: we are going to test for errors later */
    f = fopen(full_path, "r");
    if((f != NULL) && (errno == 0)){ /* using errno avoids opening directories*/
#ifdef DEBUG
      fprintf(stderr, "search_for_file_in_search_path(): found\n");
#endif /* #ifdef DEBUG */
      fclose(f);
      return full_path;
    }
    errno = 0; /* Reset errno, for the next iteration */

    /* De-allocate full_path: */
    free(full_path);
  } /* for */
#ifdef DEBUG
  fprintf(stderr, "search_for_file_in_search_path(): not found\n");
#endif /* #ifdef DEBUG */

  /* If we arrived here then there is no suitable file: */
  return NULL;
}

/* Calls search_for_file_in_search_path(), and aborts when
   search_for_file_in_search_path() returns NULL; else
   search_for_file_or_fail() returns the value returned by
   search_for_file_in_search_path() */
char* search_for_file_or_fail(char* file_name,
			      search_path_t path){
  char* r = search_for_file_in_search_path(file_name, path);
  
  if(r == NULL){
    fprintf(stderr, "About %s:\n", file_name);
    //dump_search_path(path);
    fatal("could not find file");
  }
  return r;
}

void destroy_search_path(search_path_t path){
  int i;
  /* Destroy all (initialized) directories: */
  for(i = 0; i < path->directories_no; i++)
    free(path->directories[i]);
  
  /* Free the directores array: */
  free(path->directories);
  
  /* Free the structure: */
  free(path);
}

/* Useful for debugging: */
void dump_search_path(search_path_t p){
  int i;

  fprintf(stderr, "Search path: --------------------------------\n");
  for(i = 0; i < p->directories_no; i++)
    fprintf(stderr, "%2i. %s\n", i, p->directories[i]);
  fprintf(stderr, "---------------------------------------------\n");
}

/* Returns a malloc()'d string holding the search directories, separated
   by DIRECTORY_SEPARATORS: */
char* search_path_to_string(search_path_t path){
  int i, size = 1 /* terminator */, start_of_substring = 0;
  char *r;
  
  /* Compute the size of the string: */
  for(i = 0; i < path->directories_no; i++)
    size +=
      strlen(path->directories[i]) + strlen(DIRECTORY_SEPARATORS);

  /* Create and fill the string: */
  r = (char*) xmalloc(size);
  for(i = 0; i < path->directories_no; i++){
    sprintf(r + start_of_substring, "%s", path->directories[i]);
    start_of_substring += strlen(path->directories[i]);
    if(i < path->directories_no - 1){
      /* if this iteration is not the last one add the separator: */
      sprintf(r + start_of_substring, "%s", DIRECTORY_SEPARATORS);
      start_of_substring += strlen(DIRECTORY_SEPARATORS);      
    } /* if */
  } /* for */

  return r;
}


// To do: remove this main()
/*
int main(){
  search_path_t p;
  char* s;

  p = create_search_path();
  
  append_path_to_search_path(".:/tmp:/usr/gnome/lib:/usr/lib", p);

  s = search_for_file_in_search_path("usr", p);

  fprintf(stderr, ">%s< in \"%s\"\n", s, search_path_to_string(p));

  dump_search_path(p);

  destroy_search_path(p);
  return 0;
}
*/
