/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2002 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */

#ifndef _ENVIRONMENT_H_
#define _ENVIRONMENT_H_

#include "types.h"
#include "epsilon.h"

typedef int environment_id;
typedef int global_environment_id;

void init_environment ();
void destroy_environment ();

environment_id bind_not_refreshable (environment_id e, char *identifier,
				     int pos, type_t type);
environment_id bind_not_refreshable_with_precedence (environment_id e,
						     char *identifier,
						     int pos, type_t type,
						     int precedence);
environment_id bind_refreshable (environment_id e, char *identifier, int pos,
				 type_t type);
environment_id bind_refreshable_with_precedence (environment_id e,
						 char *identifier, int pos,
						 type_t type, int precedence);

/* Returns non-zero on success: */
int declare_global (char *identifier, type_t type);
int declare_global_with_precedence (char *identifier, type_t type,
				    int precedence);
int has_been_declared (char *identifier);	/* in this module, by default */
int has_been_defined (char *identifier);	/* in this module, by default */
int has_been_declared_or_defined (char *identifier);	/* in this module, by default */

/* Returns non-zero on success: */
int bind_abstract_type_name (char *identifier);
int bind_concrete_type_name (char *identifier);
int bind_concrete_type_constructor (char *identifier);
int bind_exception_name (char *unqualified_name);

/* Returns non-zero on success: */
int bind_global (char *identifier, type_t type);
int bind_global_with_precedence (char *identifier, type_t type,
				 int precedence);
void dump_global_environment ();

void push_environment (environment_id e);
environment_id top_environment ();
environment_id previous_environment ();
environment_id pop_environment ();

int next_free_position (environment_id e);

/* When these functions find a global they write '-1' into *jumps_no and the
   address into *pos.
   Return 0  on success
          -1 on error
	  -2 on ambiguous reference to global
 */
int lookup (char *identifier, int *jumps_no, int *pos, type_t * t);	/* returns 0 iff succeeds */
int lookup_with_precedence (char *identifier, int *jumps_no, int *pos, type_t * t, int *precedence);	/* returns 0 iff succeeds */

void create_environment_structures ();
void free_environment_structures ();

/* Manage a stack whose element holds the number of the bindings in a
   local environment: */
void push_bindings ();
void pop_bindings ();
int *current_bindings ();

void dump_environment ();

int are_there_undefined_but_declared_identifiers (char *module_name);
char *first_undefined_identifier (char *module_name);

/* Qualify using the current module: */
char *qualify_with_this_module (char *identifier);

char *qualify (char *identifier);	/* Qualify with the current module if no better option exists.  
					   In case of ambiguities use the most recently defined. */


char *qualify_abstract_type_name (char *identifier);	/* Qualify with the current module if no better option exists.
							   In case of ambiguities use the most recently defined. */

char *qualify_concrete_type_name (char *identifier);	/* Qualify with the current module if no better option exists.
							   In case of ambiguities use the most recently defined. */
char *qualify_concrete_type_constructor (char *identifier);	/* Qualify with the current module if no better option exists.
								   In case of ambiguities use the most recently defined. */
char *qualify_exception (char *identifier);	/* Qualify with the current module if no better option exists.
						   In case of ambiguities use the most recently defined. */

int is_qualified (char *identifier);	/* I assume the identifier to be valid */
char *bare_identifier (char *maybe_qualified_identifier);	/* The identifier may not be qualified */
char *bare_module (char *maybe_qualified_identifier);	/* The identifier may not be qualified */


void push_pattern_matching ();
void pop_pattern_matching ();
int current_pattern_matching ();

void increment_pattern_matching_case_counter ();
int current_pattern_matching_case_counter ();

int is_current_pattern_matching_exhaustive ();
void make_current_pattern_matching_exhaustive ();

/* Returns:

   0  on success
   -1 on type mismatch */
int memoize_pattern_matching_case (int encoding, int cases_no);
int is_pattern_matching_case_present (int encoding);

int push_exception_handling ();	/* returns the new exception handling number */
int current_exception_handling ();
void pop_exception_handling ();
void increment_exception_handling_cases_number ();
int current_exception_handling_cases_number ();


#endif /* #ifndef _ENVIRONMENT_H_ */
