/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2002 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */

#include "term.h"

#include <stdio.h>
#include <malloc.h>
#include <stdlib.h>
#include <stdarg.h>

#include "../common/string_map.h"
#include "epsilon.h"

enum type_of_term_enum
{
  constructor_type,
  variable_type
};


struct constructor_struct
{
  char name[IDENTIFIER_LENGTH + 1];
  int ariety;
}
 *constructors_table;

#define CONSTRUCTORS_TABLE_INITIAL_SIZE 256
int constructors_table_size;
constructor_t next_constructor;


/* The variables table is only created just before unification
   begins and is destroyed right after its end. */
struct variable_struct
{
  char name[IDENTIFIER_LENGTH + 1];
}
 *variables_table;

#define VARIABLES_TABLE_INITIAL_SIZE 1
int variables_table_size;
variable_t next_variable;


struct term_struct
{
  enum type_of_term_enum type_of_term;
  union
  {
    struct
    {
      constructor_t constructor;
      term_t *subterms;
    }
    non_variable;
    variable_t variable;
  }
  variable_or_not_variable;	/* union */
  term_t set;			/* to implement unification */
}
 *terms_table;

#define TERMS_TABLE_INITIAL_SIZE 16
int terms_table_size;
term_t next_term;

constructor_t error_c;
term_t error;			/* a special term */

void
init_terms ()
{
  constructors_table =
    (struct constructor_struct *) malloc (sizeof (struct constructor_struct) *
					  CONSTRUCTORS_TABLE_INITIAL_SIZE);
  constructors_table_size = CONSTRUCTORS_TABLE_INITIAL_SIZE;
  next_constructor = 0;

  variables_table =
    (struct variable_struct *) malloc (sizeof (struct variable_struct) *
				       VARIABLES_TABLE_INITIAL_SIZE);
  variables_table_size = VARIABLES_TABLE_INITIAL_SIZE;
  next_variable = 0;

  terms_table =
    (struct term_struct *) malloc (sizeof (struct term_struct) *
				   TERMS_TABLE_INITIAL_SIZE);
  terms_table_size = TERMS_TABLE_INITIAL_SIZE;
  next_term = 0;

  error = term (error_c = constructor ("<error>", 0));
}

void
free_terms ()
{
  int i;

  free (constructors_table);
  free (variables_table);
  for (i = 0; i < next_term; i++)
    if (terms_table[i].type_of_term == constructor_type)
      free (terms_table[i].variable_or_not_variable.non_variable.subterms);
  free (terms_table);
}

constructor_t
constructor (char *name, int ariety)
{
  constructor_t r;

  if (next_constructor == constructors_table_size)
    {
      constructors_table_size *= 2;
      constructors_table =
	(struct constructor_struct *) realloc (constructors_table,
					       constructors_table_size *
					       sizeof (struct
						       constructor_struct));
    }
  r = next_constructor++;
  strncpy (constructors_table[r].name, name, IDENTIFIER_LENGTH);
  constructors_table[r].name[IDENTIFIER_LENGTH] = '\0';
  constructors_table[r].ariety = ariety;

  return r;
}

constructor_t
term_to_constructor (term_t t)
{				/* t must _not_ be a variable */
  return terms_table[t].variable_or_not_variable.non_variable.constructor;
}

/* This is only a test. To do: use an hash table. */
variable_t
lookup_variable (char *name)
{
  variable_t i;

  for (i = 0; i < next_variable; i++)
    if (!strcmp (variables_table[i].name, name))
      return i;

  return -1;
}

/* This is only a test. To do: use an hash table. */
term_t
lookup_variable_term (variable_t v)
{
  term_t i;

  for (i = 0; i < next_term; i++)
    if (terms_table[i].type_of_term == variable_type)
      if (terms_table[i].variable_or_not_variable.variable == v)
	return i;

  return -1;
}

variable_t
variable (char *name)
{
  variable_t r;

  /* If we already have this variable then return the one we stored. */
  if ((r = lookup_variable (name)) != -1)
    return r;

  if (next_variable == variables_table_size)
    {
      variables_table_size *= 2;
      variables_table =
	(struct variable_struct *) realloc (variables_table,
					    variables_table_size *
					    sizeof (struct variable_struct));
    }
  r = next_variable++;
  strncpy (variables_table[r].name, name, IDENTIFIER_LENGTH);
  variables_table[r].name[IDENTIFIER_LENGTH] = '\0';

  return r;
}

variable_t
new_variable (char *base)
{
  static int n = 1;
  char name[20];

  sprintf (name, "%s%i", base, n++);

  return variable (name);
}

term_t
vterm (variable_t v)
{
  term_t r;

  /* If a term made up by this variable is already present then use the one we created before: */
  if ((r = lookup_variable_term (v)) != -1)
    return r;

  if (next_term == terms_table_size)
    {
      terms_table_size *= 2;
      terms_table =
	(struct term_struct *) realloc (terms_table,
					terms_table_size *
					sizeof (struct term_struct));
    }
  r = next_term++;
  terms_table[r].type_of_term = variable_type;
  terms_table[r].variable_or_not_variable.variable = v;
  terms_table[r].set = -1;

  return r;
}

term_t
term (constructor_t c, ...)
{
  va_list variadics;
  term_t r;
  int i;

  va_start (variadics, c);

  if (next_term == terms_table_size)
    {
      terms_table_size *= 2;
      terms_table =
	(struct term_struct *) realloc (terms_table,
					terms_table_size *
					sizeof (struct term_struct));
    }
  r = next_term++;
  terms_table[r].type_of_term = constructor_type;
  terms_table[r].variable_or_not_variable.non_variable.constructor = c;
  terms_table[r].variable_or_not_variable.non_variable.subterms =
    (term_t *) malloc (sizeof (term_t) * constructors_table[c].ariety);
  terms_table[r].set = -1;

  for (i = 0; i < constructors_table[c].ariety; i++)
    terms_table[r].variable_or_not_variable.non_variable.subterms[i] =
      va_arg (variadics, term_t);

  va_end (variadics);
  return r;
}

/* Only used in the unification algorithm: */
term_t
skeletal_term (constructor_t c)
{
  term_t r;

  if (next_term == terms_table_size)
    {
      terms_table_size *= 2;
      terms_table =
	(struct term_struct *) realloc (terms_table,
					terms_table_size *
					sizeof (struct term_struct));
    }
  r = next_term++;
  terms_table[r].type_of_term = constructor_type;
  terms_table[r].variable_or_not_variable.non_variable.constructor = c;
  terms_table[r].variable_or_not_variable.non_variable.subterms =
    (term_t *) malloc (sizeof (term_t) * constructors_table[c].ariety);
  terms_table[r].set = -1;
  /* subterms are not initialized */

  return r;
}

void
output_term (FILE * f, term_t t)
{
  //fprintf(f,"<%i>",t);
  switch (terms_table[t].type_of_term)
    {
    case variable_type:
      {
	fprintf (f, "%s",
		 variables_table[terms_table[t].variable_or_not_variable.
				 variable].name);

	//fprintf(f,"<%i>",terms_table[t].variable_or_not_variable.variable);
	break;
      }

    case constructor_type:
      {
	int i;

	fprintf (f, "%s",
		 constructors_table[terms_table[t].variable_or_not_variable.
				    non_variable.constructor].name);
	if (constructors_table
	    [terms_table[t].variable_or_not_variable.non_variable.
	     constructor].ariety != 0)
	  {
	    fprintf (f, "(");
	    for (i = 0;
		 i <
		 constructors_table[terms_table[t].variable_or_not_variable.
				    non_variable.constructor].ariety; i++)
	      {
		output_term (f,
			     terms_table[t].variable_or_not_variable.
			     non_variable.subterms[i]);
		if (i !=
		    constructors_table[terms_table[t].
				       variable_or_not_variable.non_variable.
				       constructor].ariety - 1)
		  fprintf (f, ", ");
	      }			/* for */
	    fprintf (f, ")");
	  }			/* if */
	//fprintf(f,"<%i>",t);

	break;
      }

    default:
      {
	fprintf (stderr, "output_term(): This cannot happen\n");
	exit (EXIT_FAILURE);
      }
    }				/* switch */

  /*
     if(terms_table[t].set!=-1){
     fprintf(f,"<");output_term(f,terms_table[t].set);fprintf(f,">");
     }
   */
}

term_t
subterm (term_t t, int n)
{
  return terms_table[t].variable_or_not_variable.non_variable.subterms[n];
}

term_t
find (term_t x)
{
  if (terms_table[x].set == -1)
    return x;
  else
    return find (terms_table[x].set);
}

int
is_nullary_constructor (term_t x)
{
  if (terms_table[x].type_of_term != constructor_type)
    return 0;
  else
    return constructors_table[terms_table[x].variable_or_not_variable.
			      non_variable.constructor].ariety == 0;
}

int
are_same_nullary_constructor (term_t x, term_t y)
{
  if ((!is_nullary_constructor (x)) || (!is_nullary_constructor (y)))
    return 0;
  else
    return terms_table[x].variable_or_not_variable.non_variable.constructor ==
      terms_table[y].variable_or_not_variable.non_variable.constructor;
}

int
is_nonnullary_constructor (term_t x)
{
  if (terms_table[x].type_of_term != constructor_type)
    return 0;
  else
    return constructors_table[terms_table[x].variable_or_not_variable.
			      non_variable.constructor].ariety > 0;
}

int
are_same_nonnullary_constructor (term_t x, term_t y)
{
  if ((!is_nonnullary_constructor (x)) || (!is_nonnullary_constructor (y)))
    return 0;
  else
    return terms_table[x].variable_or_not_variable.non_variable.constructor ==
      terms_table[y].variable_or_not_variable.non_variable.constructor;
}

int
is_variable (term_t x)
{
  return terms_table[x].type_of_term == variable_type;
}

/* Adapted from [Aho, Sethi, Ullman]: p.378 and succ. */

/* Merge the equivalence classes containing x and y: */
term_t
union_function (term_t m, term_t n)
{
  term_t rep_m = find (m), rep_n = find (n);

  if (!is_variable (rep_m))
    return terms_table[rep_n].set = rep_m;
  else
    // If n is a non-variable I must choose n, else both n and m can do; so I always use n.
    return terms_table[rep_m].set = rep_n;
}

term_t
recursive_representative (term_t t_,
			  term_t * terms_in_the_root_to_node_path_stack,
			  int terms_in_the_root_to_node_path_pointer)
{
  term_t t = find (t_);

  if (is_variable (t))
    return t;
  else if (is_nullary_constructor (t))
    return t;
  else
    {				/* t is a non-nullary constructor */
      term_t r =
	skeletal_term (terms_table[t].variable_or_not_variable.non_variable.
		       constructor);
      term_t subterm;
      int i;

      /* Have we already seen this term in the root->...->t_ path? */
      for (i = 0; i < terms_in_the_root_to_node_path_pointer; i++)
	if (terms_in_the_root_to_node_path_stack[i] == t)
	  return error;

      /* Remember we visited this node: */
      terms_in_the_root_to_node_path_stack
	[terms_in_the_root_to_node_path_pointer] = t;

      /* Use the recursive representatives of subterms to compose the recursive representative of the whole term: */
      for (i = 0;
	   i <
	   constructors_table[terms_table[t].variable_or_not_variable.
			      non_variable.constructor].ariety; i++)
	{
	  subterm =
	    recursive_representative (terms_table[t].variable_or_not_variable.
				      non_variable.subterms[i],
				      terms_in_the_root_to_node_path_stack,
				      terms_in_the_root_to_node_path_pointer +
				      1);
	  if (subterm != error)
	    terms_table[r].variable_or_not_variable.non_variable.subterms[i] =
	      subterm;
	  else
	    return error;
	}

      return r;
    }
}

int
does_variable_occour_in_term (variable_t v, term_t t)
{
  switch (terms_table[t].type_of_term)
    {
    case variable_type:
      {
	if (terms_table[t].variable_or_not_variable.variable == v)
	  return 1;
	else
	  return 0;

	break;
      }
    case constructor_type:
      {
	int i;

	for (i = 0;
	     i <
	     constructors_table[terms_table[t].variable_or_not_variable.
				non_variable.constructor].ariety; i++)
	  if (does_variable_occour_in_term
	      (v,
	       terms_table[t].variable_or_not_variable.non_variable.
	       subterms[i]))
	    return 1;

	return 0;

	break;
      }
    default:
      {
	fprintf (stderr,
		 "does_variable_occour_in_term(): this cannot happen!\n");
	exit (EXIT_FAILURE);
      }
    }				/* switch */
}

term_t
unify_after_initialization (term_t m, term_t n)
{
  /* Find representative nodes for m and n: */
  term_t s = find (m), t = find (n);

  if ((s == t) || are_same_nullary_constructor (s, t))
    return s;
  else if (are_same_nonnullary_constructor (s, t))
    {
      int i;
      term_t r;

      /* Apply 'union': */
      union_function (s, t);

      /* Try unifying the pairs of subterms: return non-error iff they all match. */
      r =
	skeletal_term (terms_table[s].variable_or_not_variable.non_variable.
		       constructor);
      for (i = 0;
	   i <
	   constructors_table[terms_table[s].variable_or_not_variable.
			      non_variable.constructor].ariety; i++)
	if ((terms_table[r].variable_or_not_variable.non_variable.
	     subterms[i] =
	     (unify_after_initialization
	      (terms_table[s].variable_or_not_variable.non_variable.
	       subterms[i],
	       terms_table[t].variable_or_not_variable.non_variable.
	       subterms[i]))) == error)
	  return error;

      return r;			/* Ok, subterms unifications succeeded. */
    }
  else if (is_variable (s) || is_variable (t))
    {
      /* Ok, match a variable with something: */
      return union_function (s, t);
    }
  else
    /* interior nodes with different constructors */
    return error;
}

void
initialize (term_t x)
{
  int i;

  terms_table[x].set = -1;
  if (terms_table[x].type_of_term == constructor_type)
    for (i = 0;
	 i <
	 constructors_table[terms_table[x].variable_or_not_variable.
			    non_variable.constructor].ariety; i++)
      initialize (terms_table[x].variable_or_not_variable.non_variable.
		  subterms[i]);
}

term_t
unify (term_t m, term_t n)
{
  term_t unified_term;
  term_t *terms_in_the_root_to_node_path_stack;

  initialize (m);
  initialize (n);
  unified_term = unify_after_initialization (m, n);

  /* Create the stack (larger than needed): */
  terms_in_the_root_to_node_path_stack =
    (term_t *) malloc (sizeof (term_t) * next_term);

  unified_term =
    recursive_representative (unified_term,
			      terms_in_the_root_to_node_path_stack, 0);

  /* Destroy the stack: */
  free (terms_in_the_root_to_node_path_stack);

  return unified_term;
}

int
are_terms_equal (term_t x, term_t y)
{
  int i;

  /* Any term equals itself */
  if (x == y)
    return 1;

  /* A variable is different from a term with constructor */
  if (terms_table[x].type_of_term != terms_table[x].type_of_term)
    return 0;

  if (terms_table[x].type_of_term == variable_type)
    /* x and y are both variables */
    return terms_table[x].variable_or_not_variable.variable == terms_table[y].variable_or_not_variable.variable;	/* variables only appear once in the variables table */
  else
    {
      /* x and y are both contructors with parameters */

      /* If x and y have different constructors then they are different */
      if (terms_table[x].variable_or_not_variable.non_variable.constructor !=
	  terms_table[y].variable_or_not_variable.non_variable.constructor)
	return 0;

      /* x and y have the same constructor; test subterms: */
      for (i = 0;
	   i <
	   constructors_table[terms_table[x].variable_or_not_variable.
			      non_variable.constructor].ariety; i++)
	if (!are_terms_equal
	    (terms_table[x].variable_or_not_variable.non_variable.subterms[i],
	     terms_table[y].variable_or_not_variable.non_variable.
	     subterms[i]))
	  return 0;
      return 1;
    }
}

/* Used by is_term_more_general_than_or_equal_to.
   Here I assume m and n unify */
int
is_term_more_general_than_or_equal_to_dont_normalize (term_t m, term_t n)
{				// To do: (A -> B) <=> (A -> A)
  if (is_variable (m))
    {
      if (!is_variable (n))
	return 1;
      else
	{			/* m and n are both variables */
	  return 1;
	}
    }
  else
    {				/* m is not a variable */
      if (is_variable (n))
	return 0;
      else
	{			/* m and n are both non-variables (and they share the constructor) */
	  int i;
	  for (i = 0;
	       i <
	       constructors_table[terms_table[m].variable_or_not_variable.
				  non_variable.constructor].ariety; i++)
	    if (is_term_more_general_than_or_equal_to_dont_normalize
		(terms_table[m].variable_or_not_variable.non_variable.
		 subterms[i],
		 terms_table[n].variable_or_not_variable.non_variable.
		 subterms[i]) == 0)
	      return 0;
	  return 1;
	}
    }
}

int
is_term_more_general_than_or_equal_to (term_t m, term_t n)
{
  //fprintf(stderr,"is_term_more_general_than_or_equal_to: init\n");
  //fprintf(stderr,"[m (%i) = ",m);output_type(stderr,m);fprintf(stderr,"]\n");
  //fprintf(stderr,"[n (%i) = ",n);output_type(stderr,n);fprintf(stderr,"]\n");

  if (is_error (unify (m, n)))
    {
      //fprintf(stderr,"is_term_more_general_than_or_equal_to: done\n");
      return 0;
    }
  else
    {
      m = normalize_term (m);
      n = normalize_term (n);

      //fprintf(stderr,"is_term_more_general_than_or_equal_to: done\n");
      return is_term_more_general_than_or_equal_to_dont_normalize (m, n);
    }
}

int
are_terms_equal_excluding_variables (term_t m, term_t n)
{
  ///////////////////////////////////////
  //fprintf(stderr,"[[are_terms_equal_excluding_variables():]]\n");
  //fprintf(stderr,"[[m e` ");output_type(stderr,normalize_term(m));fprintf(stderr,"]\n");
  //fprintf(stderr,"[[n e` ");output_type(stderr,normalize_term(n));fprintf(stderr,"]\n");
  ///////////////////////////////////////

  return are_terms_equal (normalize_term (m), normalize_term (n));
}

term_t
canonicize_with_map (term_t t, struct string_map * map, char *prefix, int *x)
{
  term_t r;

  switch (terms_table[t].type_of_term)
    {
    case variable_type:
      {
	//fprintf(stderr,"Searching for \"%s\" -> ",variables_table[terms_table[t].variable_or_not_variable.variable].name);

	char new_identifier[IDENTIFIER_LENGTH + 1];
	char *lookup_result =
	  access_string_map (map,
			     variables_table[terms_table[t].
					     variable_or_not_variable.
					     variable].name);

	if (lookup_result == NULL)
	  {
	    //fprintf(stderr,"found %p\n",lookup_result);
#define LETTERS_NO 26
	    if ((*x) < LETTERS_NO)
	      sprintf (new_identifier, "%s%c", prefix, (char) ('a' + *x));
	    else
	      sprintf (new_identifier, "%sa%i", prefix,
		       (*x) - LETTERS_NO + 1);

	    /* Quite dirty hack: here I assume I'll call this function with &next_variable as x: */
	    if ((x == &next_variable)
		&& (next_variable == variables_table_size))
	      {
		variables_table_size *= 2;
		variables_table = (struct variable_struct *)
		  realloc (variables_table,
			   variables_table_size *
			   sizeof (struct variable_struct));
	      }
	    (*x)++;
	    insert_into_string_map (map,
				    variables_table[terms_table[t].
						    variable_or_not_variable.
						    variable].name,
				    new_identifier);
	    //fprintf(stderr,"{MEMOIZED \"%s\" AS \"%s\"}\n",variables_table[terms_table[t].variable_or_not_variable.variable].name,new_identifier);
	    r = vterm (variable (new_identifier));
	  }
	else
	  {
	    //fprintf(stderr,"found \"%s\"\n",lookup_result);
	    r = lookup_variable_term (lookup_variable (lookup_result));
	    //printf("{SEARCHED \"%s\", FOUND \"%s\"}\n",variables_table[terms_table[t].variable_or_not_variable.variable].name,lookup_result);
	  }
	break;
      }
    case constructor_type:
      {
	int i;

	if (constructors_table
	    [terms_table[t].variable_or_not_variable.non_variable.
	     constructor].ariety == 0)
	  return t;		/* Don't duplicate terms with no subterms */

	r =
	  skeletal_term (terms_table[t].variable_or_not_variable.non_variable.
			 constructor);
	for (i = 0;
	     i <
	     constructors_table[terms_table[t].variable_or_not_variable.
				non_variable.constructor].ariety; i++)
	  terms_table[r].variable_or_not_variable.non_variable.subterms[i] =
	    canonicize_with_map (terms_table[t].variable_or_not_variable.
				 non_variable.subterms[i], map, prefix, x);
	break;
      }
    default:
      {
	fprintf (stderr, "canonicize_with_map(): this cannot happen!\n");
	exit (EXIT_FAILURE);
      }
    }
  return r;
}

term_t
normalize_term (term_t t)
{
  term_t r;
  int first = 0;
  struct string_map *map =
    create_string_map (sizeof (char) * (IDENTIFIER_LENGTH + 1)); // updated in 2003
  r = canonicize_with_map (t, map, "'", &first);
  destroy_string_map (map);
  return r;
}

term_t
freshen_term (term_t t)
{
  term_t r;
  struct string_map *map =
    create_string_map (sizeof (char) * (IDENTIFIER_LENGTH + 1)); // updated in 2003
  r = canonicize_with_map (t, map, "'", &next_variable);
  destroy_string_map (map);
  return r;
}

/* This assumes t1 and t2 can unify with success and t2 is less general than t1. */
/* This is very O(last_term), but works very well. */
void
make_terms_equal (term_t t1, term_t t2)
{
  if (terms_table[t1].type_of_term == variable_type)
    {
      variable_t variable_to_be_replaced =
	terms_table[t1].variable_or_not_variable.variable;
      int i;
      for (i = 0; i < next_term; i++)
	if ((terms_table[i].type_of_term == variable_type) &&
	    (terms_table[i].variable_or_not_variable.variable ==
	     variable_to_be_replaced))
	  {
	    //fprintf(stderr,"replacing ");output_term(stderr,i);fprintf(stderr," with ");
	    //output_term(stderr,t2);fprintf(stderr,"...\n");

	    if (terms_table[t2].type_of_term == variable_type)
	      terms_table[i].variable_or_not_variable.variable =
		terms_table[t2].variable_or_not_variable.variable;
	    else
	      {
		terms_table[i].type_of_term = terms_table[t2].type_of_term;
		terms_table[i].variable_or_not_variable.non_variable.
		  constructor =
		  terms_table[t2].variable_or_not_variable.non_variable.
		  constructor;
		terms_table[i].variable_or_not_variable.non_variable.
		  subterms =
		  (term_t *) malloc (sizeof (term_t) *
				     constructors_table[terms_table[t2].
							variable_or_not_variable.
							non_variable.
							constructor].ariety);
		memcpy (terms_table[i].variable_or_not_variable.non_variable.
			subterms,
			terms_table[t2].variable_or_not_variable.non_variable.
			subterms,
			sizeof (term_t) *
			constructors_table[terms_table[t2].
					   variable_or_not_variable.
					   non_variable.constructor].ariety);
	      }
	    //fprintf(stderr,"replaced ");output_term(stderr,i);fprintf(stderr," with ");
	    //output_term(stderr,t2);fprintf(stderr,"...\n");
	  }			//if
    }
  else
    {				// terms_table[t1].type_of_term==constructor_type
      int i;
      for (i = 0;
	   i <
	   constructors_table[terms_table[t2].variable_or_not_variable.
			      non_variable.constructor].ariety; i++)
	make_terms_equal (terms_table[t1].variable_or_not_variable.
			  non_variable.subterms[i],
			  terms_table[t2].variable_or_not_variable.
			  non_variable.subterms[i]);
    }
}

void
make_term_unify_with_term (term_t t1, term_t t2)
{
  term_t t1_unify_t2 = unify (t1, t2);
  /*  
     if(t1_unify_t2==error){
     *t1=error;
     return;
     }
   */

  /* Ok, unification has success. */
  make_terms_equal (t1, t1_unify_t2);
}

int
is_error (term_t t)
{
  if (terms_table[t].type_of_term == variable_type)
    return 0;
  else
    return terms_table[t].variable_or_not_variable.non_variable.constructor ==
      error_c;
}

int
is_constructed_with (term_t t, constructor_t c)
{
  if (terms_table[t].type_of_term != constructor_type)
    return 0;
  else
    return terms_table[t].variable_or_not_variable.non_variable.constructor ==
      c;
}

int
are_subterms_equal (int ariety, term_t * subterms_1, term_t * subterms_2)
{
  int i;

  for (i = 0; i < ariety; i++)
    if (!are_terms_equal (subterms_1[i], subterms_2[i]))
      return 0;
  return 1;
}

term_t
latest_term ()
{
  return next_term - 1;
}

void
replace_term_with_term (term_t t1, term_t t2)
{
  if (terms_table[t1].type_of_term == constructor_type)
    free (terms_table[t1].variable_or_not_variable.non_variable.subterms);

  terms_table[t1].type_of_term = terms_table[t2].type_of_term;
  if (terms_table[t2].type_of_term == constructor_type)
    {
      int i;

      terms_table[t1].variable_or_not_variable.non_variable.constructor =
	terms_table[t2].variable_or_not_variable.non_variable.constructor;
      terms_table[t1].variable_or_not_variable.non_variable.subterms =
	(term_t *) malloc (sizeof (term_t) *
			   constructors_table[terms_table[t2].
					      variable_or_not_variable.
					      non_variable.constructor].
			   ariety);
      for (i = 0;
	   i <
	   constructors_table[terms_table[t2].variable_or_not_variable.
			      non_variable.constructor].ariety; i++)
	terms_table[t1].variable_or_not_variable.non_variable.subterms[i] =
	  terms_table[t2].variable_or_not_variable.non_variable.subterms[i];
    }
  else
    terms_table[t1].variable_or_not_variable.variable =
      terms_table[t2].variable_or_not_variable.variable;
}

/*/
int main(){
  init_terms();

  constructor_t function_c=constructor("function",2);
  constructor_t pair_c=constructor("pair",2);
  constructor_t list_c=constructor("list",1);
  constructor_t int_c=constructor("int",0);
  constructor_t bool_c=constructor("bool",0);

  term_t x_t=vterm(new_variable("type"));
  term_t y_t=vterm(new_variable("type"));
  term_t z_t=vterm(new_variable("type"));
  term_t t_t=vterm(new_variable("type"));
  
  term_t t1=x_t;//term(function_c, x_t, z_t);
  term_t t2=y_t;//term(function_c,term(list_c,y_t),term(list_c,x_t));

  term_t t3=term(function_c,x_t,y_t);
  term_t t4=term(function_c,y_t,term(int_c));

  term_t t5=vterm(x_t);
 
  output_term(stdout,t1);
  printf("\n\tunify\n");
  output_term(stdout,t2);
  printf("\n--------------------------------------------------------------\n[");
  t2=unify(t1,t2);
  output_term(stdout,t2);//unify(t1,t5));
  return 0;
  printf("]\n~\nREFRESHED  -> [");
  output_term(stdout,freshen_term(t2));//unify(t1,t5)));
  printf("]\n~\nNORMALIZED -> [");
  output_term(stdout,normalize_term(t2));//unify(t1,t5)));
  printf("]\n~\nAGAIN:     -> [");
  output_term(stdout,t2);
  printf("]\n");
  printf("\n--------------------------------------------------------------\n"
         "Unificazione distruttiva\n--------------------------------------------------------------\n");
  make_term_unify_with_term(t1,t2);
  output_term(stdout,t1);
 
  free_terms();
}
/**/
