/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2002, 2003 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */

#ifndef TYPES_H_
#define TYPES_H_

#include <stdio.h>
#include "term.h"

typedef term_t type_t;

extern type_t type_error;	/* a special type */
extern type_t void_type, integer_type, float_type, boolean_type, string_type,
  character_type, c_type;
extern type_t tuple_terminator_type;

void initialize_types_table ();
void destroy_types_table ();

int do_types_match (type_t a, type_t b);

type_t new_io_type (type_t x);
type_t new_list_type (type_t x);
type_t new_pair_type (type_t x, type_t y);
type_t new_function_type (type_t x, type_t y);
type_t new_array_type (type_t x);
type_t new_promise_type (type_t x);
type_t new_postfix_operator_type (type_t from, type_t to);
type_t new_infix_operator_left_type (type_t from_1, type_t from_2, type_t to);
type_t new_infix_operator_right_type (type_t from_1, type_t from_2,
				      type_t to);
type_t new_tuple_type (type_t x, type_t tuple);

//type_t new_abstract_type(type_t x);
//type_t new_concrete_type(type_t x);

int is_a_void_type (type_t a);
int is_an_integer_type (type_t a);
int is_a_float_type (type_t a);
int is_a_boolean_type (type_t a);
int is_a_string_type (type_t a);
int is_a_character_type (type_t a);
int is_a_c_type (type_t a);

int is_an_io_type (type_t a);
int is_a_list_type (type_t a);
int is_a_pair_type (type_t a);
int is_a_function_type (type_t a);
int is_an_array_type (type_t a);
int is_a_promise_type (type_t a);
int is_a_postfix_operator_type (type_t a);
int is_an_infix_operator_left_type (type_t a);
int is_an_infix_operator_right_type (type_t a);

int is_an_abstract_type (type_t a);
int is_a_concrete_type (type_t a);

int is_error_type (type_t a);

int arity_of_tuple_type (type_t a);	/* this assumes a is a tuple type */
type_t element_of_tuple_type (type_t x, int n);	/* n starts at 1. */

type_t generic_type (char *name);
type_t new_generic_type ();
int is_a_generic_type (type_t a);

type_t element_type (type_t t, int n);

int is_type_more_general_than_or_equal_to (type_t x, type_t y);
int are_types_equal (type_t m, type_t n);
int are_types_equal_excluding_variables (type_t m, type_t n);

type_t normalize_type (type_t t);
type_t freshen_type (type_t t);

/* When unifying a generic in t1 with a generic in t2 it's the one
   in t2 to be returned as the unified type. */
type_t unify_types (type_t t1, type_t t2);

/* When unifying a generic in t1 with a generic in t2 it's the one
   in t2 to win. The hidden types table is modified in the entry for
   t1: this is a rough emulation of a parameter passed 'by reference';
   t2 instead is passed 'by constant'. */
void make_type_unify_with_type (type_t t1, type_t t2);

type_t new_tuple_type_from_selector (int n, type_t type);
type_t append_to_tuple_type (type_t tuple, type_t t);

void output_type (FILE * f, type_t t);

int is_an_exception_name (char *maybe_qualified_name);

/* Returns 0 on success, -1 on 'already exists' */
int create_exception (char *unqualified_name, type_t parameter);

/* Lookup an exception

   returns:
   0 on success
   -1 for 'not found'
*/
int lookup_exception (char *maybe_qualified_name, type_t * parameter);

int is_exception_defined (char *maybe_qualified_name);

/* Create a new undefined abstract type, to be qualified in _this_ module:
   
   returns:
   0 on success
   -1 for 'already exists'
*/
int create_undefined_abstract_type (char *maybe_qualified_name,
				    type_t formal_parameters
				    /* or tuple_terminator_type */ );


/* Replace every instance to an abstract type with its instantiated definition:

   returns:
   0 on success
   -1 for 'already exists'
   -2 for 'not compatible with declaration'
   -3 for 'warning: not declared'
*/
int define_abstract_type (char *maybe_qualified_name,
			  type_t formal_parameters, type_t definition);


/* 'exists' means 'is declared or is defined' */
int does_abstract_type_exist (char *maybe_qualified_name);

int is_abstract_type_defined (char *maybe_qualified_name);


/* maybe_qualified_name must be defined: */
//type_t instantiate_abstract_type(char* maybe_qualified_name, type_t actual_parameters);

/* Returns the abstract type with actual parameters if a definition is not available,
   else the instantiation or
   -1 for 'parameters don't match'
   -2 for 'undeclared abstract type' */
type_t lookup_abstract_type (char *maybe_qualified_name,
			     type_t actual_parameters);


int is_an_abstract_type_name (char *maybe_qualified_name);

/////////////////////////////////////////////////

/* Compute an instantiated concrete type from a given instantiated case: */
type_t concrete_type_constructor_instance_to_concrete_type (char
							    *maybe_qualified_constructor_name,
							    type_t parameter);


/* Lookup a concrete type case

   returns:
   0 on success
   -1 for 'not found'
*/
int lookup_concrete_type_constructor (char *maybe_qualified_constructor_name,
				      int *concrete_type_index,
				      type_t * definition, int *encoding);

/* Returns nonzero iff the name represents a concrete type */
int is_a_concrete_type_name (char *maybe_qualified_name);

/* Returns nonzero iff the name represents a concrete type constructor */
int is_a_concrete_type_constructor_name (char *maybe_qualified_name);

/* Create a new undefined concrete type, to be qualified in _this_ module:
   
   returns:
   0 on success
   -1 for 'already exists'
*/
int create_undefined_concrete_type (char *maybe_qualified_name,
				    type_t formal_parameters
				    /* or tuple_terminator_type */,
				    int only_declaring);


/* Add a case to the latest concrete type:

   returns:
   0 on success
   -1 for 'case already exists'
*/
int define_concrete_type_case (char *concrete_type_name,
			       char *maybe_qualified_constructor_name,
			       type_t case_definition);


/* Returns the concrete type instantiated with the given parameters for the given case,
   or type_error if actual parameters are not compatible.

   maybe_qualified_name must be defined: */
type_t instantiate_concrete_type_given_a_constructor (char *maybe_qualified_type_name,	// returned
						      char
						      *maybe_qualified_constructor_name,
						      type_t definition);

/* Returns the concrete type instantiated with the given parameters,
   or type_error if actual parameters are not compatible.

   maybe_qualified_name must be defined: */
type_t instantiate_concrete_type (char *maybe_qualified_type_name,
				  type_t actual_parameters);

char *concrete_type_to_name (type_t x);

int concrete_type_to_cases_number (type_t x);	/* x _must_ be concrete */

void concrete_type_to_nth_case (type_t intantiated_type,	/* _must_ be concrete */
				int n,	/* n must be in range */
				char **constructor_name,
				int *encoding, type_t * actual_parameters);

/* ------------------------------------------------------------------------ */

/* The code is returned. Here types don't need to be normalized: */
int memoize_output_procedure_for_concrete_type (type_t t);

/* Here types don't need to be normalized.
   Returns 0 if the procedure does not exist, else its index (!= 0)
*/
int concrete_type_output_procedure_to_code (type_t);

#endif /* #ifndef TYPES_H_ */
