/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2002, 2003 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */


// To do: refactor the code. This can be made *beautiful*, it's a pity there
// is so much code duplication.

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>

#include "eam.h"

#include "../commontext.h"
#include "bytecode.h"
#include "instructions.h"


#define EXECUTABLE_HEADER  "eEXE"	/* this MUST be four characters long */
#define MODULE_HEADER  "eMOD"	/* this MUST be four characters long */
#define LIBRARY_HEADER "eLIB"	/* this MUST be four characters long */

void
destroy_epsilon_executable (struct epsilon_executable *e)
{
  integer_t i;

  /* Free the allocated fields of the epsilon_executable structure: */
  free (e->instructions);

  /* Free the string constants: */
  for (i = 0; i < e->string_constants_no; i++)
    free (e->string_constants[i]);
  free (e->string_constants);

  /* Free the exceptions: */
  for (i = 0; i < e->exceptions_no; i++)
    free (e->exceptions[i]);
  free (e->exceptions);

  /* Free the dynamic C libraries: */
  for (i = 0; i < e->c_libraries_no; i++)
    free (e->c_libraries[i]);
  free (e->c_libraries);

  /* Free the C symbols: */
  for (i = 0; i < e->c_symbols_no; i++)
    free (e->c_symbols[i].symbol_name);
  free (e->c_symbols);

  /* Free the epsilon_executable structure: */
  free (e);
}

void
destroy_epsilon_module (struct epsilon_module *m)
{
  integer_t i;

  /* Free the allocated fields of the epsilon_module structure: */
  free (m->name);
  for (i = 0; i < m->needed_modules_no; i++)
    free (m->needed_modules[i]);
  free (m->needed_modules);
  free (m->instructions);

  /* Free the string constants: */
  for (i = 0; i < m->string_constants_no; i++)
    free (m->string_constants[i]);
  free (m->string_constants);

  /* Free the gloabls: */
  for (i = 0; i < m->globals_no; i++)
    free (m->globals[i]);
  free (m->globals);

  /* Free the exceptions: */
  for (i = 0; i < m->exceptions_no; i++)
    free (m->exceptions[i]);
  free (m->exceptions);

  /* Free the dynamic C libraries: */
  for (i = 0; i < m->c_libraries_no; i++)
    free (m->c_libraries[i]);
  free (m->c_libraries);

  /* Free the C symbols: */
  for (i = 0; i < m->c_symbols_no; i++)
    free (m->c_symbols[i].symbol_name);
  free (m->c_symbols);

  /* Free the epsilon_module structure: */
  free (m);
}

void
destroy_epsilon_library (struct epsilon_library *l)
{
  integer_t i;

  /* Destroy every module: */
  for (i = 0; i < l->modules_no; i++)
    destroy_epsilon_module (l->modules[i]);

  /* Destroy the modules array: */
  free (l->modules);

  /* Destroy the structure: */
  free (l);
}

/* Returns NULL on failure: */
struct epsilon_executable *
read_epsilon_executable (FILE * f)
{
  struct epsilon_executable *r;
  integer_t i, y;
  char header[5];

  /* Read header: */
  fread (header, sizeof (char), 4, f);
  header[4] = '\0';
  if (strcmp (header, EXECUTABLE_HEADER))
    {
      fprintf (stderr, "The file is not an epsilon executable\n");
      return NULL;
    }

  /* Allocate storage: */
  r =
    (struct epsilon_executable *) malloc (sizeof (struct epsilon_executable));

  /* Read bytecode version: */
  fread (r->bytecode_version, sizeof (char), 4, f);
  r->bytecode_version[4] = '\0';

  /* Read pragmas: */
  fread (&(r->pragmas), sizeof (integer_t), 1, f);

  /* Read registers_no: */
  fread (&(r->word_registers_no), sizeof (integer_t), 1, f);
  fread (&(r->wide_integer_registers_no), sizeof (integer_t), 1, f);
  fread (&(r->float_registers_no), sizeof (integer_t), 1, f);
  fread (&(r->wide_float_registers_no), sizeof (integer_t), 1, f);
  fread (&(r->wide_wide_float_registers_no), sizeof (integer_t), 1, f);

  /* Read instructions_no: */
  fread (&(r->instructions_no), sizeof (integer_t), 1, f);

  /* Read instructions */
  r->instructions =
    (instruction_t *) malloc (sizeof (instruction_t) * r->instructions_no);
  fread (r->instructions, sizeof (instruction_t), r->instructions_no, f);

  /* Read string_constants_no: */
  fread (&(r->string_constants_no), sizeof (integer_t), 1, f);
  r->string_constants =
    (char **) malloc (sizeof (char *) * r->string_constants_no);

  /* Read string_constants: */
  for (i = 0; i < r->string_constants_no; i++)
    {
      fread (&y, sizeof (integer_t), 1, f);	/* strlen of string constant */
      r->string_constants[i] = (char *) malloc (sizeof (char) * (y + 1));
      fread (r->string_constants[i], sizeof (char), y, f);
      r->string_constants[i][y] = '\0';
    }

  /* Read globals_no: */
  fread (&(r->globals_no), sizeof (integer_t), 1, f);

  /* Read exceptions_no: */
  fread (&(r->exceptions_no), sizeof (integer_t), 1, f);
  
  /* Allocate and read exceptions: */
  r->exceptions = (char **) malloc (sizeof (char *) * r->exceptions_no);
  for (i = 0; i < r->exceptions_no; i++)
    {
      fread (&y, sizeof (integer_t), 1, f);	/* strlen of string constant */
      r->exceptions[i] = (char *) malloc (sizeof (char) * (y + 1));
      fread (r->exceptions[i], sizeof (char), y, f);
      r->exceptions[i][y] = '\0';
    }
  
  /* Read c_libraries_no: */
  fread (&(r->c_libraries_no), sizeof (integer_t), 1, f);

  /* Allocate and read c_libraries: */
  r->c_libraries =
    (char **) malloc (sizeof (char *) * r->c_libraries_no);
  for (i = 0; i < r->c_libraries_no; i++)
    {
      fread (&y, sizeof (integer_t), 1, f);	/* strlen of string constant */
      r->c_libraries[i] = (char *) malloc (sizeof (char) * (y + 1));
      fread (r->c_libraries[i], sizeof (char), y, f);
      r->c_libraries[i][y] = '\0';
    }

  /* Read c_symbols_no: */
  fread (&(r->c_symbols_no), sizeof (integer_t), 1, f);

  /* Allocate and read c_symbols: */
  r->c_symbols =
    (c_symbol_t *) malloc (sizeof (c_symbol_t) * r->c_symbols_no);
  for (i = 0; i < r->c_symbols_no; i++)
    {
      /* Read the i-th symbol_name: */
      fread (&y, sizeof (integer_t), 1, f);	/* strlen of string constant */
      r->c_symbols[i].symbol_name = (char *) malloc (sizeof (char) * (y + 1));
      fread (r->c_symbols[i].symbol_name, sizeof (char), y, f);
      r->c_symbols[i].symbol_name[y] = '\0';

      /* Read the i-th library_index: */
      fread (&(r->c_symbols[i].library_index), sizeof (integer_t), 1, f);
      //      fprintf(stderr, "[%s from %s]\n",
      //	      r->c_symbols[i].symbol_name,
      //	      r->c_libraries[r->c_symbols[i].library_index]);
    }
  //  fprintf(stderr, "libraries are "INTEGER_T_FORMAT", symbols are "
  //	  INTEGER_T_FORMAT"\n",
  //	  r->c_libraries_no, r->c_symbols_no);
  
  /* Return: */
  return r;
}

/* Returns NULL on failure: */
struct epsilon_module *
read_epsilon_module (FILE * f)
{
  struct epsilon_module *r;
  integer_t i, y;
  char header[5];

  /* Read header: */
  fread (header, sizeof (char), 4, f);
  header[4] = '\0';
  if (strcmp (header, MODULE_HEADER))
    {
      fprintf (stderr, "The file is not an epsilon module\n");
      return NULL;
    }

  /* Allocate storage: */
  r = (struct epsilon_module *) malloc (sizeof (struct epsilon_module));

  /* Read bytecode version: */
  fread (r->bytecode_version, sizeof (char), 4, f);
  r->bytecode_version[4] = '\0';

  /* Read pragmas: */
  fread (&(r->pragmas), sizeof (integer_t), 1, f);

  /* Read name: */
  fread (&y, sizeof (integer_t), 1, f);	/* strlen of name */
  r->name = (char *) malloc (sizeof (char) * (y + 1));
  fread (r->name, sizeof (char), y, f);
  r->name[y] = '\0';
  if (strcmp (r->bytecode_version, BYTECODE_VERSION))
    fprintf (stderr, "Warning: bytecode version is %s instead of %s in %s\n",
	     r->bytecode_version, BYTECODE_VERSION, r->name);

  /* Read needed_modules: */
  fread (&(r->needed_modules_no), sizeof (integer_t), 1, f);	/* length of needed_modules */
  r->needed_modules =
    (char **) malloc (sizeof (char *) * (r->needed_modules_no));
  for (i = 0; i < r->needed_modules_no; i++)
    {
      fread (&y, sizeof (integer_t), 1, f);	/* strlen of name */
      r->needed_modules[i] = (char *) malloc (sizeof (char) * (y + 1));
      fread (r->needed_modules[i], sizeof (char), y, f);
      r->needed_modules[i][y] = '\0';
    }

  /* Read registers_no: */
  fread (&(r->word_registers_no), sizeof (integer_t), 1, f);
  fread (&(r->wide_integer_registers_no), sizeof (integer_t), 1, f);
  fread (&(r->float_registers_no), sizeof (integer_t), 1, f);
  fread (&(r->wide_float_registers_no), sizeof (integer_t), 1, f);
  fread (&(r->wide_wide_float_registers_no), sizeof (integer_t), 1, f);

  /* Read instructions_no: */
  fread (&(r->instructions_no), sizeof (integer_t), 1, f);

  /* Read instructions: */
  r->instructions =
    (instruction_t *) malloc (sizeof (instruction_t) * r->instructions_no);
  fread (r->instructions, sizeof (instruction_t), r->instructions_no, f);

  /* Read string_constants_no: */
  fread (&(r->string_constants_no), sizeof (integer_t), 1, f);
  r->string_constants =
    (char **) malloc (sizeof (char *) * r->string_constants_no);

  /* Read string_constants: */
  for (i = 0; i < r->string_constants_no; i++)
    {
      fread (&y, sizeof (integer_t), 1, f);	/* strlen of string constant */
      r->string_constants[i] = (char *) malloc (sizeof (char) * (y + 1));
      fread (r->string_constants[i], sizeof (char), y, f);
      r->string_constants[i][y] = '\0';
    }

  /* Read globals_no: */
  fread (&(r->globals_no), sizeof (integer_t), 1, f);
  r->globals = (char **) malloc (sizeof (char *) * r->globals_no);

  /* Read globals: */
  for (i = 0; i < r->globals_no; i++)
    {
      fread (&y, sizeof (integer_t), 1, f);	/* strlen of string constant */
      r->globals[i] = (char *) malloc (sizeof (char) * (y + 1));
      fread (r->globals[i], sizeof (char), y, f);
      r->globals[i][y] = '\0';
    }

  /* Read exceptions_no: */
  fread (&(r->exceptions_no), sizeof (integer_t), 1, f);
  r->exceptions = (char **) malloc (sizeof (char *) * r->exceptions_no);

  /* Read exceptions: */
  for (i = 0; i < r->exceptions_no; i++)
    {
      fread (&y, sizeof (integer_t), 1, f);	/* strlen of string constant */
      r->exceptions[i] = (char *) malloc (sizeof (char) * (y + 1));
      fread (r->exceptions[i], sizeof (char), y, f);
      r->exceptions[i][y] = '\0';
    }

  /* Read c_libraries_no: */
  fread (&(r->c_libraries_no), sizeof (integer_t), 1, f);

  /* Allocate and read c_libraries: */
  r->c_libraries =
    (char **) malloc (sizeof (char *) * r->c_libraries_no);
  for (i = 0; i < r->c_libraries_no; i++)
    {
      fread (&y, sizeof (integer_t), 1, f);	/* strlen of string constant */
      r->c_libraries[i] = (char *) malloc (sizeof (char) * (y + 1));
      fread (r->c_libraries[i], sizeof (char), y, f);
      r->c_libraries[i][y] = '\0';
    }

  /* Read c_symbols_no: */
  fread (&(r->c_symbols_no), sizeof (integer_t), 1, f);

  /* Allocate and read c_symbols: */
  r->c_symbols =
    (c_symbol_t *) malloc (sizeof (c_symbol_t) * r->c_symbols_no);
  for (i = 0; i < r->c_symbols_no; i++)
    {
      /* Read the i-th symbol_name: */
      fread (&y, sizeof (integer_t), 1, f);	/* strlen of string constant */
      r->c_symbols[i].symbol_name = (char *) malloc (sizeof (char) * (y + 1));
      fread (r->c_symbols[i].symbol_name, sizeof (char), y, f);
      r->c_symbols[i].symbol_name[y] = '\0';

      /* Read the i-th library_index: */
      fread (&(r->c_symbols[i].library_index), sizeof (integer_t), 1, f);
    }

  /* Return: */
  return r;
}

/* Returns NULL on failure: */
struct epsilon_library *
read_epsilon_library (FILE * f)
{
  struct epsilon_library *r;
  int i;
  char header[5];

  /* Read header: */
  fread (header, sizeof (char), 4, f);
  header[4] = '\0';
  if (strcmp (header, LIBRARY_HEADER))
    {
      fprintf (stderr, "The file is not an epsilon library\n");
      return NULL;
    }

  /* Allocate storage: */
  r = (struct epsilon_library *) malloc (sizeof (struct epsilon_library));

  /* Read pragmas: */
  fread (&(r->pragmas), sizeof (integer_t), 1, f);

  /* Read modules_no: */
  fread (&(r->modules_no), sizeof (integer_t), 1, f);

  /* Read modules: */
  r->modules = (struct epsilon_module **)
    malloc (sizeof (struct epsilon_module *) * (r->modules_no));
  for (i = 0; i < r->modules_no; i++)
    r->modules[i] = read_epsilon_module (f);

  /* Return: */
  return r;
}

/* Returns NULL on failure */
struct epsilon_executable *
read_epsilon_executable_file (const char *filename)
{
  FILE *f;
  struct epsilon_executable *r;

  /* Open file: */
  if ((f = fopen (filename, "r")) == NULL)
    {
      fprintf (stderr, "Could not open the file \'%s\'\n", filename);
      return NULL;
    }

  /* Read executable from file, then close file: */
  r = read_epsilon_executable (f);
  fclose (f);

  /* Return: */
  return r;
}

/* Returns NULL on failure */
struct epsilon_module *
read_epsilon_module_file (const char *filename)
{
  FILE *f;
  struct epsilon_module *r;

  /* Open file: */
  if ((f = fopen (filename, "r")) == NULL)
    {
      fprintf (stderr, "Could not open the file \'%s\'\n", filename);
      return NULL;
    }

  /* Read module from file, then close file: */
  r = read_epsilon_module (f);
  fclose (f);

  /* Return: */
  return r;
}

/* Returns NULL on failure */
struct epsilon_library *
read_epsilon_library_file (const char *filename)
{
  FILE *f;
  struct epsilon_library *r;

  /* Open file: */
  if ((f = fopen (filename, "r")) == NULL)
    {
      fprintf (stderr, "Could not open the file \'%s\'\n", filename);
      return NULL;
    }

  /* Read library from file, then close file: */
  r = read_epsilon_library (f);
  fclose (f);

  /* Return: */
  return r;
}

/* Returns 0 on success: */
int
write_epsilon_executable (const struct epsilon_executable *r, FILE * f)
{
  integer_t i, y;

  /* Write header: */
  fwrite (EXECUTABLE_HEADER, sizeof (char), 4, f);

  /* Write bytecode version: */
  fwrite (&(r->bytecode_version), sizeof (char), 4, f);

  /* Write pragmas: */
  fwrite (&(r->pragmas), sizeof (integer_t), 1, f);

  /* Write registers_no: */
  fwrite (&(r->word_registers_no), sizeof (integer_t), 1, f);
  fwrite (&(r->wide_integer_registers_no), sizeof (integer_t), 1, f);
  fwrite (&(r->float_registers_no), sizeof (integer_t), 1, f);
  fwrite (&(r->wide_float_registers_no), sizeof (integer_t), 1, f);
  fwrite (&(r->wide_wide_float_registers_no), sizeof (integer_t), 1, f);

  /* Write instructions_no: */
  fwrite (&(r->instructions_no), sizeof (integer_t), 1, f);

  /* Write instructions: */
  fwrite (r->instructions, sizeof (instruction_t), r->instructions_no, f);

  /* Write string_constants_no: */
  fwrite (&(r->string_constants_no), sizeof (integer_t), 1, f);

  /* Write string_constants: */
  for (i = 0; i < r->string_constants_no; i++)
    {
      y = strlen (r->string_constants[i]);
      fwrite (&y, sizeof (integer_t), 1, f);
      fwrite (r->string_constants[i], sizeof (char), y, f);
    }

  /* Write globals_no: */
  fwrite (&(r->globals_no), sizeof (integer_t), 1, f);

  /* Write exceptions_no: */
  fwrite (&(r->exceptions_no), sizeof (integer_t), 1, f);

  /* Write exceptions: */
  for (i = 0; i < r->exceptions_no; i++)
    {
      y = strlen (r->exceptions[i]);
      fwrite (&y, sizeof (integer_t), 1, f);
      fwrite (r->exceptions[i], sizeof (char), y, f);
    }

  /* Write c_libraries_no: */
  fwrite (&(r->c_libraries_no), sizeof (integer_t), 1, f);

  /* Write c_libraries: */
  for (i = 0; i < r->c_libraries_no; i++)
    {
      y = strlen(r->c_libraries[i]);
      fwrite (&y, sizeof (integer_t), 1, f);
      fwrite (r->c_libraries[i], sizeof (char), y, f);
    }

  /* Write c_symbols_no: */
  fwrite (&(r->c_symbols_no), sizeof (integer_t), 1, f);

  /* Write c_symbols: */
  for (i = 0; i < r->c_symbols_no; i++)
    {
      /* Write the i-th symbol_name: */
      y = strlen(r->c_symbols[i].symbol_name);
      fwrite (&y, sizeof (integer_t), 1, f);
      fwrite (r->c_symbols[i].symbol_name, sizeof (char), y, f);

      /* Write the i-th library_index: */
      fwrite (&(r->c_symbols[i].library_index), sizeof (integer_t), 1, f);
    }

  /* Return: */
  return 0;
}

/* Returns 0 on success: */
int
write_epsilon_module (const struct epsilon_module *r, FILE * f)
{
  integer_t i, y;

  /* Write header: */
  fwrite (MODULE_HEADER, sizeof (char), 4, f);

  /* Write bytecode version: */
  fwrite (&(r->bytecode_version), sizeof (char), 4, f);

  /* Write pragmas: */
  fwrite (&(r->pragmas), sizeof (integer_t), 1, f);

  /* Write name: */
  y = strlen (r->name);
  fwrite (&y, sizeof (integer_t), 1, f);
  fwrite (r->name, sizeof (char), y, f);

  /* Write needed_modules: */
  fwrite (&(r->needed_modules_no), sizeof (integer_t), 1, f);
  for (i = 0; i < r->needed_modules_no; i++)
    {
      y = strlen (r->needed_modules[i]);
      fwrite (&y, sizeof (integer_t), 1, f);
      fwrite (r->needed_modules[i], sizeof (char), y, f);
    }

  /* Write registers_no: */
  fwrite (&(r->word_registers_no), sizeof (integer_t), 1, f);
  fwrite (&(r->wide_integer_registers_no), sizeof (integer_t), 1, f);
  fwrite (&(r->float_registers_no), sizeof (integer_t), 1, f);
  fwrite (&(r->wide_float_registers_no), sizeof (integer_t), 1, f);
  fwrite (&(r->wide_wide_float_registers_no), sizeof (integer_t), 1, f);

  /* Write instructions_no: */
  fwrite (&(r->instructions_no), sizeof (integer_t), 1, f);

  /* Write instructions: */
  fwrite (r->instructions, sizeof (instruction_t), r->instructions_no, f);

  /* Write string_constants_no: */
  fwrite (&(r->string_constants_no), sizeof (integer_t), 1, f);

  /* Write string_constants: */
  for (i = 0; i < r->string_constants_no; i++)
    {
      y = strlen (r->string_constants[i]);
      fwrite (&y, sizeof (integer_t), 1, f);
      fwrite (r->string_constants[i], sizeof (char), y, f);
    }

  /* Write globals_no: */
  fwrite (&(r->globals_no), sizeof (integer_t), 1, f);

  /* Write globals: */
  for (i = 0; i < r->globals_no; i++)
    {
      y = strlen (r->globals[i]);
      fwrite (&y, sizeof (integer_t), 1, f);
      fwrite (r->globals[i], sizeof (char), y, f);
    }

  /* Write exceptions_no: */
  fwrite (&(r->exceptions_no), sizeof (integer_t), 1, f);

  /* Write exceptions: */
  for (i = 0; i < r->exceptions_no; i++)
    {
      y = strlen (r->exceptions[i]);
      fwrite (&y, sizeof (integer_t), 1, f);
      fwrite (r->exceptions[i], sizeof (char), y, f);
    }

  /* Write c_libraries_no: */
  fwrite (&(r->c_libraries_no), sizeof (integer_t), 1, f);

  /* Write c_libraries: */
  for (i = 0; i < r->c_libraries_no; i++)
    {
      y = strlen(r->c_libraries[i]);
      fwrite (&y, sizeof (integer_t), 1, f);
      fwrite (r->c_libraries[i], sizeof (char), y, f);
    }

  /* Write c_symbols_no: */
  fwrite (&(r->c_symbols_no), sizeof (integer_t), 1, f);

  /* Write c_symbols: */
  for (i = 0; i < r->c_symbols_no; i++)
    {
      /* Write the i-th symbol_name: */
      y = strlen(r->c_symbols[i].symbol_name);
      fwrite (&y, sizeof (integer_t), 1, f);
      fwrite (r->c_symbols[i].symbol_name, sizeof (char), y, f);

      /* Write the i-th library_index: */
      fwrite (&(r->c_symbols[i].library_index), sizeof (integer_t), 1, f);
    }

  /* Return: */
  return 0;
}

/* Returns 0 on success: */
int
write_epsilon_library (const struct epsilon_library *r, FILE * f)
{
  integer_t i;

  /* Write header: */
  fwrite (LIBRARY_HEADER, sizeof (char), 4, f);

  /* Write pragmas: */
  fwrite (&(r->pragmas), sizeof (integer_t), 1, f);

  /* Write modules_no: */
  fwrite (&(r->modules_no), sizeof (integer_t), 1, f);

  /* Write modules: */
  for (i = 0; i < r->modules_no; i++)
    write_epsilon_module (r->modules[i], f);

  /* Return: */
  return 0;
}

/* Returns 0 on success: */
int
write_epsilon_executable_file (const struct epsilon_executable *m,
			       const char *filename)
{
  int r = 0;
  FILE *f = fopen (filename, "w");

  if (f == NULL)
    {
      fprintf (stderr, "Could not open the file \'%s\'\n", filename);
      return -1;
    }
  r = write_epsilon_executable (m, f);

  fclose (f);
  return r;
}

/* Returns 0 on success: */
int
write_epsilon_module_file (const struct epsilon_module *m,
			   const char *filename)
{
  int r = 0;
  FILE *f = fopen (filename, "w");

  if (f == NULL)
    {
      fprintf (stderr, "Could not open the file \'%s\'\n", filename);
      return -1;
    }
  r = write_epsilon_module (m, f);

  fclose (f);
  return r;
}

/* Returns 0 on success: */
int
write_epsilon_library_file (const struct epsilon_library *l,
			    const char *filename)
{
  int r = 0;
  FILE *f;

  f = fopen (filename, "w");
  if (f == NULL)
    {
      fprintf (stderr, "Could not open the file \'%s\'\n", filename);
      return -1;
    }
  r = write_epsilon_library (l, f);
  
  fclose (f);
  return r;
}
