/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2003 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */


#include "eam_types.h"
#include "../commontext.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include "instructions.h"
#include "instructions.h"
#include "bytecode.h"

#include "gc/gc.h" /* for MAXIMUM_SMALL_HOMOGENEOUS_SIZE */

struct epsilon_module* module;
integer_t verbose;

void check_command_line(int argc, char** argv){
  if(argc != 2)
    command_line_fatal("there must be one and only one argument");
  if(!has_extension(argv[1], "eamo"))
    command_line_fatal("the argument must have extension .eamo");
}

/* All the optimizing functions return the number of changes made. */


/* Replaces j's to another jump (or return) instruction X with the X
   instruction: */
integer_t optimize_j_over_jump_for_opcode(int opcode){
  integer_t i;
  integer_t replacements_no = 0;
  
  /* Look at each instruction: */
  for(i = 0; i < module->instructions_no; i++)
    if(module->instructions[i].opcode == j)
      if(module->instructions[module->instructions[i].label_parameter].opcode ==
	 opcode){
	/* The i-th instruction is a j to return_opcode. */

	/* Replace the j instruction: */
	memcpy(&(module->instructions[i]),
	       &(module->instructions[module->instructions[i].label_parameter]),
	       sizeof(instruction_t));
	replacements_no++; /* Increment the number of replacements */
      } /* innermost if */
  
  /* Log the number of replacements, if we need to log and if there is at least
     one replacement: */
  if(verbose && (replacements_no > 0))
    fprintf(stderr, "  Optimize j over %s: "INTEGER_T_FORMAT" replacements.\n",
	    instructions_names[opcode],
	    replacements_no);
  return replacements_no;
}

/* See the comment before optimize_j_over_jump_for_opcode(): */
integer_t optimize_j_over_jump(){
  integer_t replacements_no = 0;

  replacements_no += optimize_j_over_jump_for_opcode(s_ret);
  replacements_no += optimize_j_over_jump_for_opcode(s_retn);
  replacements_no += optimize_j_over_jump_for_opcode(j);
  replacements_no += optimize_j_over_jump_for_opcode(s_thrw);
  replacements_no += optimize_j_over_jump_for_opcode(rthrw);

  return replacements_no;
}

integer_t is_an_s_return_instruction(integer_t index){
  return (module->instructions[index].opcode == s_ret)
    || (module->instructions[index].opcode == s_retn);
}

/* Return non-zero iff the index-th instruction, which *must* be
   an s_cll, is followed by zero or more s_popl's and one s_ret.
   "Followed" means either directly or via an unconditional jump.
   Void instructions such as nop and lbl are ignored for this
   purpose. */
integer_t is_s_cll_in_a_sibling_context(integer_t index){
  integer_t i;
  
  for(i = index + 1; i < module->instructions_no; i++){
    switch(module->instructions[i].opcode){
    case s_ret: case s_retn: {
      return (integer_t)1; /* Ok, this is a sibling context */
    }
    case nop: case lbl: case s_popl: {
      /* Ok, go on. */
      break;
    }
    case j: {
      /* Continue looking at the context, starting from where we're jumping.
	 This will loop if there is an infinite loop made with j (not generated
	 by the epsilon compiler). */
      i = module->instructions[i].label_parameter - (integer_t)1;
      break;
    }
    default: {
      /* There is some instruction we didn't expect between the call and the
	 return. */
      return (integer_t)0; /* Not a sibling context. */
    }
    } /* switch */
  } /* for */
  
  /* If we arrived here then we are at the end of the program, and we didn't 
     find the s_return instruction. This is not a sibling context: */
  return (integer_t)0;
}

integer_t optimize_sibling_calls(){
  integer_t i;
  integer_t replacements_no = 0;
  
  /* Look at each instruction: */
  for(i = 0; i < module->instructions_no - 2; i++)
    if(module->instructions[i].opcode == s_cll)
      if(is_s_cll_in_a_sibling_context(i)){ /* this call can be made sibling */
	module->instructions[i].opcode = s_scll;

        replacements_no++; /* Increment the number of replacements */
      } /* innermost if */
  /* Log the number of replacements, if we need to log and if there is at least
     one replacement: */
  if(verbose && (replacements_no > 0))
    fprintf(stderr, "  Optimize sibling calls: "INTEGER_T_FORMAT" replacements.\n",
	    replacements_no);
  return replacements_no;
}

/* See the comment before optimize_j_on_conditions(): */
integer_t optimize_j_on_condition(int condition_opcode,
				  int jump_on_condition_opcode,
				  int jump_on_opposite_condition_opcode){
  integer_t i;
  integer_t replacements_no = 0;

  /* Look at each instruction: */
  for(i = 0; i < module->instructions_no - 1; i++)
    if(module->instructions[i].opcode == condition_opcode){
      if(module->instructions[i + 1].opcode == s_jnz){
        /* This conditional jump can be optimized: */
        module->instructions[i].opcode = jump_on_condition_opcode;
	module->instructions[i].label_parameter = 
	  module->instructions[i + 1].label_parameter;
        module->instructions[i + 1].opcode = nop; /* the jump is now at i */
        replacements_no++; /* Increment the number of replacements */
	i++; /* don't look at the next nop, it's a waste of time */
      } /* innermost if */
      else if(module->instructions[i + 1].opcode == s_jz){
        /* This conditional jump can be optimized: */
        module->instructions[i].opcode = jump_on_opposite_condition_opcode;
        module->instructions[i].label_parameter =
          module->instructions[i + 1].label_parameter;
        module->instructions[i + 1].opcode = nop; /* the jump is now at i */
        replacements_no++; /* Increment the number of replacements */
	i++; /* don't look at the next nop, it's a waste of time */
      } /* else */
    } /* external if */
  
  /* Log the number of replacements, if we need to log and if there is at least
     one replacement: */
  if(verbose && (replacements_no > 0))
    fprintf(stderr, "  Optimize jump on %s: "INTEGER_T_FORMAT" replacements.\n",
            instructions_names[condition_opcode],
            replacements_no);
  return replacements_no;
}

/* Replace
       s_STACK_OP
       s_jnz L:
     with
       s_jSTACK_OP L:
     and replace
       s_jSTACK_OP
       s_jz L:
     with
       s_jOPPOSITE_STACK_OP L: */
integer_t optimize_j_on_conditions(){
  integer_t replacements_no = 0;

  replacements_no += optimize_j_on_condition(s_lti,  s_jlti,  s_jgtei);
  replacements_no += optimize_j_on_condition(s_ltei, s_jltei, s_jgti);
  replacements_no += optimize_j_on_condition(s_gti,  s_jgti,  s_jltei);
  replacements_no += optimize_j_on_condition(s_gtei, s_jgtei, s_jlti);
  replacements_no += optimize_j_on_condition(s_eqi,  s_jeqi,  s_jneqi);
  replacements_no += optimize_j_on_condition(s_neqi, s_jneqi, s_jeqi);
  replacements_no += optimize_j_on_condition(s_andi, s_jandi, s_jnandi);
  replacements_no += optimize_j_on_condition(s_ori,  s_jori,  s_jnori);
  replacements_no += optimize_j_on_condition(s_xori, s_jxori, s_jnxori);
  replacements_no += optimize_j_on_condition(s_noti, s_jz,    s_jnz);

  return replacements_no;
}

/* Replace
     ...
     pshci N
     INSTRUCTION
   with
     ...
     INSTRUCTION_i N
   where INSTRUCTION is a nullary instruction taking two parameters from the
   stack, and INSTRUCTION_i is another instruction making the same computation,
   but taking the first parameter from the stack and the second one immediate.
*/
integer_t introduce_immediates_for_instruction(int stack_stack_opcode,
					  int stack_immediate_opcode){
  integer_t i;
  integer_t replacements_no = 0;

  /* Look at each instruction: */
  for(i = 0; i < module->instructions_no - 1; i++)
    if((module->instructions[i].opcode == pshci) &&
       (module->instructions[i + 1].opcode == stack_stack_opcode)){
        /* This instruction can be replaced with the immediate version: */
        module->instructions[i].opcode = stack_immediate_opcode;
        module->instructions[i + 1].opcode = nop; /* we anticipated computation */
	
        replacements_no++; /* Increment the number of replacements */
      } /* if */
  
  /* Log the number of replacements, if we need to log and if there is at least
     one replacement: */
  if(verbose && (replacements_no > 0))
    fprintf(stderr, "  Introduce immediate for %s: "INTEGER_T_FORMAT" replacements.\n",
            instructions_names[stack_stack_opcode],
            replacements_no);
  return replacements_no;
}

/* See the comment before introduce_immediates_for_instruction(): */
integer_t introduce_immediates(){
  integer_t replacements_no = 0;
  
  replacements_no += introduce_immediates_for_instruction(s_addi, s_addi_i);
  replacements_no += introduce_immediates_for_instruction(s_subi, s_subi_i);
  replacements_no += introduce_immediates_for_instruction(s_muli, s_muli_i);
  replacements_no += introduce_immediates_for_instruction(s_powi, s_powi_i);
  /* This was commented out to prevent changing
       pshci  0
       s_divi
     into
       s_divi_i 0,
     which does not deal with division by zero. */
  /* replacements_no += introduce_immediates_for_instruction(s_divi, s_divi_i); */
  replacements_no += introduce_immediates_for_instruction(s_eqi,  s_eqi_i);
  replacements_no += introduce_immediates_for_instruction(s_neqi, s_neqi_i);
  replacements_no += introduce_immediates_for_instruction(s_lti,  s_lti_i);
  replacements_no += introduce_immediates_for_instruction(s_ltei, s_ltei_i);
  replacements_no += introduce_immediates_for_instruction(s_gti,  s_gti_i);
  replacements_no += introduce_immediates_for_instruction(s_gtei, s_gtei_i);

  return replacements_no;
}

/* Replace 
     pshci N
     STACK_OP
     s_jnz L: [jz L:]
   with
     s_jSTACK_OP_i N L: [s_jSTACK_REVERSE_OP_i N L:]
*/
integer_t introduce_conditional_jumps_with_immediates_for_opcode(
                      int condition_opcode,
                      int jump_on_condition_with_immediate_opcode,
                      int jump_on_opposite_condition_with_immediate_opcode){
  integer_t i;
  integer_t replacements_no = 0;

  /* Look at each instruction: */
  for(i = 0; i < module->instructions_no - 2; i++)
    if((module->instructions[i].opcode == pshci)
       &&(module->instructions[i + 1].opcode == condition_opcode)){
      if(module->instructions[i + 2].opcode == s_jnz){
        /* This conditional jump can be optimized: */
        module->instructions[i].opcode =
	  jump_on_condition_with_immediate_opcode;
	/* The integer parameter is already in its right place. */
	module->instructions[i].label_parameter = 
	  module->instructions[i + 2].label_parameter;
        module->instructions[i + 1].opcode = nop; /* the jump is now at i */
        module->instructions[i + 2].opcode = nop; /* the jump is now at i */
        replacements_no++; /* Increment the number of replacements */
	i += 2; /* don't look at the next nops, it's a waste of time */
      } /* innermost if */
      else if(module->instructions[i + 2].opcode == s_jz){
        /* This conditional jump can be optimized: */
        module->instructions[i].opcode =
	  jump_on_opposite_condition_with_immediate_opcode;
	/* The integer parameter is already in its right place. */
	module->instructions[i].label_parameter = 
	  module->instructions[i + 2].label_parameter;
        module->instructions[i + 1].opcode = nop; /* the jump is now at i */
        module->instructions[i + 2].opcode = nop; /* the jump is now at i */
        replacements_no++; /* Increment the number of replacements */
	i += 2; /* don't look at the next nops, it's a waste of time */
      } /* else */
    } /* external if */
  
  /* Log the number of replacements, if we need to log and if there is at least
     one replacement: */
  if(verbose && (replacements_no > 0))
    fprintf(stderr, "  Introduce jump on cond. with imm. for %s: "INTEGER_T_FORMAT" replacements.\n",
            instructions_names[condition_opcode],
            replacements_no);
  return replacements_no;
}

/* See the comment before
   introduce_conditional_jumps_with_immediates_for_opcode(): */
integer_t introduce_conditional_jumps_with_immediates(){
  integer_t replacements_no = 0;

  replacements_no+=
    introduce_conditional_jumps_with_immediates_for_opcode(s_lti,
							   s_jlti_i,
							   s_jgtei_i);
  replacements_no+=
    introduce_conditional_jumps_with_immediates_for_opcode(s_ltei,
							   s_jltei_i,
							   s_jgti_i);
  replacements_no+=
    introduce_conditional_jumps_with_immediates_for_opcode(s_gti,
							   s_jgti_i,
							   s_jltei_i);
  replacements_no+=
    introduce_conditional_jumps_with_immediates_for_opcode(s_gtei,
							   s_jgtei_i,
							   s_jlti_i);
  replacements_no+=
    introduce_conditional_jumps_with_immediates_for_opcode(s_eqi,
							   s_jeqi_i,
							   s_jneqi_i);
  replacements_no+=
    introduce_conditional_jumps_with_immediates_for_opcode(s_neqi,
							   s_jneqi_i,
							   s_jeqi_i);
  return replacements_no;
}

integer_t introduce_fast_instructions(){
  integer_t i;
  integer_t replacements_no = 0;

  /* Look at each instruction: */
  for(i = 0; i < module->instructions_no; i++)
    if((module->instructions[i].opcode == s_mka_i) &&
       (((integer_t)(module->instructions[i].parameter_1) == (integer_t)2))){
      /* This instruction can be replaced with an s_cns: */
      module->instructions[i].opcode = s_cns;
      
      replacements_no++; /* Increment the number of replacements */
    } /* if */
    else if((module->instructions[i].opcode == s_cll)
	    && ((integer_t)(module->instructions[i].parameter_1) > (integer_t)0)
	    && ((integer_t)(module->instructions[i].parameter_1) <=
		(integer_t)MAXIMUM_SMALL_HOMOGENEOUS_SIZE)){
      /* This instruction can be replaced with an s_fcll: */
      module->instructions[i].opcode = s_fcll;
      
      replacements_no++; /* Increment the number of replacements */
    } /* else */
    else if((module->instructions[i].opcode == s_scll)
	    && ((integer_t)(module->instructions[i].parameter_1) > (integer_t)0)
	    && ((integer_t)(module->instructions[i].parameter_1) <=
		(integer_t)MAXIMUM_SMALL_HOMOGENEOUS_SIZE)){
      /* This instruction can be replaced with an s_fscll: */
      module->instructions[i].opcode = s_fscll;
      
      replacements_no++; /* Increment the number of replacements */
    } /* else */
  
  /* Log the number of replacements, if we need to log and if there is at least
     one replacement: */
  if(verbose && (replacements_no > 0))
    fprintf(stderr, "  Introduce fast instructions: "INTEGER_T_FORMAT" replacements.\n",
            replacements_no);
  return replacements_no;
}

integer_t optimize_j_over_next_instruction(){
  integer_t replacements_no = 0;
  integer_t i;

  /* The pattern
          j   L:
       L: xxxx
     is very common, generated by if-then-else. We can safely replace j with
     nop. */
  
  /* Look at each instruction: */
  for(i = 0; i < module->instructions_no - 2; i++)
    if((module->instructions[i].opcode == j)
       &&(module->instructions[i].label_parameter == (i + 2))
       &&(module->instructions[i + 1].opcode == lbl)){
      /* Remove the jump, it is not needed. The nop and the lbl will be removed
	 later. */
      module->instructions[i].opcode = nop;
      
      /* We made a substitution: */
      replacements_no++;
    }
  
  /* Log the number of replacements, if we need to log and if there is at least
     one replacement: */
  if(verbose && (replacements_no > 0))
    fprintf(stderr, "  Optimize j over next instruction: "INTEGER_T_FORMAT" replacements.\n",
            replacements_no);
  return replacements_no;
}

integer_t remove_nop(){
  /* To do */
  return 0;
}

integer_t remove_lbl(){
  /* To do */
  return 0;
}

/* Make idempotent optimizations, i.e. the ones such that it makes no
   sense to execute more than once: */
integer_t make_idempotent_optimizations(){
  integer_t replacements_no = 0;
  
  replacements_no += introduce_fast_instructions();
  replacements_no += remove_lbl();
  
  return replacements_no;
}

/* Execute every non-idempotent optimization once: */
integer_t make_non_idempotent_optimizations(){
  integer_t replacements_no = 0;
  
  /* Make each optimization once: */
  replacements_no += optimize_j_over_jump();
  /* This prevents to optimize jumps on conditions and immediates but
     yields better code, so we make this first. */
  replacements_no += optimize_sibling_calls();

  /* This must be done after optimize_sibling_calls(), since it would
     prevent it, and tail calls are more important. */
  replacements_no += optimize_j_over_next_instruction();

  replacements_no += introduce_conditional_jumps_with_immediates();
  replacements_no += optimize_j_on_conditions();
  replacements_no += introduce_immediates();

  replacements_no += remove_nop();
  
  return replacements_no;
}

/* Load filename into *module, or abort on error: */
void load(char* filename){
  module = read_epsilon_module_file(filename);
  if(module == NULL){
    fprintf(stderr, "About %s:\n", filename);
    fatal("could not read file");
  }
}

/* Save *module into filename, or abort on error: */
void save(char* filename){
  if(write_epsilon_module_file(module, filename) != 0){
    fprintf(stderr, "About %s:\n", filename);
    fatal("could not write file");
  }
}

void load_optimize_and_save(char* filename){
  integer_t replacements_no, total_replacements_no = 0;
  
  /* Load module, and abort on failure: */
  load(filename);
  
  /* Make non-idempotent optimizations, continuing until nothing changes: */
  do{
    replacements_no = make_non_idempotent_optimizations();
    total_replacements_no += replacements_no;
  } while(replacements_no != 0);
  
  /* Make idempotent optimizations: */
  total_replacements_no += make_idempotent_optimizations();

  if(verbose)
    fprintf(stderr, "  "INTEGER_T_FORMAT" replacements made.\n",
	    total_replacements_no);
    
  /* Save the optimized module, and abort on failure: */
  save(filename);
}

int main(int argc, char** argv){
  /* Define command line behavior: */
  set_program_name ("eampo (" PACKAGE_NAME ")");
  set_general_help_message
    ("Executes peephole optimizations in the given eAM bytecode module,\n"
     "replacing it with the optimized version.");
  set_synopsis_string ("eampo OPTIONS FILE.eamo");
  set_version_string (VERSION_STRING);
  set_copyright_string (COPYRIGHT_STRING);
  set_license_message (LICENSE_STRING);
  set_bug_reporting_message (BUG_REPORTING_MESSAGE);

  add_toggle_option ("verbose", 'v', 0,
                     "Be more verbose while processing");

  /* Parse command line and check consistency: */
  parse_command_line (&argc, argv);
  check_command_line(argc, argv);

  /* Cache long_option_to_value("verbose"), for efficiency's sake: */
  verbose = long_option_to_value("verbose");
  
  /* Optimize the given module: */
  load_optimize_and_save(argv[1]);

  /* Exit with success: */
  return EXIT_SUCCESS;
}
