/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2003 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */


#include "../../config.h"
#include "../eam_types.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <pthread.h>

#include "../../common/malloc.h"
#include "gc.h"
#include "heap.h"
#include "homogeneous.h"
#include "large.h"
#include "set.h"
#include "stack.h"

#include <time.h>

/* a global flag set by command-line in the main compilation unit: */
extern integer_t verbose_gc;

set_t set_of_homogeneous_pages;
set_t set_of_large_objects;

large_page_t list_of_large_pages;

double total_gc_time = 0.0; /* in milliseconds */

typedef struct lists_of_pages{
  homogeneous_page_t non_full_pages;
  homogeneous_page_t full_pages;
  integer_t inexact_size;
} lists_of_pages_t;

/* This flag, automatically set by a concurrent thread, says if it's time to
   collect (nonzero) or we can still wait (zero): */
integer_t should_we_collect = 0;

/* The descriptor of the concurrent thread: */
pthread_t gc_thread;
pthread_mutex_t gc_mutex = PTHREAD_MUTEX_INITIALIZER;
pthread_cond_t gc_cond = PTHREAD_COND_INITIALIZER;

integer_t largest_homogeneous_object;
lists_of_pages_t* homogeneous_pages;

DEFINE_STACK(gc_roots)
DEFINE_STACK(homogeneous)
DEFINE_STACK(large)

/* This creates the first homogeneous page for objects with size 'size',
   making possible to use exact allocation for objects made of 'size'
   words. */
void add_homogeneous_size(integer_t size){
  /* size must be less than largest_homogeneous_object: */
  if(size > largest_homogeneous_object){
    fprintf(stderr,"About size " INTEGER_T_FORMAT ":\n", size);
    fatal("add_homogeneous_size(): size > largest_homogeneous_object");
  }

  /* It makes no sense to call this function more than once with the same
     argument: */
  if(homogeneous_pages[size].non_full_pages != NULL)
    return;
  
  /* Create a page with the given size, and add it to the right list: */
  homogeneous_pages[size].non_full_pages = create_homogeneous_page(size);
}

void set_inexact_sizes(){
  integer_t i;
  integer_t approximation = 0, largest_approximation = 0;

  /* Link each i-th element of homogeneous_pages to the best element which
     approximates i, which is itself if there is an homogeneous page of size,
     i, else the j-th element, where j is the smallest size of homogenous
     object greater than i. */
  for(i = largest_homogeneous_object; i >= 0; i--){
    if(homogeneous_pages[i].non_full_pages != NULL){
      approximation = i;
      if(largest_approximation == 0)
	largest_approximation = approximation;
    }
    homogeneous_pages[i].inexact_size = approximation;
    //printf(" "INTEGER_T_FORMAT" -> "INTEGER_T_FORMAT"\n", i, approximation);
  }

  /* Objects of size zero are approximated by the smallest homogeneous objects: */
  homogeneous_pages[0].inexact_size = homogeneous_pages[1].inexact_size;

  /* largest_homogeneous_object becomes the greatest homogensous object's
     size. */
  largest_homogeneous_object = largest_approximation;

  //printf("LARGEST: "INTEGER_T_FORMAT"\n", largest_approximation);
}

void set_default_homogeneous_sizes(){
  integer_t i;
  
  /* It is useful to have homogeneous pages for objects of small sizes: */
  for(i = 1; i < MAXIMUM_SMALL_HOMOGENEOUS_SIZE; i++)
    add_homogeneous_size(i);
  
  /* Add homogenous pages for objects of size 1, 2, 4, 8, ...
     MAXIMUM_HOMOGENEOUS_SIZE: */
  for(i = MAXIMUM_SMALL_HOMOGENEOUS_SIZE; i <= MAXIMUM_HOMOGENEOUS_SIZE; i *= 2)
    add_homogeneous_size(i);
  
  /* Even if MAXIMUM_HOMOGENEOUS_SIZE were not a power of two, we would
     like to have however homogenous pages for objects of its size: */
  add_homogeneous_size(MAXIMUM_HOMOGENEOUS_SIZE);
}

/* The function executed by the concurrent thread which controls whether we
   should garbage collect: */
void* work_in_gc_thread(){
  struct timespec required_time;
  struct timespec remaining;

  required_time.tv_sec = 0;
  required_time.tv_nsec = GC_TEST_TIMEOUT;
  
  /* Repeat forever: */
  while(1){
    /* Wait a relatively long time (nanosleep is a cancellation point): */
    nanosleep(&required_time, &remaining);
    /*
    sleep(1);fprintf(stderr, ".");
    sleep(1);fprintf(stderr, ".");
    sleep(1);fprintf(stderr, ".");
    sleep(1);fprintf(stderr, ".");
    */
    /* Call the heuristic and set should_we_collect if it's true: */
    pthread_mutex_lock(&gc_mutex);
    if(should_we_garbage_collect()){
      //fprintf(stderr, "it's time to collect ("INTEGER_T_FORMAT").\n", allocated_words_since_last_gc);
      should_we_collect = 1;
      while(should_we_collect != 0)
	pthread_cond_wait(&gc_cond, &gc_mutex);
    }
    pthread_mutex_unlock(&gc_mutex);
    /* else do nothing. should_we_collect is reset to zero by 
       garbage_collect(). */
  } /* while */
  
  /* This is unreachable code; it's just to avoid compiler warnings: */
  return NULL; 
}

/* Create and start the concurrent thread which controls whether we should
   garbage collect: */
void start_gc_thread(){
  int status;

  /* Create the thread and abort on error: */
  status = pthread_create (&gc_thread,
			   NULL,  /* attributes */
			   work_in_gc_thread, /* function to call in thread */
			   NULL); /* argument */
  if(status != 0)
    fatal("could not create thread\n");
  
  /* Cancellation state is by default PTHREAD_CANCEL_ENABLE, with cancel
     type the PTHREAD_CANCEL_DEFERREND. This is what we want. */
}

void terminate_gc_thread(){
  /* Send a cancellation request to the thread; it will honour it soon, since
     it runs a loop containing a nanosleep() call, and nanosleep() is a
     cancellation point: */
  if(pthread_cancel(gc_thread) != 0)
    fatal("could not cancel thread");
}

/* We just cancel the concurrent thread; all data structures should
   remain, since heap allocated object could be used again and heap
   allocation can easily continue. */
void terminate_garbage_collector(){
  terminate_gc_thread();
}

void initialize_garbage_collector(){
  integer_t i;

  /* Memoize the largest_homogeneous_object size: */
  largest_homogeneous_object = MAXIMUM_HOMOGENEOUS_SIZE;
  
  /* Create the sets of homogenous and large pages: */
  set_of_homogeneous_pages =
    create_set(SIZE_OF_SET_OF_HOMOGENEOUS_PAGES);
  set_of_large_objects =
    create_set(SIZE_OF_SET_OF_LARGE_OBJECTS);
  
  /* Create the structure holding homogeneous pages: */
  homogeneous_pages = (lists_of_pages_t*)
    malloc((largest_homogeneous_object + 1) * sizeof(lists_of_pages_t));
  if(homogeneous_pages == NULL)
    fatal("malloc() failed when creating homogeneous_pages");
  for(i = 0; i <= largest_homogeneous_object; i++){
    homogeneous_pages[i].non_full_pages = NULL;
    homogeneous_pages[i].full_pages = NULL;
  }
  
  /* There are no large pages yet: */
  list_of_large_pages = NULL;

  /* Create the GC stacks: */
  initialize_stack(gc_roots);
  initialize_stack(homogeneous);
  initialize_stack(large);

  /* Add all the default sizes for homogenous pages: */
  set_default_homogeneous_sizes();

  /* Compute the approximations to non-homogeneous sizes to their best
     homogeneous sizes: */
  set_inexact_sizes();

  /* Start the concurrent thread which decides whether we should collect: */
  start_gc_thread();
}

/* The caller *must* be sure that a homogeneous page for objects of size size_
   actually exists. */
word_t allocate_exact(integer_t size /* in words */){
  word_t r;
  homogeneous_page_t first_page;

  /* This is an *exact* allocation, so homogeneous_pages[size].non_full_pages must not
     be NULL: */
  r = allocate_from_homogeneous_page(homogeneous_pages[size].non_full_pages);
  if(r != NULL)
    return r;
  
  /* If we arrived here then the page was full. */
  
  /* Move the first page from the list of non-full pages to the list of full pages: */
  homogeneous_pages[size].non_full_pages =
    remove_pointed_homogeneous_page_from_list(homogeneous_pages[size].non_full_pages,
					      &first_page);
  homogeneous_pages[size].full_pages = 
    insert_homogeneous_page_into_list(first_page,
				      homogeneous_pages[size].full_pages);
  
  /* If there are no non-full pages left, then create a new page and attach it to the list: */
  if(homogeneous_pages[size].non_full_pages == NULL)
    homogeneous_pages[size].non_full_pages = create_homogeneous_page(size);
  
  /* Now try with the new head of homogeneous_pages[size].non_full_pages: */
  return allocate_exact(size);
}

/* If a homogeneous page for objects of size_ words does not exist a larger
   object is returned, either from a homogenous page or from a large page. */
word_t allocate_inexact(integer_t size /* in words */){
  /* If size is larger than the largest homogeneous object then create a large
     page: */
  if(size > largest_homogeneous_object){
    large_page_t new_large_page = create_large_page(size);
    list_of_large_pages = insert_large_page_into_list(new_large_page,
						      list_of_large_pages);
    return large_page_to_large_object(new_large_page);
  }
  
  /* If we arrived here then we can use an approximated size: */
  return allocate_exact(homogeneous_pages[size].inexact_size);
}

/* Sweeps every page of list, and then places it into the right list: 
   into *new_non_full_list if it is non-full, else 
   into *new_full_list.
   Returns the number of freed words. */
integer_t sweep_list_of_homogeneous_pages(homogeneous_page_t list,
					  homogeneous_page_t* new_non_full_list,
					  homogeneous_page_t* new_full_list){
  homogeneous_page_t p;
  integer_t r = 0;  
  
  while(list != NULL){
    /* Extract a page p from list: */
    list = remove_pointed_homogeneous_page_from_list(list, &p);

    /* Sweep p: */
    r += sweep_homogeneous_page(p);

/* destroy_homogeneous_page() does nothing on systems which can not free() blocks
   allocated with memalign(); it would be a waste of time to check the conditions
   on such systems: */
#ifdef CAN_FREE_MEMALIGNED_BLOCKS
#warning To do: destroy pages (where possible) only in major collections
    /* If p is now empty, and there is at least a page in the new non-full list
       (which must always be non-empty: it would make no sense to make it empty now
       and to create a new page later) then destroy it and skip to the next
       iteration: */
    if(is_homogeneous_page_empty(p) && (*new_non_full_list != NULL)){
      destroy_homogeneous_page(p);
      continue;
    }
#endif

    /* Place p in the right new list: */
    if(is_homogeneous_page_full(p))
      *new_full_list =
	insert_homogeneous_page_into_list(p, *new_full_list);
    else
      *new_non_full_list =
	insert_homogeneous_page_into_list(p, *new_non_full_list);
  } /* while */
  
  /* Return the number of freed words: */
  return r;
}

/* Clear the GC bits of every page in the list: */
void unmark_list_of_homogeneous_pages(homogeneous_page_t list){
  while(list != NULL){
    /* Unmark all the objects in the page pointed by list: */
    clear_gc_bits_in_homogeneous_page(list);

    /* Advance the pointer to the current list element: */
    list = homogeneous_page_to_next(list);
  } /* while */
}

/* Clear the GC bit of every page in the list: */
void unmark_list_of_large_pages(large_page_t list){
  while(list != NULL){
    /* Unmark the object in the page pointed by list: */
    unmark_large_page(list);
    
    /* Advance the pointer to the current list element: */
    list = large_page_to_next(list);
  } /* while */
}

/* Unmark all the objects in the heap: */
void unmark_all_objects(){
  integer_t i;
  
  /* For each possible size of homogeneous object: */
  i = homogeneous_pages[0].inexact_size;
  do{
    /* Unmark homogeneous objects of size i: */
    unmark_list_of_homogeneous_pages(homogeneous_pages[i].non_full_pages);
    unmark_list_of_homogeneous_pages(homogeneous_pages[i].full_pages);
    
    /* Make i become the next valid homogneous size, or exit: */
    if(i == largest_homogeneous_object)
      break;
    else
      i = homogeneous_pages[i + 1].inexact_size;
  }while(1);

  /* Unmark large pages: */
  unmark_list_of_large_pages(list_of_large_pages);
}

/* Sweep the non-full and full homogeneous pages of a given size.
   Returns the number of freed words: */
integer_t sweep_homogeneous_pages(integer_t size){
  homogeneous_page_t new_non_full_list = NULL;
  homogeneous_page_t new_full_list = NULL;
  lists_of_pages_t* current_lists = &(homogeneous_pages[size]);
  integer_t r = 0;

  homogeneous_page_t p;

  /* Sweep non-full pages: */
  r += sweep_list_of_homogeneous_pages(homogeneous_pages[size].non_full_pages,
				       &new_non_full_list,
				       &new_full_list);

  /* Sweep full pages: */
  r += sweep_list_of_homogeneous_pages(homogeneous_pages[size].full_pages,
				       &new_non_full_list,
				       &new_full_list);

  /* Update lists in homogeneous_pages[size]: */
  homogeneous_pages[size].non_full_pages = new_non_full_list;
  homogeneous_pages[size].full_pages = new_full_list;

  /* The non-full pages list must *not* be empty: if now it is, then
     add a new page: */
  if(homogeneous_pages[size].non_full_pages == NULL)
    homogeneous_pages[size].non_full_pages = create_homogeneous_page(size);
  
  /* Return the number of freed words: */
  return r;
}

/* Returns the number of freed words: */
integer_t sweep_large_pages(){
  large_page_t old_list = list_of_large_pages;
  large_page_t new_list_of_large_pages = NULL;
  large_page_t p;
  integer_t r = 0;

  while(old_list != NULL){
    /* Extract a page p from old_list: */
    old_list = remove_pointed_large_page_from_list(old_list, &p);

    /* If p is marked then and add it to the new list, else destroy it;
       the page will be unmarked only in major collections: */
    if(is_large_page_marked(p))
      new_list_of_large_pages =
	insert_large_page_into_list(p, new_list_of_large_pages);
    else
      r += destroy_large_page(p);
  } /* while */

  /* Make the new list to be the current list: */
  list_of_large_pages = new_list_of_large_pages;

  /* Return the number of freed words: */
  return r;
}

/* Returns the number of freed words: */
integer_t sweep(){
  integer_t i;
  time_t before, after;
  integer_t r = 0;
  
  if(verbose_gc){
    fprintf(stderr, "Sweeping: ");
    before = clock();
  }
  /* For each homogeneous size sweep all the pages and place each of
     them into the right list (non-full or full): */
  i = homogeneous_pages[0].inexact_size;
  do{
    /* Sweep the pages of size i: */
    r += sweep_homogeneous_pages(i);
    
    /* Make i become the next valid homogneous size, or exit: */
    if(i == largest_homogeneous_object)
      break;
    else
      i = homogeneous_pages[i + 1].inexact_size;
  }while(1);
  
  /* Sweep all large pages and destroy the dead ones: */
  r += sweep_large_pages();

  if(! is_stack_empty(gc_roots))
    fatal("gc_roots is not empty");
  if(! is_stack_empty(homogeneous))
    fatal("homogeneous is not empty");
  if(! is_stack_empty(large))
    fatal("large is not empty");

  if(verbose_gc){
    after = clock();
    fprintf(stderr, "%f seconds and freed "INTEGER_T_FORMAT" words.\n",
	    ((double) (after - before)) / CLOCKS_PER_SEC,
	    r);
  }

  /* We have just finished a collection: allocated words since last GC is 0 */
  allocated_words_since_last_gc = 0;
  
  /* Return the number of freed words: */
  return r;
}

/* If object is an heap pointer then push it onto the right stack
   (homogeneous_stack or large_stack), else do nothing. */
void push_onto_the_right_stack(word_t object){
  /* Don't even begin if object is not word-aligned: in this case
     it simply cannot be a pointer: */
  if(((unsigned_integer_t)object) % sizeof(word_t) != 0)
    return;
  
  /* Decice whether object is homogeneous, large, or not a heap
     pointer: */
  if(object_to_actual_homogeneous_page(object) != NULL){
    push_onto_stack(object, homogeneous);
  }
  else if(object_to_actual_large_page(object) != NULL){
    push_onto_stack(object, large);
  }
  /* else object is not a heap pointer. */
}

void add_gc_root(word_t p){
  push_onto_stack(p, gc_roots);
}

/* Add the content of a buffer to the GC roots word by word, from
   *buffer to *(buffer + words_no - 1) */
void add_gc_roots(word_t* buffer, integer_t words_no){
  integer_t i;
  for(i = 0; i < words_no; i++)
    push_onto_stack(buffer[i], gc_roots);
}

/* Mark object if it is not marked (it *must* belong to a homogeneous
   page); then, if object was not marked, push its elements that are
   heap pointers: */
void mark_homogeneous_object_and_push_its_children(word_t object){
  homogeneous_page_t homogeneous_page;
  integer_t number_of_elements;
  word_t element;
  integer_t i;
  
  /* Compute the page of object given its pointer (object *must* belong
     to a homogeneous page: I use this function because it is more
     efficient than object_to_actual_homogeneous_page(). */
  homogeneous_page = object_to_hypothetical_homogeneous_page(object);
  
  /* Is object already marked? If it is, then exit, else mark it and go on: */
  if(mark_in_homogeneous_page(object, homogeneous_page) != 0){
    /*
    fprintf(stderr, "The cons %p with car "INTEGER_T_FORMAT
	    " belongs to the hom. page %p and is MARKED\n",
	    object, *((integer_t*)object), homogeneous_page); */
    return;
  }
  else{
    /*
    fprintf(stderr, "The cons %p with car "INTEGER_T_FORMAT
	    " belongs to the hom. page %p and was NOT marked\n",
	    object, *((integer_t*)object), homogeneous_page); */
  }

  /* If we arrived here then object was not marked; compute the number of 
     its elements: */
  number_of_elements = homogeneous_page_to_object_size(homogeneous_page);

  /* Push all the elements that are pointers onto the right stack: */
  for(i = 0; i < number_of_elements; i++){
    element = ((word_t*)object)[i];
    push_onto_the_right_stack(element);
  }
}

/* Mark object if it is not marked (it *must* belong to a large
   page); then, if object was not marked, push its elements that are
   heap pointers: */
void mark_large_object_and_push_its_children(word_t object){
  integer_t number_of_elements;
  word_t element;
  integer_t i;
  
  /* Is object already marked? If it is, then exit, else mark it and go on: */
  if(mark_large_object(object) != 0)
    return;
  
  /* If we arrived here then object was not marked; compute the number of 
     its elements: */
  number_of_elements = large_object_to_object_size(object);

  /* Push all the elements that are pointers onto the right stack: */
  for(i = 0; i < number_of_elements; i++){
    element = ((word_t*)object)[i];
    push_onto_the_right_stack(element);
  }
}

/* GC roots must already be in the right stacks when this is called. */
void mark(){
  word_t object;
  clock_t before, after;

  if(verbose_gc){
    fprintf(stderr, "Marking: ");
    before = clock();
  }

  /* GC roots are in the stacks: pop and mark until both stacks are empty: */
  do{
    /* Work with the homogeneous stack until it becomes empty: */
    while(! is_stack_empty(homogeneous)){
      object = pop_from_stack(homogeneous);
      mark_homogeneous_object_and_push_its_children(object);
    }
    
    /* Work with the large stack until it becomes empty: */
    while(! is_stack_empty(large)){
      object = pop_from_stack(large);
      mark_large_object_and_push_its_children(object);
    }    
  }while(! (is_stack_empty(homogeneous) && is_stack_empty(large)));


  if(verbose_gc){
    after = clock();
    fprintf(stderr, "%f seconds\n",
	    ((double) (after - before)) / CLOCKS_PER_SEC);
  }
}

void mark_roots(){
  clock_t before, after;

  if(verbose_gc){
    fprintf(stderr, "Marking roots: ");
    before = clock();
  }

  while(! is_stack_empty(gc_roots)){
    word_t top;
    
    top = pop_from_stack(gc_roots);
    push_onto_the_right_stack(top);
  }
  if(verbose_gc){
    after = clock();
    fprintf(stderr, "%f seconds\n",
	    ((double) (after - before)) / CLOCKS_PER_SEC);
  }
}

static gc_cycles_counter = 0;

/* Mark roots, recursively mark the heap and sweep: */
void garbage_collect(){
  integer_t swept_words;
  integer_t is_this_gc_cycle_major;
  clock_t before, after;
  double gc_time;

  before = clock();
  is_this_gc_cycle_major = ((++gc_cycles_counter) % MINOR_GC_CYCLES_NO == 0);

  if(verbose_gc){
    if(is_this_gc_cycle_major)
      fprintf(stderr, "[MAJOR GC... ");
    else
      fprintf(stderr, "[Minor GC... ");
  }

  /* If this GC cycle is major then unmark all objects: */
  if(is_this_gc_cycle_major)
    unmark_all_objects();
  
  /* Do the actual GC (mark roots, mark and sweep): */
  mark_roots();
  mark();
  swept_words = sweep();
  
  /* Unset the 'should_we_collect' flag so that the concurrent thrad can
     continue to work, and wake it: */
  pthread_mutex_lock(&gc_mutex);
  should_we_collect = 0;
  pthread_cond_signal(&gc_cond);
  pthread_mutex_unlock(&gc_mutex);
  
  after = clock();
  gc_time =
    (integer_t)(((double) (after - before)) * 1000.0 / (double)CLOCKS_PER_SEC);
  total_gc_time += gc_time;

  if(verbose_gc){
    fprintf(stderr, "freed "INTEGER_T_FORMAT" bytes in "INTEGER_T_FORMAT"ms; heap size is "INTEGER_T_FORMAT" bytes]\n",
	    swept_words * sizeof(word_t),
	    (integer_t)gc_time,
	    total_heap_size * sizeof(word_t));
  }
}

/* Useful for profiling; return the total number of milliseconds spent in GC
   until now: */
double get_total_gc_time(){
  return total_gc_time;
}
