/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2003 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */


#ifndef GC_H_
#define GC_H_

#include "../eam_types.h"
#include <stdio.h>

/* This initializes the collector structures, adding default homogenous 
   sizes. */
void initialize_garbage_collector();

/* This only stops the concurrent thread updating should_we_collect: the
   allocator continues to work, and the heap objects are untouched. */
void terminate_garbage_collector();

/* The caller *must* be sure that a homogeneous page for objects of size size_
   actually exists. */
word_t allocate_exact(integer_t size_ /* in words */);

/* If a homogeneous page for objects of size_ words exists then this one behaves
   like allocate_exact(), else returns a larger object, either from a homogenous
   page or from a large page */
word_t allocate_inexact(integer_t size_ /* in words */);


/* All roots must be added *before each* collection. */
void add_gc_root(word_t p);

/* All roots must be added *before each* collection. 
   Add the content of a buffer to the GC roots word by word, from
   *buffer to *(buffer + words_no - 1) */
void add_gc_roots(word_t* buffer, integer_t words_no);

/* A flag suggesting when to call garbage_collect(). It's automatically set
   by a concurrent thread created by initialize_garbage_collector(), and should
   not need to be set by the user. When the flag has nonzero value it is time to
   collect. */
extern integer_t should_we_collect;

/* Collect not reachable data. This currently suspends the mutator. */
void garbage_collect();

/* Useful for profiling; return the total number of milliseconds spent in GC
   until now: */
double get_total_gc_time();

/* The following macros are used to tune the system. You should have no need to
   touch them. */

/* The concurrent thread which checks whether we should collect awakes every
   GC_TEST_TIMEOUT nanoseconds. This *must* be less then 1 second. */
#define GC_TEST_TIMEOUT 500000000 /* 1/2 second */

/* The current heuristic for updating should_we_collect says to collect when
   allocated_words_since_last_gc becomes greater than GC_THRESHOLD. */
#define GC_THRESHOLD (1 << 20) //8000000//2000000 /* in bytes */


/* Once every MINOR_GC_CYCLES_NO minor GC cycles there is a major GC cycle: */
#define MINOR_GC_CYCLES_NO 20

/* There are homogenous pages for objects of all the sizes 
   1, 2, 3, 4, ... MAXIMUM_SMALL_HOMOGENEOUS_SIZE.
   MAXIMUM_SMALL_HOMOGENEOUS_SIZE *must* be greater than 1. */
#define MAXIMUM_SMALL_HOMOGENEOUS_SIZE 8

/* And there are homogeneous pages for objects of all the sizes
   MAXIMUM_SMALL_HOMOGENEOUS_SIZE, MAXIMUM_SMALL_HOMOGENEOUS_SIZE * 2,
   MAXIMUM_SMALL_HOMOGENEOUS_SIZE * 4, ... up to MAXIMUM_HOMOGENEOUS_SIZE.
   Evan if MAXIMUM_HOMOGENEOUS_SIZE is not a power of two, pages for
   MAXIMUM_HOMOGENEOUS_SIZE-sized objects always do exist: */
#define MAXIMUM_HOMOGENEOUS_SIZE 1024

/* These are sizes of hash tables; it's better if these are prime numbers,
   since this statistically minimizes collisions: */
#define SIZE_OF_SET_OF_HOMOGENEOUS_PAGES 10007 
#define SIZE_OF_SET_OF_LARGE_OBJECTS     19997

#endif /* #ifndef GC_H_ */
