/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2003 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */


#ifndef HOMOGENEOUS_H_
#define HOMOGENEOUS_H_

#include "heap.h"


typedef struct homogeneous_page_header* homogeneous_page_t;

homogeneous_page_t create_homogeneous_page(size_t object_size);

/* Free the storage for the homogeneous page p. This does nothing on systems
   which can not free blocks allocated with memalign(). */
void destroy_homogeneous_page(homogeneous_page_t p);

integer_t homogeneous_page_to_object_size(homogeneous_page_t p);

/* Returns NULL on failure: */
word_t allocate_from_homogeneous_page(homogeneous_page_t p);

/* object must belong to p */
void free_from_homogeneous_page(word_t object, homogeneous_page_t p);

/* object must belong to p. Returns the GC bit as it was before: */
integer_t mark_in_homogeneous_page(word_t object, homogeneous_page_t p);

/* object must belong to p */
void unmark_in_homogeneous_page(word_t object, homogeneous_page_t p);

/* object must belong to p */
integer_t is_marked_in_homogeneous_page(word_t object, homogeneous_page_t p);

/* Sweep the page, unmark all surviving objects and returns the number of
   freed words */
int sweep_homogeneous_page(homogeneous_page_t p); /*Does *not* clear GC bits.*/

int is_homogeneous_page_full(homogeneous_page_t p);

int is_homogeneous_page_empty(homogeneous_page_t p);

/* Clear all GC bits of p, which must be an existing homogeneous page. */
void clear_gc_bits_in_homogeneous_page(homogeneous_page_t p);

/* This is only a computation on pointers; it is fast, but does *not* check
   whether object actually belongs to a homogeneous page. */
homogeneous_page_t object_to_hypothetical_homogeneous_page(word_t object);

/* Returns NULL if the object does not belong to an homogeneous page,
   else return its page: */
homogeneous_page_t object_to_actual_homogeneous_page(word_t object);

/* Both pages and lists are represented as homogeneous_page_t's. This enables us to
   make some optimizations. */
homogeneous_page_t homogeneous_page_to_previous(homogeneous_page_t p);
homogeneous_page_t homogeneous_page_to_next(homogeneous_page_t p);

/* Returns NULL if the list is empty: */
homogeneous_page_t first_element_of_list_of_homogeneous_pages(homogeneous_page_t list);

/* Destructively updates p and list, returning the new version of list: */
homogeneous_page_t insert_homogeneous_page_into_list(homogeneous_page_t p,
                                         homogeneous_page_t list);

/* Destructively updates list, returning the new version of it.
   The address of the removed page is copied into the location pointed by
   removed: */
homogeneous_page_t remove_pointed_homogeneous_page_from_list(homogeneous_page_t list,
                                                 homogeneous_page_t* removed);

/* Useful for debugging */
void dump_list_of_homogeneous_pages(homogeneous_page_t list);
void dump_free_list(homogeneous_page_t list);

#endif /* #ifndef HOMOGENEOUS_H_ */
