/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2003 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */


#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <limits.h>
#include <math.h>

#include "set.h"
#include "heap.h"
#include "large.h"

#undef DEBUG

/* The header is the first part of a page: in the same malloc()'d block there
   is the payload */
struct large_page_header{ /* Size must be a multiple of a word size */
  large_page_t previous; /* all large pages are in a doubly-linked list */
  large_page_t next;     /* all large pages are in a doubly-linked list */
  size_t object_size; /* in words */
  integer_t gc_bit; /* This is used like a GC bit, but it is a whole word.
		       Using a single bit would make no sense; conceptually
		       it's a boolean, and the structure's size must be a
		       multiple of a word. */
};

/* This works since large_page_header has a size which is multiple of a word */
#define HEADER_SIZE (sizeof(struct large_page_header) / sizeof(word_t)) /* in words */

/* Defined in gc.c: */
extern set_t set_of_large_objects;

large_page_t create_large_page(size_t object_size){
  large_page_t r;
  
#ifdef DEBUG
  fprintf(stderr, "Creating a large page for objects of %i words...\n", object_size);
#endif
  /* Allocate storage and check for malloc() failure: */
  r = (large_page_t) malloc(sizeof(word_t) * (HEADER_SIZE + object_size));
  if(r == NULL)
    fatal("could not allocate large page");
  
  /* Initialize fields: */
  r->previous = NULL;
  r->next = NULL;
  r->object_size = object_size;
  r->gc_bit = 0;

  /* Insert the object into the set of large objects: */
  insert_pointer_into_set(large_page_to_large_object(r), set_of_large_objects);

  /* Increment total heap size: */
  total_heap_size += r->object_size;

  /* Increment the global counter of allocated words: */
  allocated_words_since_last_gc += r->object_size;

  /* Return the allocated object: */
  return r;
}

/* Return the number of freed words, i.e. the size of the object
   held by p: */
integer_t destroy_large_page(large_page_t p){
  integer_t r = p->object_size;

  /* Remove the object from the set of large objects: */
  remove_pointer_from_set(large_page_to_large_object(p), set_of_large_objects);

  /* Decrement total heap size: */
  total_heap_size -= p->object_size;

  /* Decrement the global counter of allocated words: */
  allocated_words_since_last_gc -= p->object_size;

#ifdef DEBUG
  fprintf(stderr, "Destroyed a large page of "INTEGER_T_FORMAT" words.\n",
	  p->object_size);
#endif

  /* Free the storage for p, which includes payload. */
  free(p);

  /* Return the number of freed words: */
  return r;
}

integer_t large_page_to_object_size(large_page_t p){
  return p->object_size;
}

/* object *must* belong to a large page. */
integer_t large_object_to_object_size(word_t object){
  return object_to_hypothetical_large_page(object)->object_size;
}

/* This does *not* check that p exists as a large page; it's only a
   computation on pointers to find the object which belongs to the
   page *if* the page exists. */
word_t large_page_to_large_object(large_page_t p){
  return (word_t)(((word_t*)p) + HEADER_SIZE);
}

/* This does *not* check that object actually belongs to a large page;
   it's only a computation on pointers to find the page which the object
   *can* belong to. */
large_page_t object_to_hypothetical_large_page(word_t object){
  return (large_page_t)(((word_t*)object) - HEADER_SIZE);
}

/* object must belong to a large page. Returns the GC bit as it was before: */
integer_t mark_large_object(word_t object){
  /* If the object was already marked then return 1 and do nothing: */
  if(object_to_hypothetical_large_page(object)->gc_bit != 0)
    return 1;
  
  /* Mark the only object in the page: */
  object_to_hypothetical_large_page(object)->gc_bit = 1;

  /* If we arrived here then the object was not previously marked: */
  return 0;
}

void unmark_large_object(word_t object){
  /* Unmark the only object in the page: */
  object_to_hypothetical_large_page(object)->gc_bit = 0;
}

void unmark_large_page(large_page_t p){
  p->gc_bit = 0;
}

integer_t is_large_object_marked(word_t object){
  /* Check the only object in the page (the object pointer itself is not
     needed): */
  return object_to_hypothetical_large_page(object)->gc_bit;
}

integer_t is_large_page_marked(large_page_t p){
  /* Check the only object in the page (the object pointer itself is not
     needed): */
  return p->gc_bit;
}

integer_t does_object_belong_to_a_large_page(word_t object){
  return does_pointer_belong_to_set(object, set_of_large_objects);
}

/* Returns NULL if the object does not belong to a large page,
   else return its page: */
large_page_t object_to_actual_large_page(word_t object){
  /* If the object is not word-aligned then fail: */
  if((unsigned_integer_t)((char*)object) % sizeof(word_t) != 0)
    return NULL;

  /* Check whether the object actually belongs to a large page: if it does
     then return its large page, else return NULL: */
  if(does_object_belong_to_a_large_page(object))
    return object_to_hypothetical_large_page(object);
  else
    return NULL;
}

large_page_t insert_large_page_into_list(large_page_t p, large_page_t list){
  /* Do nothing if p is NULL: */
  if(p == NULL)
    return list;
  
  /* Update the links: */
  p->next = list;
  p->previous = NULL;
  if(list != NULL)
    list->previous = p;
  
  /* Return the new list which starts with p: */
  return p;
}

/* Destructively updates p and list. The address of the removed page is
   copied into the location pointed by removed: */
large_page_t remove_pointed_large_page_from_list(large_page_t list, 
						 large_page_t* removed){
  large_page_t former_previous;
  large_page_t former_next;

  /* Just return if the list is null: */
  if(list == NULL){
    *removed = NULL;
    return NULL;
  }

  /* Update pointers: */
  former_previous = list->previous;
  former_next = list->next;
  if(former_previous != NULL)
    former_previous->next = former_next;
  if(former_next != NULL)
    former_next->previous = former_previous;

  /* Update *removed with the removed page and return the updated list: */
  list->previous = list->next = NULL;
  *removed = list;
  if(former_previous != NULL)
    return former_previous;
  else
    return former_next;
}

large_page_t first_element_of_list_of_large_pages(large_page_t list){
  /* Return NULL if the list is empty */
  if(list == NULL)
    return NULL;

  /* Find beginning: */
  while(list->previous != NULL)
    list = list->previous;

  /* Return the first element: */
  return list;
}

large_page_t large_page_to_previous(large_page_t p){
  return p->previous;
}

large_page_t large_page_to_next(large_page_t p){
  return p->next;
}

void dump_list_of_large_pages(large_page_t list){
  large_page_t p;
  
  if(list == NULL){
    printf("[ ]\n");
    return;
  }
  
  /* Find beginning: */
  p = first_element_of_list_of_large_pages(list);

  /* Output list: */
  printf("List of large pages: [ ");
  while(p != NULL){
    if(p == list)
      printf("<*>");
    printf("%p ", p);
    p = p->next;
  }
  printf("]\n");
}
