/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2003 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */


#ifndef LARGE_H_
#define LARGE_H_

#include "heap.h"


typedef struct large_page_header* large_page_t;

large_page_t create_large_page(size_t object_size);

/* Return the number of freed words, i.e. the size of the object
   held by p: */
integer_t destroy_large_page(large_page_t p);

integer_t large_page_to_object_size(large_page_t p);

/* object *must* belong to a large page. */
integer_t large_object_to_object_size(word_t object);

/* This does *not* check that p exists as a large page; it's only a
   computation on pointers to find the object which belongs to the
   page *if* the page exists. */
word_t large_page_to_large_object(large_page_t p);

/* This does *not* check that object actually belongs to a large page;
   it's only a computation on pointers to find the page which the object
   *can* belong to. */
large_page_t object_to_hypothetical_large_page(word_t object);

/* object must belong to a large page. Returns the GC bit as it was before: */
integer_t mark_large_object(word_t object);

void unmark_large_object(word_t object);
void unmark_large_page(large_page_t p);

integer_t is_large_object_marked(word_t object);
integer_t is_large_page_marked(large_page_t p);

integer_t does_object_belong_to_a_large_page(word_t object);

/* Returns NULL if the object does not belong to a large page,
   else return its page: */
large_page_t object_to_actual_large_page(word_t object);

/* Both pages and lists are represented as large_page_t's. This enables us to
   make some optimizations. */
large_page_t large_page_to_previous(large_page_t p);
large_page_t large_page_to_next(large_page_t p);

/* Returns NULL if the list is empty: */
large_page_t first_element_of_list_of_large_pages(large_page_t list);

/* Destructively updates p and list, returning the new version of list: */
large_page_t insert_large_page_into_list(large_page_t p,
					 large_page_t list);

/* Destructively updates list, returning the new version of it.
   The address of the removed page is copied into the location pointed by
   removed: */
large_page_t remove_pointed_large_page_from_list(large_page_t list,
						 large_page_t* removed);

/* Useful for debugging */
void dump_list_of_large_pages(large_page_t list);

#endif /* #ifndef LARGE_H_ */
