/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2003 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */


/* A simple and efficient implementation of a stack of pointers made with
   macros. In the garbage-collector speed is crucial. */

#ifndef STACK_H_
#define STACK_H_

#include "../eam_types.h"

/* To be used in .c files: */
#define DEFINE_STACK(name) \
  word_t* name ## _stack; \
  integer_t name ## _stack_size; \
  integer_t name ## _stack_elements;

/* To be used in headers: */
#define DECLARE_STACK(name) \
  extern word_t* name ## _stack; \
  extern integer_t name ## _stack_size; \
  extern integer_t name ## _stack_elements;

/* Allocate storage and initialize index: */
#define initialize_stack(name) { \
  name ## _stack_elements = 0; \
  name ## _stack_size = INITIAL_STACK_SIZE; \
  name ## _stack = (word_t*) xmalloc(sizeof(word_t) * INITIAL_STACK_SIZE); \
}

/* Free storage: */
#define destroy_stack(name) \
  free(name ## _stack)

/* Push a pointer onto a stack: */
#define push_onto_stack(pointer, name) \
  (PUSH_ONTO_STACK(pointer, name ## _stack, name ## _stack_size, name ## _stack_elements))

/* Pop pointer from a stack: */
#define pop_from_stack(name) \
  (POP_FROM_STACK(name ## _stack, name ## _stack_elements))

/* Returns the top of a stack, without popping: */
#define top_of_stack(name) \
  (name ## _stack [name ## _stack_elements - 1])

/* Returns 0 iff the stack is empty: */
#define is_stack_empty(name) \
  IS_STACK_EMPTY(name ## _stack_elements)  


/* The following part should not be needed to the user: */


#define INITIAL_STACK_SIZE 1024

#define PUSH_ONTO_STACK(pointer, stack, stack_size, stack_elements) { \
  /* Enlarge the GC stack if needed: */ \
  \
  if(stack_elements == stack_size){ \
    /*fprintf(stderr, "<STACK GROWS TO "INTEGER_T_FORMAT" ELEMENTS>", stack_elements);*/ \
    stack_size *= 2; \
    stack = (word_t*) \
      realloc(stack, sizeof(word_t) * stack_size); \
    if(stack == NULL) \
      fatal("could not enlarge stack"); \
  } \
  \
  /* Push element: */ \
  stack[stack_elements++] = pointer; \
}

#define POP_FROM_STACK(stack, stack_elements) \
  (stack[--stack_elements])

#define IS_STACK_EMPTY(stack_elements) \
  (stack_elements == 0)

#endif /* #ifndef STACK_H_ */
