/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2003 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */


#include "c_libraries.h"
#include "dynamic_libraries.h"
#include "../../common/string_map.h"
#include "../../common/search_path.h"
#include <stdio.h>

/* C libraries are implemented with dynamic libraries, but it is useful to
   keep them conceptually separate, since they serve different purposes.
   See the comment in the header. */

/* defined in dynamic_libraries.c, only needed here */
extern struct string_map *c_library_to_handle;
extern search_path_t c_libraries_search_path;

c_library_handle_t open_c_library(char* library_name){
  c_library_handle_t * pointer;
  c_library_handle_t r;
  void (*initialize_function)();
  char* library_filename;
  char* library_full_pathname;

  /* If we have already opened this same library, then return the handle we
     obtained the first time. It's important not to call initialize_c_library()
     more than once for each library: */
  pointer = access_string_map(c_library_to_handle, library_name);
  if(pointer != NULL){ /* aldready loaded */
  //fprintf(stderr, "open_c_library(): %s was already initialized.\n", library_name);
    return *pointer;
  }
  
  /* Ok, if we arrived here then we have not opened library_name yet. */

  /* Compute the filename and the full pathname of the library: */
  //  fprintf(stderr, "Opening C library %s\n", library_name);
  library_filename = (char*)
    malloc(strlen(library_name) + 10);
  sprintf(library_filename, "%s.c.so", library_name);
  //  fprintf(stderr, "Opening C library with filename %s\n", library_filename);
  library_full_pathname =
    search_for_file_in_search_path(library_filename, c_libraries_search_path);
  if(library_full_pathname == NULL){
    fprintf(stderr,
	    "Looking for %s in %s :\n",
	    library_filename,
	    search_path_to_string(c_libraries_search_path));
    fatal("coult not find file in search path");
  }
  //  fprintf(stderr, "Opening C library with full pathname %s\n", library_full_pathname);

  r = open_dynamic_library(library_full_pathname);

  /* Free the strings we don't need any more: */
  free(library_filename);
  free(library_full_pathname);
  
  /* Execute the initialization code of the C library: */
  initialize_function = load_symbol_from_handle(r, "initialize_c_library");
  initialize_function();

  /* Remember we opened the C library, and remember its handle: */
  insert_into_string_map(c_library_to_handle, library_name, &r);

  return r;
}

c_value_t load_c_symbol(c_library_handle_t handle, char* symbol_name){
  c_value_t r;
  //fprintf(stderr, "Loading C symbol %s from %p\n", symbol_name, handle);
  r = load_symbol_from_handle(handle, symbol_name);
  //fprintf(stderr, "We got %p\n", r);
  return r;
}

/* Used in use_everything.c */
integer_t c_libraries_constant = 10;
