/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2003 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */


#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <dlfcn.h>

#include "../eam_types.h"
#include "../../common/string_map.h"
#include "../../common/search_path.h"
#include "../../common/malloc.h"
#include "../prefixes.h"
#include "dynamic_libraries.h"


/* Global data structures for dynamic libraries: */
static struct string_map* dynamic_library_to_handle;
static void** handles_stack;
static int handles_stack_pointer;
#define INITIAL_HANDLES_STACK_SIZE 1 // To do: make this bigger after testing
static int handles_stack_size;

/* Used for C libraries (it's very important not to call each initialization
   function more than once). We only *initialize* the structure here, all the
   updates will be made in c_libraries.c.
   This is in order not to require two different initialization functions. */
struct string_map* c_library_to_handle;
search_path_t c_libraries_search_path;

void initialize_dynamic_libraries(){
  char* value_of_environment_variable;

  /* Initialize structures: */
  dynamic_library_to_handle = create_string_map(sizeof(void* /* handle */));
  handles_stack_pointer = -1;
  handles_stack_size = INITIAL_HANDLES_STACK_SIZE;
  handles_stack = (void**) xmalloc(sizeof(void*) * handles_stack_size);
  c_library_to_handle = create_string_map(sizeof(void* /* handle */));

  /* Create the search path: */
  c_libraries_search_path = create_search_path();

  /* Prepend the directories in EPSILON_C_LIBRARIES_PATH: */
  value_of_environment_variable = getenv("EPSILON_C_LIBRARIES_PATH");
  if(value_of_environment_variable != NULL)
    append_path_to_search_path(value_of_environment_variable, 
			       c_libraries_search_path);
  
  /* Append default directories: */
  append_directory_to_search_path(".",
				  c_libraries_search_path);
  append_directory_to_search_path(EPSILON_C_LIBRARIES_PATH,
				  c_libraries_search_path);
}

void close_dynamic_libraries(){
  int i;

  /* dclose() every dlopen()ed library: */
  for(i = handles_stack_pointer; i >= 0; i--)
    dlclose(handles_stack[i]);

  /* Destroy structures: */
  destroy_string_map(dynamic_library_to_handle);
  free(handles_stack);
  destroy_string_map(c_library_to_handle);
  destroy_search_path(c_libraries_search_path);
}

/* Fatal error on failure. */
dynamic_library_handle_t open_dynamic_library(const char* name){
  dynamic_library_handle_t handle;

  /* If we have already dlopen()ed this library then return the same
     handle and do nothing more: */
  if(is_string_map_defined_on(dynamic_library_to_handle, name)){
    dynamic_library_handle_t handle =
      *((void**)(access_string_map(dynamic_library_to_handle, name)));
    //fprintf(stderr, "The library %s (handle %p) was already dlopen()ed\n", name, handle);
    return handle;
  }

  /* Try dlopen()ing the library: */
  handle = dlopen(name,
		  RTLD_GLOBAL | /* symbols will be seen by next dlopen()s */
	     	  RTLD_LAZY);   /* load each symbol when needed */
  // 		  RTLD_NOW);    /* load all symbols *now*, or fail */

  /* If dlopen() failed then exit with a fatal error: */
  if(handle == NULL)
    fatal(dlerror());

  /* If we arrived here then store the handle: push onto the stack... */
  if(handles_stack_pointer == handles_stack_size){ /* double the stack: */
    //fprintf(stderr, "\t[Doubling the stack]\n");
    handles_stack_size *= 2;
    handles_stack = xrealloc(handles_stack,
			     sizeof(void*) * handles_stack_size);
  }
  handles_stack[++handles_stack_pointer] = handle;
  
  /* ...and insert an entry into the string map: */
  insert_into_string_map(dynamic_library_to_handle, name, &handle);

  //fprintf(stderr, "Loaded %s (handle %p)\n", name, handle);

  /* Also return the handle: */
  return handle;
}

/* Try loading the symbol. In case of error set *failed to a
   meaningful string error message and return NULL, else set
   it to NULL and return a valid pointer (which however *can*
   be NULL!). The message must be used before calling any other
   function in this compilation unit, since it comes from a static
   buffer which can be overwritten.
   Only for internal use, not exported.
   *Not* thread-safe. */
symbol_value_t failibly_load_symbol_from_handle(dynamic_library_handle_t handle,
						const char* symbol,
						char** failed){
  symbol_value_t r;

  //fprintf(stderr, "Search for %s in (handle %p)\n", symbol, handle);

  /* Try dlsym()ming the symbol: */
  r = dlsym(handle, symbol);

  /* dlsym() failed? */
  *failed = dlerror(); /* NULL in case of success */
  if(*failed != NULL)  /* we actually failed. */
    return NULL;
  
  /* If we arrived here then return the value of the symbol and set
     *failed to NULL: */
  *failed = NULL;
  return r;
}

/* Fatal error on failure.
   *Not* thread-safe. */
symbol_value_t load_symbol_from_handle(dynamic_library_handle_t handle,
				       const char* symbol){
  static char* error_message;
  symbol_value_t r;

  /* Call the failable version: */
  r = failibly_load_symbol_from_handle(handle,
				       symbol,
				       &error_message);
  if(error_message != NULL) /* we failed */
    fatal(error_message);

  /* If we arrived here then we didn't fail. */
  return r;
}

/* Fatal error on failure.
   *Not* thread-safe. */
symbol_value_t load_symbol_from_dynamic_library(const char* library,
						const char* symbol){
  dynamic_library_handle_t* handle_pointer =
    access_string_map(dynamic_library_to_handle, library);

  /* Check if the library was actually loaded: */
  if(handle_pointer == NULL) /* this should not happen */
    fatal("tried to use an handle for a library which wasn't loaded");

  /* Load the symbol from the handle we found: */
  return load_symbol_from_handle(*handle_pointer, symbol);
}

/* Try loading the symbol from each dlopen()'ed library, starting from the one
   which was dlopen()'d more recently.
   Fatal error if the symbol is not found in any library.
   *Not* thread-safe. */
symbol_value_t load_symbol(const char* symbol){
  symbol_value_t r;
  int i;
  
  for(i = handles_stack_pointer; i >= 0; i--){
    static char* failed;
    r = failibly_load_symbol_from_handle(handles_stack[i],
					 symbol,
					 &failed);
    if(! failed)
      return r; /* we found a valid definition of symbol: return */
  }
  
  /* If we arrived here then we didn't found the definition. Fatal. */
  fprintf(stderr, "About the symbol %s:\n", symbol);
  fatal("could not find the symbol in any opened library");
  return 0; /* just to avoid a GCC warning */
}

/* Used in use_everything.c */
integer_t dynamic_libraries_constant = 10;

/*
int main(){
  void* handle;
  double (*sine)(double);

  initialize_dynamic_libraries();
  handle = open_dynamic_library("libm.so");
  handle = open_dynamic_library("libxml.so");
  handle = open_dynamic_library("libX11.so");
  handle = open_dynamic_library("libgtkhtml-2.so");
  handle = open_dynamic_library("/usr/gnome2/lib/libglade/2.0/libgnome.so");
  handle = open_dynamic_library("libncurses.so");
  handle = open_dynamic_library("libguile.so");
  handle = open_dynamic_library("libpgeasy.so");
  handle = open_dynamic_library("libpgtcl.so");
  handle = open_dynamic_library("libXt.so");
  handle = open_dynamic_library("libXt.so");
  handle = open_dynamic_library("libgdk.so");
  handle = open_dynamic_library("libgtk.so");
  handle = open_dynamic_library("libglut.so");
  handle = open_dynamic_library("libz.so");
  handle = open_dynamic_library("libxmms.so");
  handle = open_dynamic_library("libxpcom.so");
  handle = open_dynamic_library("libunicode.so");
  handle = open_dynamic_library("libpng.so");
  handle = open_dynamic_library("libaa.so");
  handle = open_dynamic_library("libtalkfilters.so");

  //sine = load_symbol_from_dynamic_library("libm.so", "sin");
  //sine = load_symbol_from_handle(handle, "sin");
  sine = load_symbol("sin");

  printf("The function returned %f\n", sine(M_PI / 2.0));

  close_dynamic_libraries();

  return 0;
}
*/
