/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2003 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */


#include "../eam_types.h"
#include "../../common/malloc.h"
#include "../gc/gc.h"
#include "list.h"

integer_t length_of_list(word_t list){
  integer_t length = 0;
  
  /* Increment length once for each element: */
  while(list != NULL){
    length++;
    list = ((word_t*)list)[1];
  }

  return length;
}

/* Convert the given C array with the given size into an epsilon list,
   allocated with allocate_exact(): */
word_t c_array_to_list(word_t array, integer_t size){
  integer_t i;
  word_t* list = NULL;
  
  /* Copy each element of the array into a list: */
  for (i = size - 1; i >= 0; i--){
    word_t* new_list = allocate_exact(2);
    new_list[0] = ((word_t*)array)[i];
    new_list[1] = list;
    list = new_list;
  }
  
  /* Return the list we built: */
  return list;
}

/* Convert the given C array with the given size into an epsilon array,
   allocated with allocate_inexact(): */
word_t c_array_to_epsilon_array(word_t array, integer_t size){
  word_t* r = allocate_inexact(size + 1);
  integer_t i;

  /* Fill the epsilon array: */
  r[0] = (word_t)((integer_t)size);
  for(i = 0; i < size; i++)
    r[i + 1] = ((word_t*)array)[i];

  /* Return the epsilon array: */
  return (word_t)r;
}

/* Convert the given epsilon array into a list, allocated with
   allocate_exact(): */
word_t epsilon_array_to_list(word_t array){
  integer_t size = ((integer_t*)array)[0]; /* size of array */

  return c_array_to_list(((word_t*)array) + 1, size);
}

/* Convert the given list into an epsilon array, allocated with
   allocate_inexact(): */
word_t list_to_epsilon_array(word_t list){
  integer_t i, length = length_of_list(list);
  word_t pointer = list; /* A pointer used to visit the list */
  word_t* array;

  /* Allocate the array: */
  array = allocate_inexact(length + 1);
  
  /* Fill the array: */
  array[0] = (word_t)length;
  for(i = 1; i <= length; i++){
    array[i] = ((word_t*)pointer)[0];
    pointer = ((word_t*)pointer)[1];
  }
  
  return array;
}

/* Used in use_everything.c */
integer_t list_constant = 10;
