/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2002, 2003 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */

#ifndef RUNTIME_H_
#define RUNTIME_H_

#include <errno.h>
#include "primitive_exceptions.h"

typedef struct exceptions_stack_element{
  word_t handler_pointer; /* the instruction where the handler begins */
  integer_t frame_pointer;
  integer_t undertop_stack_pointer;
  word_t* environment;
} exception_t;

extern char** exceptions_names; /* defined in evm_c.head or in the generated C code */

#define UNDERTOP \
  (stack[undertop_stack_pointer])

#define UNDERUNDERTOP \
  (stack[undertop_stack_pointer - 1])

/*
#define ASSIGN_MY_MALLOC(x, words) \
  x = malloc((words) * sizeof(word_t))
*/

#define ASSIGN_MY_MALLOC(x, words_no) \
  x = allocate_inexact(words_no)

#define ASSIGN_CREATE_ARRAY(x, elements) \
  ASSIGN_MY_MALLOC(x, elements)

#define PUSH_OBJECT(x) { \
  stack[++undertop_stack_pointer] = top; \
  top = x; \
}

#define POP \
  top = stack[undertop_stack_pointer--]

/* n must be > 0 */
#define POP_MULTIPLE(n) \
  top = stack[(undertop_stack_pointer -= n) + 1];

#define ASSIGN_CONS(x, a, b) \
  ASSIGN_MY_MALLOC(x, 2); x[0]=a; x[1]=b

#define ENLARGE_STACK(minimum_needed_size) {\
  /* Double the dimension if it is needed to satisfy the constraint:   */ \
  /* It can be shown that _one_ doubling is enough if the initial size */ \
  /* is sufficient for one frame: */ \
  if(stack_size < minimum_needed_size) \
    stack = (word_t*) realloc(stack, \
			     sizeof(word_t) * (stack_size *= 2)); \
}

#define REDUCE_STACK(minumum_allowed_size) { \
  /* Keep reducing the dimension until it gets too small by a single */ \
  /* doubling: */ \
  while(stack_size > minimum_allowed_size) \
    stack_size /= 2; \
  /* Double the dimension so that we respect the constraint: */ \
  stack_size *= 2; \
  /* Resize the stack: */ \
  stack = (word_t*) realloc(stack, sizeof(word_t) * stack_size); \
}

/* Complex macro use.
   This macro converts 12 into L12, etc. When I need to labelify a
   parameter in another macro I MUST call this; directly labelifying 
   it would paste together L and the _name_ of the parameter */
#define LABELIFY(n) \
  L ## n

#ifdef THREADING
#define GOTO_IMMEDIATE(x) \
  goto* x		//goto LABELIFY(x)
#else
#warning To do: implement jumps without threading in generated C
#endif /* THREADING */

#ifdef THREADING
#define GOTO_GENERIC(x) \
     goto* ((word_t)x)
#else
#warning To do: implement generic jumps without threading in generated C
#endif /* THREADING */

#define THROW(TYPE, VALUE) \
  {/* Set exception type and exception value, so that they are available to the handler: */ \
    exception_type = (integer_t)(TYPE);\
    exception_value = (word_t)(VALUE);\
    \
    /* Update FP, SP and environment: */ \
    frame_pointer = \
      exceptions_stack[exceptions_stack_pointer].frame_pointer; \
    if(undertop_stack_pointer != exceptions_stack[exceptions_stack_pointer].undertop_stack_pointer) \
      top = stack[exceptions_stack[exceptions_stack_pointer].undertop_stack_pointer + 1]; \
    undertop_stack_pointer = \
      exceptions_stack[exceptions_stack_pointer].undertop_stack_pointer; \
    environment = \
      exceptions_stack[exceptions_stack_pointer].environment; \
    \
    /* Jump to the handler: */ \
    GOTO_GENERIC(exceptions_stack[exceptions_stack_pointer].handler_pointer); \
  }

/* Test the flag updated by the concurrent thread, and if it is not zero then
   mark roots and collect: */
#define GC_IF_NEEDED() { \
  if(should_we_collect){ \
    MARK_GC_ROOTS();  \
    garbage_collect(); \
  } \
}

/* Stop the GC thread and exit with the given return code: */
#define EXIT_EAM(EXIT_CODE) { \
  double end_time = ((double)clock()) * 1000.0 / (double)CLOCKS_PER_SEC; \
  double runtime = end_time - begin_time; \
  double gc_time = get_total_gc_time(); \
  \
  if(long_option_to_value("benchmark")){ \
    fprintf(stderr, "Total runtime was "INTEGER_T_FORMAT"ms.\n", \
	    (integer_t)runtime); \
    fprintf(stderr, "GC time was "INTEGER_T_FORMAT"ms.\n", \
	    (integer_t)gc_time); \
    if(runtime != 0.0) \
      fprintf(stderr, "GC time was "INTEGER_T_FORMAT"%%.\n", \
	      (integer_t)(gc_time / runtime * 100.0)); \
    fprintf(stderr, "Exiting with code " INTEGER_T_FORMAT "...\n", EXIT_CODE); \
  } \
  terminate_garbage_collector(); \
  exit((int)(integer_t)EXIT_CODE); \
}

#endif /* #ifndef RUNTIME_H_ */
