;;; This file is part of GNU epsilon
;;; Copyright (C) 2003 Luca Saiu

;;; GNU epsilon is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published
;;; by the Free Software Foundation; either version 2, or (at your
;;; option) any later version.

;;; GNU epsilon is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;; General Public License for more details.

;;; You should have received a copy of the GNU General Public License
;;; along with epsilon; see the file COPYING.  If not, write to the
;;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;;; Boston, MA 02111-1307, USA.



;;; The "try" macro emulates the exception handling machinery of GNU epsilon in
;;; Scheme.
;;; See the example at the end of this file.


;;; (throw exception-name parameters) is the form intended to be
;;; directly called, which is expanded into
;;; (throw-low-level 'exception exception-name parameters)
(defmacro throw (exception-name parameters)
  `(throw-low-level (list 'exception ,exception-name ,parameters)))

;;; (try-low-level body exception variable-for-parameter handler) is the
;;; form internally used. It only enables to use *one* handler.
(defmacro try-low-level (b e parameters-variable h)
  `(let ((result (call-with-current-continuation
		  (lambda (throw-low-level)
		    ,b))))
;     (display "Debug: ")(display result)(newline)
     (if (pair? result)
	 (if (eq? (car result) 'exception)
	     (if (eq? (cadr result) ,e)
		 (let ((,parameters-variable (caddr result)))
		   ,h)
		 (throw-low-level (cons 'exception (cdr result)))) ; re-throw
	     result)
	 result)))

;;; The user-friendly form is (try body handlers), where handlers is a list of
;;; lists of three elements: exception-name, variable-for-parameter, handler.
(defmacro try (body handlers)
  (do ((h handlers (cdr h))
       (r body (list 'try-low-level
		     r
		     (car (car h))      ; exception name
		     (cadr (car h))     ; variable for exception parameters
		     (caddr (car h))))) ; handler
      ((null? h) r)))

#!
;; An example:
(try
 (throw 'e1 'parameters-of-e1) ; body
 (('e1 x (begin ; exception-name, parameter-variable, handler
	   (display "e1 ")
	   (display x)
	   (newline)))
  ('e2 x (begin ; exception-name, parameter-variable, handler
	   (display "e2 ")
	   (display x)
	   (newline)))))
!#
