/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2002, 2003 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <ctype.h>
#include <errno.h>

/* Either use the real readline() or, if the real readline() is not
   available, use a simple replacement: */
#include "../common/readline_wrapper.h"

#include "../commontext.h"
#include "../common/license.h"
#include "temporary.h"
#include "../compiler/epsilon.h"
#include "../compiler/multibuffer.h"
#include "../compiler/types.h"
#include "../compiler/environment.h"
#include "../compiler/epsilonparser.h"
#include "structures.h"

#include "../common/command_line.h"

//int is_a_repl_command=0, is_a_define=0, is_an_undefine=0, is_an_import=0, is_an_unimport=0;

void yyerror(char* string){
  /* Do nothing: this version should not show any message. */
}

void show_logo(){
  printf(
"-------------------------------------------------------------------------------\n"
"                                        i  ll                                  \n"
"                 eeeeee                     l                  version " VERSION_STRING "\n"
"                ee      p pppp   ssss  ii   l   oooo  n nnnn                   \n"
"                 eeee    p    p s       i   l  o    o  nn   n                  \n"
"                ee       p    p  sses   i   l  o    o  n    n                  \n"
"                ee       p    p      s  i   l  o    o  n    n                  \n"
"                 eeeeee  ppppp   sses  iii lll  oooo   n    n                  \n"
"                         p                                                     \n"
"                         p                  http://www.gnu.org/software/epsilon\n"
"                        ppp                                                    \n"
"-------------------------------------------------------------------------------\n"
/*
"-------------------------------------------------------------------------------\n"
"                    eeee               i  ll                                   \n"
"                   ee    p ppp   sss       l   ooo  n nnn                      \n"
"                    eee   p   p s     ii   l  o   o  nn  n                     \n"
"                   ee     p   p  sss   i   l  o   o  n   n                     \n"
"                    eeee  pppp      s iii lll  ooo   n   n                     \n"
"                          p      sss                                           \n"
"                         ppp           http://savannah.gnu.org/projects/epsilon\n"
"-------------------------------------------------------------------------------\n"
*/
PACKAGE_NAME " %s, %s\n"
PACKAGE_NAME " comes with ABSOLUTELY NO WARRANTY; for details type `:no-warranty'.\n\n"
"This is free software, and you are welcome to redistribute it under certain\n"
"conditions; type `:license' for details.\n"
"\n"
"[This REPL is essentially broken. I plan to provide a real meta-interpreter\n"
" as soon as possible.                       --positrone, 17th February 2003]\n"
"\n"
"Type :? for help.\n\n",VERSION_STRING,COPYRIGHT_STRING);
}

/* This function modifies its argument: */
char* trim(char* s){
  int initial,final;

  for(initial=0;isspace(s[initial])&&(s[initial]!='\0');initial++)
    /* Do nothing */;
  s+=initial;

  for(final=strlen(s)-1;isspace(s[final])&&(final>=0);final--)
    /* Do nothing */;
  s[final+1]='\0';

  return s;
}

void quit(){
  //dump_defines(stdout);
  destroy_temporaries();
  destroy_repl_data_structures();
  printf("Goodbye.\n");
  exit(EXIT_SUCCESS);
}

void help(){
  printf(
"Help:\n"
":?, :h or :help\tShow help message.\n"
":no-warranty\tExplain that there is no warranty\n"
":license\tShow the full text of the GNU GPL\n"
":banner\t\tShow the epsilon banner\n"
":listing\tShow the program defined in this session\n"
":save FILENAME\tDump the work session into a file\n"
":quit\t\tExit this read-eval-print loop\n"
  );
}

void dump(FILE* f){
  dump_defines(f);
}

void save_session(char* s){
  FILE* f;

  s=trim(s);
  if(!strcmp(s,"")){
    printf("\aYou must supply a file name with `:save'; try `:?' for information.\n");
    return;
  }    
  f=fopen(s,"r");
  if(f!=NULL){
    char* answer,*trimmed_answer;
    fclose(f);
    printf("\a`%s' exists: do you want to overwrite it or cancel the operation?\n",s);
    do{
      answer=readline("Please type `o' or `c'> ");
      if(answer==NULL){
        printf("\aCancelled.\n");
        return;
      }

      trimmed_answer=trim(answer);
      if(!strcmp(trimmed_answer,"o"))
        break;
      else if(!strcmp(trimmed_answer,"c")){
        printf("\aCancelled.\n");
        return;
      }
      free(answer); /* readline() requires this. */
    }while(strcmp(answer,"o"));
  }

  f=fopen(s,"w");
  if(f==NULL){
    printf("\aCould not write into `%s'.\n",s);
    return;
  }

  fprintf(f,"/* Session saved from the epsilon %s Read-Eval-Print Loop: */\n\n",VERSION_STRING);
  dump(f);
  fclose(f);
  printf("Session saved into `%s'.\n",s);
}

void listing(){
  dump(stdout);
}

/* This function can actually execute a command, else
   just check whether the command is valid (returns 0
   iff the command is not valid): */
int execute_command(char *s,int really_execute){
  s++;
  if((!strcmp(s,"?"))||(!strcmp(s,"h"))||(!strcmp(s,"help"))){
    if(really_execute)
      help();
  }
  else if(!strcmp(s,"quit")){
    if(really_execute)
      quit();
  }
  else if(!strcmp(s,"no-warranty")){
    if(really_execute)
      printf("%s",no_warranty);
  }
  else if(!strcmp(s,"license")){
    if(really_execute){  
      char* s=(char*)malloc(sizeof(char)*(strlen(license)+100));
      sprintf(s,"more - << EOF\n%s\nEOF\n",license);
      system(s);
      free(s);
    }
  }
  else if(!strcmp(s,"banner")){
    if(really_execute)
      show_logo();
  }
  else if(!strcmp(s,"listing")){
    if(really_execute)
      listing();
  }
  else if(!strncmp(s,"save",4)){
    if(!isspace(s[4])){
      if(really_execute)
	printf("\aYou must supply some whitespace and a file name after `:save': use `:?'.\n");
      return 1; /* just not to wait for any more tokens after this malformed :save */
    }
    else if(really_execute)
      save_session(s+5);
  }
  else{
    if(really_execute)
      printf("Could not understand `%s'. Use :? if you need help.\n",s-1);
    //else
    //  printf("%s is not a command",s-1);
    return 0;
  }
  return 1; /* The command was valid */
}

int is_a_command(char* s){
  /* I don't want to list twice the commands: */
  return execute_command(s,0);
}

int evaluate(char* s){
  int x;
  
  chdir(get_temporary_directory());
  write_defines_into_repl_epb();
  append_to_repl_epb(s);
  append_to_repl_epb("\n");
  if(system("epsilonc --generate-eamx --no-verbose repl.epb")==0){
    //if((x=system("time evm a.eamx"))!=0){
    if((x = system("evm a.eamx")) != EXIT_SUCCESS){
      printf("\aThe sub-process exited with error code %i.\n",x);
    }
  }
  else{
    printf("\aError while compiling, aborted.\n");
    return -1;
  }
  
  chdir(get_working_directory());
  //is_a_define=is_an_undefine=is_an_import=is_an_unimport=0;
  
  return x;
}

void remove_define_if_not_harmful(char* identifier){
  if(is_define_present(identifier)){
    deactivate_define(identifier);
    if(evaluate("")==0)
      remove_not_active_define(identifier);
    else
      activate_define(identifier);
  }
  else
    printf("\a%s is not defined.\n");
}

void insert_define_if_not_harmful(char* identifier, char* definition){
  if(!is_define_present(identifier)){
    insert_define(identifier,definition);
    if(evaluate("")!=0)
      remove_define(identifier);
  }
  else{
    deactivate_define(identifier);
    insert_define(identifier,definition);
    if(evaluate("")==0){
      printf("\aWarning: redefining %s.\n",identifier);
      remove_not_active_define(identifier);
    }
    else{
      /* No need to issue a warning message: since evaluation failed a message
         has already been printed. */
      remove_active_define(identifier);
      activate_define(identifier);
    }
  }
}

void manage_entered_string(char* s){
  /* Trim s: */
  s=trim(s);
  if(s[0]==':')
    execute_command(s,1);
  else{
    char* identifier;
    char* definition;
    
    /* Is s a define line? */
    if(is_a_define_line(s,&identifier,&definition)){
      /* Is s valid according to the define syntax? */
      if(strcmp(identifier,""))
	insert_define_if_not_harmful(identifier,definition);
      else
	printf("\aParse error in 'define' line.\n");
      
      return;
    }
    
    /* Is s an undefine line? */
    if(is_an_undefine_line(s,&identifier)){
      /* Is s valid according to the undefine syntax? */
      if(strcmp(identifier,""))
	remove_define_if_not_harmful(identifier);
      else
	printf("\aParse error in 'undefine' line.\n");
      
      return;
    }
    /* To do: import and unimport */
    
    /* If we have arrived here then s is a common epsilon expression: */
    evaluate(s);
  }
}

#define INITIAL_WHOLE_STRING_SIZE 1


int loop(){
  int whole_entered_string_length=INITIAL_WHOLE_STRING_SIZE;
  int whole_entered_string_pointer=0;
  char* whole_entered_string=(char*)malloc(sizeof(char)*(whole_entered_string_length));
 
  char* s, *trimmed;
  int s_length,trimmed_length;
  int terminated=0, more_than_one_semicolon=0;
  int line_no=1;
  char prompt[10];

  do{
    sprintf(prompt,"%-2i> ",line_no);
    s=readline(prompt);
    if(s==NULL){
      printf("\n");
      quit();
    }
    s_length=strlen(s);

    trimmed=trim(s);
    trimmed_length=strlen(trimmed);
    
    line_no++;

    /* Work with the string if it's not empty: */
    if(strcmp(trimmed,"")){
      /* Remember the untrimmed version in the history: */
      add_history(s);
      
      /* Is the entered line correctly terminated? */
      check_entered_line(trimmed,
			 &terminated,
			 &more_than_one_semicolon);
      
      if(whole_entered_string_pointer==0)
	if(is_a_command(trimmed)){
	  terminated=1; /* A command is implicitly terminated */
	  more_than_one_semicolon=0;
	}
      
      /* Append this string to the one we are building: */
      while(whole_entered_string_length<=whole_entered_string_pointer+s_length+1){
	whole_entered_string_length*=2;
	whole_entered_string=(char*)realloc(whole_entered_string,
					    whole_entered_string_length*sizeof(char));
      }
      strcpy(whole_entered_string+whole_entered_string_pointer,s);
      whole_entered_string[whole_entered_string_pointer+s_length]='\n';
      whole_entered_string[whole_entered_string_pointer+s_length+1]='\0';
      
      whole_entered_string_pointer+=s_length+1;
      
      if(more_than_one_semicolon){
	/* Reset the string we were composing: */
	whole_entered_string_pointer=0;
	whole_entered_string_length=INITIAL_WHOLE_STRING_SIZE;
	whole_entered_string=(char*)realloc(whole_entered_string,sizeof(char)*(whole_entered_string_length));
	
	line_no=1;
	terminated=0;
	more_than_one_semicolon=0;
	
	printf("\aThe string you entered contains more than one semicolon.\n");
      }
      else if(!terminated){
	/* Do nothing... */
      }
      else{
	/* Ok, now we have a terminated string with only one semicolon: */
	
	/* Strip the trailing newline from whole_entered_string: */
	whole_entered_string[whole_entered_string_pointer-1]='\0';

        //printf("About to execute '%s'...\n",whole_entered_string);

	manage_entered_string(whole_entered_string);
	
	whole_entered_string_pointer=0;
	whole_entered_string_length=INITIAL_WHOLE_STRING_SIZE;
	whole_entered_string=(char*)realloc(whole_entered_string,sizeof(char)*(whole_entered_string_length));
	
	line_no=1;
	terminated=0;
	more_than_one_semicolon=0;
      }
    } /* if(strcmp(trimmed,"")) */

    /* free storage for string, as required by readline(): */
    free(s);
  }while(1);
}

int main(int argc, char **argv){
  set_program_name("epsilon (" PACKAGE_NAME ")");
  set_general_help_message("Executes the epsilon Read-Eval-Print-Loop.");
  set_synopsis_string("epsilon [OPTIONS]");
  set_version_string(VERSION_STRING);
  set_copyright_string(COPYRIGHT_STRING);
  set_license_message(LICENSE_STRING);
  set_bug_reporting_message(BUG_REPORTING_MESSAGE);

  parse_command_line(&argc, argv);

  if(argc != 1)
    show_help();

  create_repl_data_structures();
  create_temporaries();

  show_logo();

  return loop();
}
