/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2002 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */

#include <stdio.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include "temporary.h"
#include "structures.h"

  //#define HAS_MKDTEMP

char *new_temporary_directory(){
  const char* template="/tmp/epsilonXXXXXX";
  static char dirname[100];
  int filedes;

#ifdef HAS_MKDTEMP
  strcpy(dirname,template); /* So the string can be modified */
  if(mkdtemp(dirname)==NULL){
    fprintf(stderr,"Fatal: could not create a temporary directory\n");
    exit(EXIT_FAILURE);
  }
#else
  /* For systems lacking mkdtemp() we can use this (raw) version: */
  srand(time(NULL)+getpid());
  sprintf(dirname,"/tmp/epsilon%i",(int)(((float)random())/
					 ((float)(RAND_MAX/1000000))));
  if(mkdir(dirname,0700)!=0){
    fprintf(stderr,"Fatal: could not create a temporary directory\n");
    exit(EXIT_FAILURE);
  }
#endif
  return dirname;
}

char *directory;
char current_directory[10000];
char file_name[2000];
char assembly_file_name[2000];
char module_file_name[2000];
char executable_file_name[2000];

void reset(char *s){
  FILE* f=fopen(s,"w");
  if(f!=NULL)
    fclose(f);
  else{
    fprintf(stderr,"Fatal: could not write into %s\n",f);
    exit(EXIT_FAILURE);
  }
}

void create_temporaries(){
  FILE* f;
  directory=new_temporary_directory();
  sprintf(file_name,"%s/repl.epb",directory);
  sprintf(assembly_file_name,"%s/repl.eaml",directory);
  sprintf(module_file_name,"%s/repl.eamo",directory);
  sprintf(executable_file_name,"%s/a.eamx",directory);

  reset(file_name);
  reset(assembly_file_name);
  reset(module_file_name);
  reset(executable_file_name);

  if(getcwd(current_directory,sizeof(current_directory))==NULL){
    fprintf(stderr,"FATAL: current path is too long\n");
    exit(EXIT_FAILURE);
  }
}

void destroy_temporaries(){
  if(remove(executable_file_name)!=0)
    fprintf(stderr,"Warning: could not delete %s\n",executable_file_name);
  if(remove(module_file_name)!=0)
    fprintf(stderr,"Warning: could not delete %s\n",module_file_name);
  if(remove(assembly_file_name)!=0)
    fprintf(stderr,"Warning: could not delete %s\n",assembly_file_name);
  if(remove(file_name)!=0)
    fprintf(stderr,"Warning: could not delete %s\n",file_name);
  if(remove(directory)!=0)
    fprintf(stderr,"Warning: could not delete %s\n",directory);
}

void write_into_repl_epb(char* s){
  FILE* f=fopen(file_name,"w");

  fprintf(f,"%s",s);
  fclose(f);
}

void append_to_repl_epb(char* s){
  FILE* f=fopen(file_name,"a");

  fprintf(f,"%s",s);
  fclose(f);
}

void reset_repl_epb(){
  reset(file_name);
}

void write_defines_into_repl_epb(){
  FILE* f=fopen(file_name,"w");

  dump_defines(f);
  fclose(f);
}

char* get_working_directory(){
  return current_directory;
}

char* get_temporary_directory(){
  return directory;
}
