# Copyright (C) 2013 The ESPResSo project
# Copyright (C) 2012 Olaf Lenz
#
# This file is part of ESPResSo.
#
# ESPResSo is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# ESPResSo is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# This script generates the files featureconfig.h and featureconfig.c.
#
import sys, featuredefs, time, string

if len(sys.argv) != 4:
    print >> sys.stderr, "Usage: %s DEFFILE HFILE CFILE" % sys.argv[0]
    exit(2)

deffilename, hfilename, cfilename = sys.argv[1:4]

print "Reading definitions from " + deffilename + "..."
defs = featuredefs.defs(deffilename)
print "Done."

print "Writing " + hfilename + "..."
hfile = file(hfilename, 'w');

hfile.write("""/* 
WARNING: This file was autogenerated by

   %s on %s

   Do not modify it or your changes will be overwritten!
   Modify features.def instead.
*/
#ifndef _FEATURECONFIG_H
#define _FEATURECONFIG_H

""" % (sys.argv[0], time.asctime()))

# handle implications
hfile.write('/* Handle implications */')
implication_template = string.Template("""
// $feature implies $implied
#if defined($feature) && !defined($implied)
#define $implied
#endif
""")
for feature, implied in defs.implications:
    hfile.write(implication_template.substitute(feature=feature, implied=implied))

# output warnings if internal features are set manually
hfile.write('/* Warn when derived switches are specified manually */')
derivation_template = string.Template("""
// $feature equals $expr
#ifdef $feature
#warning $feature is a derived switch and should not be set manually!
#elif $cppexpr
#define $feature
#endif
""")
for feature, expr, cppexpr in defs.derivations:
    hfile.write(derivation_template.substitute(feature=feature, cppexpr=cppexpr, expr=expr))

# write footer
# define external FEATURES and NUM_FEATURES
hfile.write("""
extern const char* FEATURES[];
extern const int NUM_FEATURES;

#endif /* of _FEATURECONFIG_H */""")
hfile.close()
print "Done."

print "Writing " + cfilename + "..."
cfile = file(cfilename, 'w');

# handle requirements

cfile.write("""/* 
WARNING: This file was autogenerated by

   %s on %s

   Do not modify it or your changes will be overwritten!
   Modify features.def instead.
*/

/* config.h includes featureconfig.h and myconfig.h */
#include "config.h"

""" % (sys.argv[0], time.asctime()))

cfile.write('/* Handle requirements */')

requirement_template = string.Template("""
// $feature requires $expr
#if defined($feature) && !($cppexpr)
#error Feature $feature requires $expr
#endif
""")
for feature, expr, cppexpr in defs.requirements:
    cfile.write(requirement_template.substitute(feature=feature, cppexpr=cppexpr, expr=expr))

cfile.write("""

/* Feature list */
const char* FEATURES[] = {
""")

featurestring_template = string.Template(
"""#ifdef $feature
  "$feature",
#endif
""")
for feature in defs.externals:
    cfile.write(featurestring_template.substitute(feature=feature))
for feature in defs.features:
    cfile.write(featurestring_template.substitute(feature=feature))

cfile.write("""
};

const int NUM_FEATURES = sizeof(FEATURES)/sizeof(char*);
""");

cfile.close()
print "Done."
