/***************************************************************
 *                 Finite Element Method Object Library        *
 *           class p1c3d : declaration for p1c3d               *
 *                    simula+@metz.ensam.fr                    *
 *	             GNU/linux version 2.6.0	               *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2004 CREUSE Emmanuel
 * copyright © 2004 PIRAUT Frédéric
 * copyright © 2012 COLLARD Christophe
 * copyright © 2004,2012 Centre National de la Recherche Scientifique
 * copyright © 2004 Arts et Métiers ParisTech
 * copyright © 2004 Université de Valenciennes et du Hainaut Cambrésis
 * copyright © 2004 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2004 Laboratoire de Mathématiques et ses Applications de Valenciennes (LAMAV)
 * copyright © 2012 Centre d'Elaboration de Matériaux et d'Etudes Structurales (CEMES - CNRS)
 ***************************************************************/

/*! \class p1c3d
    \brief p1c3d library \n

    \htmlonly 
    <FONT color="#838383">

    p1c3d belongs to Finite Element Method Object Libraries (FEMOL++) </br>
    FEMOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \authors copyright \htmlonly &#169; \endhtmlonly  2004 CREUSE Emmanuel \n
	     copyright \htmlonly &#169; \endhtmlonly  2004 PIRAUT Frédéric \n
	     copyright \htmlonly &#169; \endhtmlonly 2012 Christophe COLLARD \n
             copyright \htmlonly &#169; 2004, 2012 Centre National de la Recherche Scientifique \endhtmlonly \n
	     copyright \htmlonly &#169; 2004 Arts et M&#233;tiers ParisTech \endhtmlonly \n
	     copyright \htmlonly &#169; 2004 Universit&#233; de Valenciennes et du Hainaut Cambr&#233;sis \endhtmlonly \n
	     copyright \htmlonly &#169; 2004 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	     copyright \htmlonly &#169; 2004 Laboratoire de Math&#233;matiques et ses Applications de Valenciennes (LAMAV) \endhtmlonly
	     copyright \htmlonly &#169; 2012 Centre d'Elaboration de Mat&#233;riaux et d'Etudes Structurales (CEMES - CNRS) \endhtmlonly \n
    \version 2.6.0
    \date 2004-2012
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type p1c3d
#endif

#ifndef _p1c3d_h
#define _p1c3d_h

#if !defined(__ASSERT_H)
#include <assert.h>
#endif

#if !defined(__VECTORS_H)
#include "MOL++/vectors.h"
#endif

#if !defined(__MATRIX_H)
#include "MOL++/matrix.h"
#endif

#if !defined(__INTEGRATION_H)
#include "MOL++/integration.h"
#endif

#if !defined(__MESH_H)
#include "FEMOL++/meshes/mesh.h"
#endif

using namespace std;
using namespace mol;

namespace femol
{


//============================================================
template <class T> class p1c3d : public mesh<tetrahedron<T>,T>
//============================================================
{
  using mesh<tetrahedron<T>,T>::alive;
  using mesh<tetrahedron<T>,T>::dimspace_;
  using mesh<tetrahedron<T>,T>::nbvertex_;
  using mesh<tetrahedron<T>,T>::nbelement_;
  using mesh<tetrahedron<T>,T>::vertices;
  using mesh<tetrahedron<T>,T>::element;

  private:
    int DofNumber_;
    int GlobalDofNumber_;
    matrix<T> RefMassMatrix_;
    vector<T> BuildBloc (T (*f)(T,T,T), int, integration<T>) const;

  public:
    p1c3d ();
    p1c3d (const char*, const char* = "");
    p1c3d (const p1c3d<T>&);
    p1c3d<T>& operator = (const p1c3d<T>&);
    /*! \return the number of node of the object */
    int DofNumber () const {return DofNumber_;}
    /*! \return the number of node of meshes */
    int GlobalDofNumber () const {return GlobalDofNumber_;}
    /*! \return Reference Mass Matrix */
    matrix<T> RefMassMatrix () const {return RefMassMatrix_;}
    vector<T> BasisFunction (T, T, T) const;
    matrix<T> DiffBasisFunction (T xref=0, T yref=0, T zref=0) const;
    virtual int operator () (int, int) const;
    template <class Tf> friend bool operator == (const p1c3d<Tf>&, const p1c3d<Tf>&);
    template <class Tf> friend bool operator != (const p1c3d<Tf>&, const p1c3d<Tf>&);
    matrix<T> LocalMassMatrix (int) const;
    matrix<T> LocalStiffnessMatrix (int) const;
    vector<T> Fk (T, T, T, int) const;
    matrix<T> DUDV () const;
    vector<T> B (T (*f)(T,T,T), int) const;
    void Dirichlet (matrix<T>&, vector<T>&, T (*g)(T,T,T)) const;
    T error_L2 (T (*u)(T,T,T), const vector<T>&, int) const;
    T error_semi_H1 (T (*dudx)(T,T,T), T (*dudy)(T,T,T), T (*dudz)(T,T,T), const vector<T>&, int) const;
};


//=====Private methods for p1c3d=============================================


//=====Public methods for p1c3d==============================================


/*!
  \brief default constructor of the p1c3d class
*/

//============================================
template <class T>
p1c3d<T>::p1c3d () : mesh<tetrahedron<T>,T> ()
//============================================
{
  DofNumber_ = GlobalDofNumber_ = 0;
}


/*!
  \brief constructor of the p1c3d class. The arguments are 2 files:file1 and file2.numerotation of each element. file1 indicates the dimension (2 or 3), the number of nodes and each node coordinate with a boundary condition (1 if face is on the boundary , 0 otherwise). File2 must be documented by the number of elements, node's numerotation of each tetrahedron.
  \param file1 string
  \param file2 string (optional if file1 and file 2 are merged)
*/

//============================================================================================
template <class T>
p1c3d<T>::p1c3d (const char* file1, const char* file2) : mesh<tetrahedron<T>,T> (file1, file2)
//============================================================================================
{
  DofNumber_ = 4;
  GlobalDofNumber_ = nbvertex_;
  RefMassMatrix_ = matrix<T> (4, 4, true, 1/120.);
  for (int i=1; i<=4; i++)
    RefMassMatrix_(i,i) *= 2;

  /* algorithme permettant de déterminer la condition frontaliere de chaque noeud
  int num_face,nbfaces_;
  int found;
  int num_neu_in_front_of_face;
  vector<int> num_neu_face (3);
  matrix<int> face_of_element_ (nbelement_, 4);
  int nb_min_faces = nbvertex_ + 2 * (nbelement_ - 1);
  type_of_node_ = vector<int> (nbvertex_);
  vector<int> head (nbvertex_);
  vector<int> next (nb_min_faces);
  matrix<int> hyp_node_of_face_ (nb_min_faces, 3);
  vector<int> hyp_type_of_face_ (nb_min_faces);
  for (int i=1; i<=nbvertex_; i++)
    type_of_node_[i] = 0;
  for (int i=1; i<=nb_min_faces; i++)
    hyp_type_of_face_[i] = 1;
  num_face = 0;
  for (int i=1; i<=nbelement_; i++)
    for (int j=1; j<=4; j++)
      { for (int k=1; k<=3; k++)
	  num_neu_face[k] = (*this).number ((*this)[i][1+(j+k-1)%4]);
	num_neu_in_front_of_face = (*this).number ((*this)[i][j]);
	order (num_neu_face);
	int exist = 0;
	for (int e=head[num_neu_face[1]]; e!=0; e=next[e])
	  if ((hyp_node_of_face_(e,2) == num_neu_face[2]) && (hyp_node_of_face_(e,3) == num_neu_face[3]))
	    { exist = 1;
	      face_of_element_(i,j) = e;
	      hyp_type_of_face_[e] = 0;
	    }
	if (!exist)
	  { num_face++;
	    for (int l=1; l<=3; l++)
	      hyp_node_of_face_(num_face,l) = num_neu_face[l];
	    next[num_face] = head[num_neu_face[1]];
	    face_of_element_(i,j) = num_face;
	    head[num_neu_face[1]] = num_face;
	  }
      }
  nbfaces_ = num_face;
  matrix<int> node_of_face_(nbfaces_,3);
  vector<int> type_of_face_(nbfaces_);
  for (int i=1; i<=nbfaces_; i++)
    { for (int j=1; j<=3; j++)
	node_of_face_(i,j) = hyp_node_of_face_(i,j);
      type_of_face_[i] = hyp_type_of_face_[i];
    }
  for (int i=1; i<=nbfaces_; i++)
    { if (type_of_face_[i] == 1)
	for (int j=1; j<=3; j++)
	  type_of_node_[node_of_face_(i,j)] = 1;
    }
*/
}


/*!
  \brief copy constructor of the p1c3d class
  \param EF reference on p1c3d object
*/

//================================================================
template <class T>
p1c3d<T>::p1c3d (const p1c3d<T>& EF) : mesh<tetrahedron<T>,T> (EF)
//================================================================
{
  DofNumber_ = EF.DofNumber_;
  GlobalDofNumber_ = EF.GlobalDofNumber_;
  RefMassMatrix_ = EF.RefMassMatrix_;
  // RefStiffnessMatrix_ = EF.RefStiffnessMatrix_;
}


/*!
  \brief affectation operator for the p1c3d class.
  \param EF reference on p1c3d object.
  \param reference on p1c3d object.
*/

//=================================================
template <class T>
p1c3d<T>& p1c3d<T>::operator = (const p1c3d<T>& EF)
//=================================================
{
  (*this).mesh<tetrahedron<T>,T>::operator = (EF);
  DofNumber_ = EF.DofNumber_;
  GlobalDofNumber_ = EF.GlobalDofNumber_;
  RefMassMatrix_ = EF.RefMassMatrix_;
  // RefStiffnessMatrix_ = EF.RefStiffnessMatrix_;

  return (*this);
}


/*!
  \param xref type T
  \param yref type T
  \param zref type T
  \return vector container the basis functions of the reference element p1c3d.
*/

//==============================================================
template <class T>
vector<T> p1c3d<T>::BasisFunction (T xref, T yref, T zref) const
//==============================================================
{
  vector<T> P (DofNumber_);
  P[1] = 1 - xref - yref - zref;
  P[2] = xref;
  P[3] = yref;
  P[4] = zref;
  return P;
}


/*!
  \param xref type T
  \param yref type T
  \param zref type T
  \return matrix container the basis functions derivate of the reference element p1c3d.
*/

//==================================================================
template <class T>
matrix<T> p1c3d<T>::DiffBasisFunction (T xref, T yref, T zref) const
//==================================================================
{
  matrix<T> DP (3, DofNumber_);
  DP(1,1) = DP(2,1) = DP(3,1) = -1;
  DP(1,2) = DP(2,3) = DP(3,4) = 1;

  return DP;
}


/*!
  \param num_element element's number
  \param j the j-th face
  \return global number of the j-th of the element num_element
*/

//======================================================
template <class T>
int p1c3d<T>::operator () (int num_element, int j) const
//======================================================
{
  assert ((j >= 1) && (j <= DofNumber_) && (num_element >= 1) && (num_element <= nbelement_));
  return number (element[num_element-1][j]);
}


/*!
  \param EF1 type const p1c3d<T>&
  \param EF2 type const p1c3d<T>&
  \return 1 if EF1 and  EF2 are equal, 0 otherwise
*/

//===========================================================
template <class Tf>
bool operator == (const p1c3d<Tf>& EF1, const p1c3d<Tf>& EF2)
//===========================================================
{
  bool boolean = true;
  boolean *= (mesh <tetrahedron<Tf>,Tf> (EF1) == mesh <tetrahedron<Tf>,Tf> (EF2));
  boolean *= (EF1.DofNumber_ == EF2.DofNumber_);
  boolean *= (EF1.GlobalDofNumber_ == EF2.GlobalDofNumber_);
  boolean *= (EF1.RefMassMatrix_ == EF2.RefMassMatrix_);

  return boolean;
}


/*!
  \param EF1 type const p1c3d<T>&
  \param EF2 type const p1c3d<T>&
  \return 1 if EF1 and  EF2 are different, 0 otherwise
*/

//===========================================================
template <class Tf>
bool operator != (const p1c3d<Tf>& EF1, const p1c3d<Tf>& EF2)
//===========================================================
{
  return !(EF1 == EF2);
}


/*!
  \param num_element element's number
  \return local mass matrix of the element num_element
*/

//=========================================================
template <class T>
matrix<T> p1c3d<T>::LocalMassMatrix (int num_element) const
//=========================================================
{
  assert ((num_element >= 1) && (num_element <= nbelement_));
  return (RefMassMatrix_ * element[num_element-1].AbsDet());
}


/*!
  \param num_element element's number
  \return local stifness  matrix of the element num_element
*/

//==============================================================
template <class T>
matrix<T> p1c3d<T>::LocalStiffnessMatrix (int num_element) const
//==============================================================
{
  assert ((num_element >= 1) && (num_element <= nbelement_));
  matrix<long double> inv = element[num_element-1].InvMk();
  return (element[num_element-1].AbsDet()/6.) * t(DiffBasisFunction()) * inv * t(inv) * DiffBasisFunction();
}


/*!
  \param x type T
  \param y type T
  \param z type T
  \param i type integer
  \return vector by the application Fk (reference element -> i-th element)
*/

//=================================================
template <class T>
vector<T> p1c3d<T>::Fk (T x, T y, T z, int i) const
//=================================================
{
  matrix<T> MAT(3,3);
  vector<T> VEC(3);
  VEC[1] = x;  VEC[2] = y;  VEC[3] = z;

  vector<T> A1A2 = element[i-1][2] - element[i-1][1];
  vector<T> A1A3 = element[i-1][3] - element[i-1][1];
  vector<T> A1A4 = element[i-1][4] - element[i-1][1];
  MAT(1,1) = A1A2[1];  MAT(1,2) = A1A3[1];  MAT(1,3) = A1A4[1];
  MAT(2,1) = A1A2[2];  MAT(2,2) = A1A3[2];  MAT(2,3) = A1A4[2];
  MAT(3,1) = A1A2[3];  MAT(3,2) = A1A3[3];  MAT(3,3) = A1A4[3];

  return MAT * VEC + element[i-1][1];
}


/*!
  \return the global stifness matrix of the p1c3d element
*/

//===============================
template <class T>
matrix<T> p1c3d<T>::DUDV () const
//===============================
{
  matrix<T> Stiff (DofNumber_, DofNumber_);
  matrix<T> DUDV_(GlobalDofNumber_, GlobalDofNumber_);
  for (int k=1; k<=nbelement_; k++)
    { Stiff = LocalStiffnessMatrix (k);
      for (int i=1; i<=DofNumber_; i++)
	for (int j=1; j<=DofNumber_; j++)
	  DUDV_((*this)(k,i), (*this)(k,j)) += Stiff(i,j);
    }

  return DUDV_;
}


/*!
  \brief Value of the contribution to the right-hand side of the variational formulation of a given element \n \n
  \param \f$ f \f$ the function   \n
  \param num_element the element on which we want to integrate \n
  \param itg the Gauss points and weight used for the numerical integration
  \return vector \f$(\int_{num_{element}} f(x,y) \varphi_1(x,y) dx dy,\int_{num_{element}} f(x,y) \varphi_2(x,y) dx dy,\int_{num_{element}} f(x,y) \varphi_3(x,y) dx dy, \int_{num_{element}} f(x,y) \varphi_4(x,y) dx dy)\f$ where \f$(\varphi_i)_{1\leq i\leq 4}\f$ is the i-th basis function associated to the element num_element.
  \n\n
*/

//======================================================================================
template <class T>
vector<T> p1c3d<T>::BuildBloc (T (*f)(T,T,T), int num_element, integration<T> itg) const
//======================================================================================
{
  vector<T> res (DofNumber_);
  vector<T> Basis (DofNumber_);
  T omegak, omegal, omegai, xk, xl, xi;
  vector<T> Fxkli;
  for (int k=1; k<=itg.NbIntPts(); k++)
    { omegak = itg.weight (k);
      xk = itg.point (k);
      for (int l=1; l<=itg.NbIntPts(); l++)
	{ omegal = itg.weight (l);
	  xl = itg.point (l);
	  for (int i=1; i<=itg.NbIntPts(); i++)
	    { omegai = itg.weight (i);
	      xi = itg.point (i);
	      Basis = BasisFunction (xk, (1-xk)*xl, (1-xk-xl+xk*xl)*xi);
	      Fxkli = Fk (xk, (1-xk)*xl, (1-xk-xl+xk*xl)*xi, num_element);
	      for (int j=1; j<=DofNumber_; j++)
		res[j] += omegak * omegal * omegai * (1-xk) * (1-xk-xl+xk*xl) * f (Fxkli[1], Fxkli[2], Fxkli[3]) * Basis[j];
	    }
	}
    }
  res *= element[num_element-1].AbsDet();

  return res;
}


/*!
  \brief Value of the right-hand side of the variational formulation \n \n
  \param \f$ f \f$ the function   \n
  \param NbIntPts the number of Gauss points to do the numerical integration \n
  \return vector v with v_i=\f$(\int_{\Omega} f(x,y) \varphi_i(x,y) dx dy \f$ where \f$(\varphi_i)_{1\leq i\leq nbvertex}\f$ is the i-th global basis function.
  \n\n
*/

//=======================================================
template <class T>
vector<T> p1c3d<T>::B (T (*f)(T,T,T), int NbIntPts) const
//=======================================================
{
  assert (NbIntPts > 0);
  integration<T> itg;
  itg.Gauss (NbIntPts, 0, 1);
  vector<T> B_(GlobalDofNumber_);
  vector<T> Bloc (DofNumber_);
  for (int k=1; k<=nbelement_; k++)
    { Bloc = BuildBloc (f, k, itg);
      for (int i=1; i<=DofNumber_; i++)
	B_[(*this)(k,i)] += Bloc[i];
    }

 return B_;
}


/*!
  \param MAT type matrix<T>
  \param F type vector<T>
  \param g type T
*/

//=========================================================================
template <class T>
void p1c3d<T>::Dirichlet (matrix<T>& MAT, vector<T>&F, T (*g)(T,T,T)) const
//=========================================================================
{
  for (int i=1; i<=GlobalDofNumber_; i++)
    if (mesh<tetrahedron<T>,T>::operator()(i).boundary_condition())
      { F[i] = (*g) (vertices[i-1][1], vertices[i-1][2], vertices[i-1][3]);
	for (int j=1; j<=GlobalDofNumber_; j++)
	  MAT (i,j) = 0;
	MAT(i,i) = 1;
	for (int k=1; k<=GlobalDofNumber_; k++)
	  if (k != i)
	    { F[k] += -MAT(k,i) * F[i];
	      MAT(k,i) = 0;
	    }
      }
}


/*!
  \param u exact solution
  \param uh vector approximation solution
  \param NbIntPts number of points of the integration
  \return error_L2
*/

//==========================================================================
template <class T>
T p1c3d<T>::error_L2 (T(*u)(T,T,T), const vector<T>& uh, int NbIntPts) const
//==========================================================================
{
  T GlobalError = 0;
  T LocalError;
  integration<T> itg;
  itg.Gauss (NbIntPts, 0, 1);
  T xk, xl, xi, omegak, omegal, omegai, SumK;
  vector<T> Fxkli;
  vector<T> Basis (DofNumber_);

  for (int num_element = 1; num_element <= nbelement_; num_element++)
    { LocalError = 0;
      for (int k=1; k<=itg.NbIntPts(); k++)
	{ omegak = itg.weight (k);
	  xk = itg.point (k);
	  for (int l=1; l<=itg.NbIntPts(); l++)
	    { omegal = itg.weight (l);
	      xl = itg.point (l);
	      for (int i=1; i<=itg.NbIntPts(); i++)
		{ omegai = itg.weight (i);
		  xi = itg.point (i);
		  Basis = BasisFunction (xk, (1-xk)*xl, (1-xk-xl+xk*xl)*xi);
		  SumK = 0;
		  for (int j=1; j<=DofNumber_; j++)
		    SumK += uh[(*this)(num_element,j)] * Basis[j];
		  Fxkli = Fk (xk, (1-xk)*xl, (1-xk-xl+xk*xl)*xi, num_element);
		  LocalError += omegak * omegal * omegai * (1-xk) * (1-xk-xl+xk*xl) * pow ((u (Fxkli[1], Fxkli[2], Fxkli[3]) - SumK), 2);
		}
	    }
	}
      GlobalError += element[num_element-1].AbsDet() * LocalError;
    }

  return sqrt (GlobalError);
}


/*!
  \param dudx type T
  \param dudy type T
  \param dudz type T
  \param uh vector approximation solution
  \param NbIntPts number of points of the integration
  \return error_semi_H1
*/

//=======================================================================================================================
template <class T>
T p1c3d<T>::error_semi_H1 (T (*dudx)(T,T,T), T (*dudy)(T,T,T), T (*dudz)(T,T,T), const vector<T>& uh, int NbIntPts) const
//=======================================================================================================================
{
  T GlobalError = 0;
  T LocalError;
  T omegak, omegal, omegai, xk, xl, xi;
  T duhdx, duhdy, duhdz;
  T duhdxhat, duhdyhat, duhdzhat;
  vector<T> Fxkli;
  integration<T> itg;
  itg.Gauss (NbIntPts, 0, 1);
  matrix<T> DiffBasis (3, DofNumber_);
  matrix<T> InvMk;
  for (int num_element = 1 ;num_element <= nbelement_; num_element++)
    { LocalError = 0;
      InvMk = element[num_element-1].InvMk();
      for (int k=1; k<=itg.NbIntPts(); k++)
	{ omegak = itg.weight (k);
	  xk = itg.point (k);
	  for (int l=1; l<=itg.NbIntPts(); l++)
	    { omegal = itg.weight (l);
	      xl = itg.point (l);
	      for (int i=1; i<=itg.NbIntPts(); i++)
		{ omegai = itg.weight(i);
		  xi = itg.point (i);
		  DiffBasis = DiffBasisFunction (xk, (1-xk)*xl, (1-xk-xl+xl*xk)*xi);
		  duhdxhat = 0;
		  duhdyhat = 0;
		  duhdzhat = 0;
		  for (int j=1; j<=DofNumber_; j++)
		    { duhdxhat += uh[(*this)(num_element,j)] * DiffBasis (1,j);
		      duhdyhat += uh[(*this)(num_element,j)] * DiffBasis (2,j);
		      duhdzhat += uh[(*this)(num_element,j)] * DiffBasis(3,j);
		    }
		  duhdx = duhdxhat * InvMk(1,1) + duhdyhat * InvMk(2,1) + duhdzhat * InvMk(3,1);
		  duhdy = duhdxhat * InvMk(1,2) + duhdyhat * InvMk(2,2) + duhdzhat * InvMk(3,2);
		  duhdz = duhdxhat * InvMk(1,3) + duhdyhat * InvMk(2,3) + duhdzhat * InvMk(3,3);
		  Fxkli = Fk (xk, (1-xk)*xl, (1-xk-xl+xl*xk)*xi, num_element);
		  LocalError += omegak * omegal * omegai * (1-xk) * (1-xk-xl+xk*xl) * (pow (dudx (Fxkli[1], Fxkli[2], Fxkli[3]) - duhdx, 2) + pow (dudy (Fxkli[1], Fxkli[2], Fxkli[3]) - duhdy, 2) + pow (dudz (Fxkli[1], Fxkli[2], Fxkli[3]) - duhdz, 2));
		}
	    }
	}
      GlobalError += LocalError * element[num_element-1].AbsDet();
    }

  return sqrt (GlobalError);
}


}


#endif
